/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.common.persistence;

import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kylin.common.persistence.RawResource;
import org.apache.kylin.common.persistence.ResourceStore;
import org.apache.kylin.common.persistence.ResourceTool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceParallelCopier {
    private static final Logger logger = LoggerFactory.getLogger(ResourceParallelCopier.class);
    private final ResourceStore src;
    private final ResourceStore dst;
    private int threadCount = 5;
    private int groupSize = 200;
    private int heartBeatSec = 20;
    private int retry = 2;

    public ResourceParallelCopier(ResourceStore src, ResourceStore dst) {
        this.src = src;
        this.dst = dst;
    }

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    public void setGroupSize(int groupSize) {
        this.groupSize = groupSize;
    }

    public void setHeartBeatSec(int heartBeatSec) {
        this.heartBeatSec = heartBeatSec;
    }

    public void setRetry(int retry) {
        this.retry = retry;
    }

    public Stats copy(String folder, String[] includes, String[] excludes) throws IOException {
        return this.copy(folder, includes, excludes, new Stats());
    }

    public Stats copy(String folder, String[] includes, String[] excludes, Stats stats) throws IOException {
        logger.info("Copy {} from {} to {}", new Object[]{folder, this.src, this.dst});
        TreeMap<String, Integer> groups = this.calculateGroupsToCopy(folder, includes, excludes);
        if (groups == null || groups.isEmpty()) {
            return stats;
        }
        this.copyGroups(groups, includes, excludes, stats);
        while (stats.hasError() && this.retry > 0) {
            --this.retry;
            stats.onRetry(stats.errorResource.get());
            this.copyGroups(this.collectErrorGroups(stats), includes, excludes, stats);
        }
        logger.info("Done copy {} from {} to {}", new Object[]{folder, this.src, this.dst});
        return stats;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyGroups(TreeMap<String, Integer> groups, String[] includes, String[] excludes, Stats stats) {
        stats.onAllStart(groups);
        ExecutorService exec = Executors.newFixedThreadPool(this.threadCount);
        try {
            this.doCopyParallel(exec, groups, includes, excludes, stats);
        }
        finally {
            exec.shutdown();
            stats.heartBeat();
            while (!exec.isTerminated()) {
                try {
                    exec.awaitTermination(this.heartBeatSec, TimeUnit.SECONDS);
                    stats.heartBeat();
                }
                catch (InterruptedException e) {
                    logger.error("interruped", (Throwable)e);
                }
            }
        }
        stats.onAllDone();
    }

    private TreeMap<String, Integer> calculateGroupsToCopy(String folder, String[] includes, String[] excludes) throws IOException {
        NavigableSet<String> all = this.src.listResourcesRecursively(folder);
        if (all == null || all.isEmpty()) {
            return null;
        }
        int sizeBeforeFilter = all.size();
        Iterator<String> it = all.iterator();
        while (it.hasNext()) {
            String path = it.next();
            if (ResourceTool.matchFilter(path, includes, excludes)) continue;
            it.remove();
        }
        int sizeAfterFilter = all.size();
        logger.info("{} resources (out of {}) to copy", (Object)sizeAfterFilter, (Object)sizeBeforeFilter);
        TreeMap<String, Integer> groupCollector = new TreeMap<String, Integer>();
        this.divideGroups(all, "/", groupCollector);
        return groupCollector;
    }

    private TreeMap<String, Integer> collectErrorGroups(Stats stats) {
        TreeMap<String, Integer> newGroups = new TreeMap<String, Integer>();
        for (String errGroup : stats.errorGroups) {
            newGroups.put(errGroup, stats.allGroups.get(errGroup));
        }
        for (String errResPath : stats.errorResourcePaths) {
            newGroups.put(errResPath, 1);
        }
        return newGroups;
    }

    void divideGroups(NavigableSet<String> resources, String prefixSoFar, TreeMap<String, Integer> groupCollector) {
        if (resources.isEmpty()) {
            return;
        }
        if (resources.size() <= this.groupSize) {
            String group = this.longestCommonPrefix(resources, prefixSoFar);
            groupCollector.put(group, resources.size());
            return;
        }
        TreeSet<String> newSet = new TreeSet<String>();
        String newPrefix = null;
        int newPrefixLen = prefixSoFar.length() + 1;
        for (String path : resources) {
            String myPrefix;
            String string = myPrefix = path.length() < newPrefixLen ? path : path.substring(0, newPrefixLen);
            if (newPrefix != null && !myPrefix.equals(newPrefix)) {
                this.divideGroups(newSet, newPrefix, groupCollector);
                newSet.clear();
                newPrefix = null;
            }
            if (newPrefix == null) {
                newPrefix = myPrefix;
            }
            newSet.add(path);
        }
        if (!newSet.isEmpty()) {
            this.divideGroups(newSet, newPrefix, groupCollector);
        }
    }

    String longestCommonPrefix(NavigableSet<String> strs, String prefixSoFar) {
        int minLen = Integer.MAX_VALUE;
        for (String s : strs) {
            minLen = Math.min(minLen, s.length());
        }
        for (int i = prefixSoFar.length(); i < minLen; ++i) {
            char c = ((String)strs.first()).charAt(i);
            for (String s : strs) {
                if (s.charAt(i) == c) continue;
                return s.substring(0, i);
            }
        }
        return ((String)strs.first()).substring(0, minLen);
    }

    private void doCopyParallel(ExecutorService exec, TreeMap<String, Integer> groups, final String[] includes, final String[] excludes, final Stats stats) {
        for (final Map.Entry<String, Integer> entry : groups.entrySet()) {
            exec.execute(new Runnable(){

                @Override
                public void run() {
                    String group = (String)entry.getKey();
                    int expectResources = (Integer)entry.getValue();
                    stats.onGroupStart(group);
                    try {
                        int actualResources = ResourceParallelCopier.this.copyGroup(group, includes, excludes, stats);
                        stats.onGroupSuccess(group, expectResources, actualResources);
                    }
                    catch (Throwable ex) {
                        stats.onGroupError(group, expectResources, ex);
                    }
                }
            });
        }
    }

    private int copyGroup(String group, final String[] includes, final String[] excludes, final Stats stats) throws IOException {
        int cut = group.lastIndexOf(47);
        String folder = cut == 0 ? "/" : group.substring(0, cut);
        final int[] count = new int[1];
        this.src.visitFolderAndContent(folder, true, new ResourceStore.VisitFilter(group), new ResourceStore.Visitor(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void visit(RawResource resource) {
                String path = resource.path();
                try {
                    if (!ResourceTool.matchFilter(path, includes, excludes)) {
                        return;
                    }
                    count[0] = count[0] + 1;
                    stats.onResourceStart(path);
                    long nBytes = ResourceParallelCopier.this.dst.putResource(path, resource.content(), resource.lastModified());
                    stats.onResourceSuccess(path, nBytes);
                }
                catch (Exception ex) {
                    stats.onResourceError(path, ex);
                }
                finally {
                    ResourceParallelCopier.this.closeQuietly(resource);
                }
            }
        });
        return count[0];
    }

    private void closeQuietly(RawResource raw) {
        try {
            if (raw != null) {
                raw.close();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public static class Stats {
        public final Map<String, Integer> allGroups = Collections.synchronizedMap(new TreeMap());
        public final Set<String> startedGroups = Collections.synchronizedSet(new TreeSet());
        public final Set<String> successGroups = Collections.synchronizedSet(new TreeSet());
        public final Set<String> errorGroups = Collections.synchronizedSet(new TreeSet());
        public final AtomicLong totalBytes = new AtomicLong();
        public final AtomicInteger totalResource = new AtomicInteger();
        public final AtomicInteger successResource = new AtomicInteger();
        public final AtomicInteger errorResource = new AtomicInteger();
        public final Set<String> errorResourcePaths = Collections.synchronizedSet(new TreeSet());
        public long createTime = System.nanoTime();
        public long startTime;
        public long endTime;

        private void reset() {
            this.endTime = 0L;
            this.startTime = 0L;
            this.allGroups.clear();
            this.startedGroups.clear();
            this.successGroups.clear();
            this.errorGroups.clear();
            this.totalBytes.set(0L);
            this.totalResource.set(0);
            this.successResource.set(0);
            this.errorResource.set(0);
            this.errorResourcePaths.clear();
        }

        void onAllStart(TreeMap<String, Integer> groups) {
            this.reset();
            logger.debug("{} groups to copy in parallel", (Object)groups.size());
            this.allGroups.putAll(groups);
            this.startTime = System.nanoTime();
        }

        void onAllDone() {
            this.endTime = System.nanoTime();
        }

        void onGroupStart(String group) {
            logger.debug("Copying group {}*", (Object)group);
            this.startedGroups.add(group);
        }

        void onGroupError(String group, int resourcesInGroup, Throwable ex) {
            logger.error("Error copying group " + group, ex);
            this.errorGroups.add(group);
            this.errorResource.addAndGet(resourcesInGroup);
        }

        void onGroupSuccess(String group, int expectResources, int actualResources) {
            this.successGroups.add(group);
            if (actualResources != expectResources) {
                logger.warn("Group {} expects {} resources but got {}", new Object[]{group, expectResources, actualResources});
            }
        }

        void onResourceStart(String path) {
            logger.trace("Copying {}", (Object)path);
            this.totalResource.incrementAndGet();
        }

        void onResourceError(String path, Throwable ex) {
            logger.error("Error copying " + path, ex);
            this.errorResource.incrementAndGet();
            this.errorResourcePaths.add(path);
        }

        void onResourceSuccess(String path, long nBytes) {
            this.successResource.incrementAndGet();
            this.totalBytes.addAndGet(nBytes);
        }

        void onRetry(int errorResourceCnt) {
        }

        void heartBeat() {
        }

        public boolean hasError() {
            return this.errorResource.get() > 0;
        }
    }
}

