/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.common.persistence;

import com.google.common.collect.Lists;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.persistence.BrokenEntity;
import org.apache.kylin.common.persistence.BrokenInputStream;
import org.apache.kylin.common.persistence.JDBCConnectionManager;
import org.apache.kylin.common.persistence.JDBCResource;
import org.apache.kylin.common.persistence.JDBCSqlQueryFormat;
import org.apache.kylin.common.persistence.JDBCSqlQueryFormatProvider;
import org.apache.kylin.common.persistence.WriteConflictException;
import org.apache.kylin.common.util.DBUtils;
import org.apache.kylin.common.util.HadoopUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JDBCResourceDAO {
    private static Logger logger = LoggerFactory.getLogger(JDBCResourceDAO.class);
    private static final String META_TABLE_KEY = "META_TABLE_KEY";
    private static final String META_TABLE_TS = "META_TABLE_TS";
    private static final String META_TABLE_CONTENT = "META_TABLE_CONTENT";
    private JDBCConnectionManager connectionManager;
    private JDBCSqlQueryFormat jdbcSqlQueryFormat;
    private String[] tableNames = new String[2];
    private KylinConfig kylinConfig;
    private long queriedSqlNum = 0L;
    private FileSystem redirectFileSystem;

    public JDBCResourceDAO(KylinConfig kylinConfig, String metadataIdentifier) throws SQLException {
        this.kylinConfig = kylinConfig;
        this.connectionManager = JDBCConnectionManager.getConnectionManager();
        this.jdbcSqlQueryFormat = JDBCSqlQueryFormatProvider.createJDBCSqlQueriesFormat(kylinConfig.getMetadataDialect());
        this.tableNames[0] = metadataIdentifier;
        this.tableNames[1] = metadataIdentifier + "_log";
        for (int i = 0; i < this.tableNames.length; ++i) {
            this.createTableIfNeeded(this.tableNames[i]);
            this.createIndex("IDX_META_TABLE_TS", this.tableNames[i], META_TABLE_TS);
        }
        try {
            this.redirectFileSystem = HadoopUtil.getReadFileSystem();
        }
        catch (IOException e) {
            throw new SQLException(e);
        }
    }

    public void close() {
        this.connectionManager.close();
    }

    public JDBCResource getResource(String resourcePath, boolean fetchContent, boolean fetchTimestamp) throws SQLException {
        return this.getResource(resourcePath, fetchContent, fetchTimestamp, false);
    }

    public JDBCResource getResource(final String resourcePath, final boolean fetchContent, final boolean fetchTimestamp, final boolean isAllowBroken) throws SQLException {
        final JDBCResource resource = new JDBCResource();
        logger.trace("getResource method. resourcePath : {} , fetchConetent : {} , fetch TS : {}", new Object[]{resourcePath, fetchContent, fetchTimestamp});
        this.executeSql(new SqlOperation(){

            @Override
            public void execute(Connection connection) throws SQLException {
                String tableName = JDBCResourceDAO.this.getMetaTableName(resourcePath);
                this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getKeyEqualSqlString(tableName, fetchContent, fetchTimestamp));
                this.pstat.setString(1, resourcePath);
                this.rs = this.pstat.executeQuery();
                if (this.rs.next()) {
                    resource.setPath(this.rs.getString(JDBCResourceDAO.META_TABLE_KEY));
                    if (fetchTimestamp) {
                        resource.setTimestamp(this.rs.getLong(JDBCResourceDAO.META_TABLE_TS));
                    }
                    if (fetchContent) {
                        try {
                            resource.setContent(JDBCResourceDAO.this.getInputStream(resourcePath, this.rs));
                        }
                        catch (Throwable e) {
                            if (!isAllowBroken) {
                                throw new SQLException(e);
                            }
                            BrokenEntity brokenEntity = new BrokenEntity(resourcePath, e.getMessage());
                            resource.setContent(new BrokenInputStream(brokenEntity));
                            logger.warn(e.getMessage());
                        }
                    }
                }
            }
        });
        if (resource.getPath() != null) {
            return resource;
        }
        return null;
    }

    public boolean existResource(String resourcePath) throws SQLException {
        JDBCResource resource = this.getResource(resourcePath, false, false);
        return resource != null;
    }

    public long getResourceTimestamp(String resourcePath) throws SQLException {
        JDBCResource resource = this.getResource(resourcePath, false, true);
        return resource == null ? 0L : resource.getTimestamp();
    }

    public TreeSet<String> listAllResource(final String folderPath, final boolean recursive) throws SQLException {
        final TreeSet<String> allResourceName = new TreeSet<String>();
        this.executeSql(new SqlOperation(){

            @Override
            public void execute(Connection connection) throws SQLException {
                String tableName = JDBCResourceDAO.this.getMetaTableName(folderPath);
                this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getListResourceSqlString(tableName));
                this.pstat.setString(1, folderPath + "%");
                this.rs = this.pstat.executeQuery();
                while (this.rs.next()) {
                    String path = this.rs.getString(JDBCResourceDAO.META_TABLE_KEY);
                    assert (path.startsWith(folderPath));
                    if (recursive) {
                        allResourceName.add(path);
                        continue;
                    }
                    int cut = path.indexOf(47, folderPath.length());
                    String child = cut < 0 ? path : path.substring(0, cut);
                    allResourceName.add(child);
                }
            }
        });
        return allResourceName;
    }

    public List<JDBCResource> getAllResource(final String folderPath, final long timeStart, final long timeEndExclusive, final boolean isAllowBroken) throws SQLException {
        final ArrayList allResource = Lists.newArrayList();
        this.executeSql(new SqlOperation(){

            @Override
            public void execute(Connection connection) throws SQLException {
                String tableName = JDBCResourceDAO.this.getMetaTableName(folderPath);
                this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getAllResourceSqlString(tableName));
                this.pstat.setString(1, folderPath + "%");
                this.pstat.setLong(2, timeStart);
                this.pstat.setLong(3, timeEndExclusive);
                this.rs = this.pstat.executeQuery();
                while (this.rs.next()) {
                    String resPath = this.rs.getString(JDBCResourceDAO.META_TABLE_KEY);
                    if (!JDBCResourceDAO.this.checkPath(folderPath, resPath)) continue;
                    JDBCResource resource = new JDBCResource();
                    resource.setPath(resPath);
                    resource.setTimestamp(this.rs.getLong(JDBCResourceDAO.META_TABLE_TS));
                    try {
                        resource.setContent(JDBCResourceDAO.this.getInputStream(resPath, this.rs));
                    }
                    catch (Throwable e) {
                        if (!isAllowBroken) {
                            throw new SQLException(e);
                        }
                        BrokenEntity brokenEntity = new BrokenEntity(resPath, e.getMessage());
                        resource.setContent(new BrokenInputStream(brokenEntity));
                        logger.warn(e.getMessage());
                    }
                    allResource.add(resource);
                }
            }
        });
        return allResource;
    }

    private boolean checkPath(String lookForPrefix, String resPath) {
        String string = lookForPrefix = lookForPrefix.endsWith("/") ? lookForPrefix : lookForPrefix + "/";
        assert (resPath.startsWith(lookForPrefix));
        int cut = resPath.indexOf(47, lookForPrefix.length());
        return cut < 0;
    }

    private boolean isJsonMetadata(String resourcePath) {
        String trim = resourcePath.trim();
        return trim.endsWith(".json") || trim.startsWith("/execute") || trim.startsWith("/execute_output");
    }

    public void deleteResource(final String resourcePath) throws SQLException {
        boolean skipHdfs = this.isJsonMetadata(resourcePath);
        this.executeSql(new SqlOperation(){

            @Override
            public void execute(Connection connection) throws SQLException {
                String tableName = JDBCResourceDAO.this.getMetaTableName(resourcePath);
                this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getDeletePstatSql(tableName));
                this.pstat.setString(1, resourcePath);
                this.pstat.executeUpdate();
            }
        });
        if (!skipHdfs) {
            try {
                this.deleteHDFSResourceIfExist(resourcePath);
            }
            catch (Throwable e) {
                throw new SQLException(e);
            }
        }
    }

    private void deleteHDFSResourceIfExist(String resourcePath) throws IOException {
        Path redirectPath = this.bigCellHDFSPath(resourcePath);
        if (this.redirectFileSystem.exists(redirectPath)) {
            this.redirectFileSystem.delete(redirectPath, true);
        }
    }

    public void putResource(final JDBCResource resource) throws SQLException {
        this.executeSql(new SqlOperation(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void execute(Connection connection) throws SQLException {
                byte[] content = JDBCResourceDAO.this.getResourceDataBytes(resource);
                String string = resource.getPath().intern();
                synchronized (string) {
                    boolean existing = JDBCResourceDAO.this.existResource(resource.getPath());
                    String tableName = JDBCResourceDAO.this.getMetaTableName(resource.getPath());
                    if (existing) {
                        this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getReplaceSql(tableName));
                        this.pstat.setLong(1, resource.getTimestamp());
                        this.pstat.setBlob(2, new BufferedInputStream(new ByteArrayInputStream(content)));
                        this.pstat.setString(3, resource.getPath());
                    } else {
                        this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getInsertSql(tableName));
                        this.pstat.setString(1, resource.getPath());
                        this.pstat.setLong(2, resource.getTimestamp());
                        this.pstat.setBlob(3, new BufferedInputStream(new ByteArrayInputStream(content)));
                    }
                    if (JDBCResourceDAO.this.isContentOverflow(content, resource.getPath())) {
                        logger.debug("Overflow! resource path: {}, content size: {}, timeStamp: {}", new Object[]{resource.getPath(), content.length, resource.getTimestamp()});
                        if (existing) {
                            this.pstat.setNull(2, 2004);
                        } else {
                            this.pstat.setNull(3, 2004);
                        }
                        JDBCResourceDAO.this.writeLargeCellToHdfs(resource.getPath(), content);
                        try {
                            int result = this.pstat.executeUpdate();
                            if (result != 1) {
                                throw new SQLException();
                            }
                        }
                        catch (SQLException e) {
                            JDBCResourceDAO.this.rollbackLargeCellFromHdfs(resource.getPath());
                            throw e;
                        }
                        if (existing) {
                            JDBCResourceDAO.this.cleanOldLargeCellFromHdfs(resource.getPath());
                        }
                    } else {
                        this.pstat.executeUpdate();
                    }
                }
            }
        });
    }

    public void checkAndPutResource(final String resPath, final byte[] content, final long oldTS, final long newTS) throws SQLException, WriteConflictException {
        logger.trace("execute checkAndPutResource method. resPath : {} , oldTs : {} , newTs : {} , content null ? : {} ", new Object[]{resPath, oldTS, newTS, content == null});
        this.executeSql(new SqlOperation(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             * Enabled aggressive block sorting
             * Enabled unnecessary exception pruning
             * Enabled aggressive exception aggregation
             */
            @Override
            public void execute(Connection connection) throws SQLException {
                String string = resPath.intern();
                synchronized (string) {
                    String tableName = JDBCResourceDAO.this.getMetaTableName(resPath);
                    if (!JDBCResourceDAO.this.existResource(resPath)) {
                        if (oldTS != 0L) {
                            throw new IllegalStateException("For not exist file. OldTS have to be 0. but Actual oldTS is : " + oldTS);
                        }
                        if (JDBCResourceDAO.this.isContentOverflow(content, resPath)) {
                            logger.debug("Overflow! resource path: {}, content size: {}", (Object)resPath, (Object)content.length);
                            this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getInsertSqlWithoutContent(tableName));
                            this.pstat.setString(1, resPath);
                            this.pstat.setLong(2, newTS);
                            JDBCResourceDAO.this.writeLargeCellToHdfs(resPath, content);
                            try {
                                int result = this.pstat.executeUpdate();
                                if (result == 1) return;
                                throw new SQLException();
                            }
                            catch (SQLException e) {
                                JDBCResourceDAO.this.rollbackLargeCellFromHdfs(resPath);
                                throw e;
                            }
                        }
                        this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getInsertSql(tableName));
                        this.pstat.setString(1, resPath);
                        this.pstat.setLong(2, newTS);
                        this.pstat.setBlob(3, new BufferedInputStream(new ByteArrayInputStream(content)));
                        this.pstat.executeUpdate();
                    } else {
                        PreparedStatement pstat2;
                        block15: {
                            this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getUpdateSqlWithoutContent(tableName));
                            this.pstat.setLong(1, newTS);
                            this.pstat.setString(2, resPath);
                            this.pstat.setLong(3, oldTS);
                            int result = this.pstat.executeUpdate();
                            if (result != 1) {
                                long realTime = JDBCResourceDAO.this.getResourceTimestamp(resPath);
                                throw new WriteConflictException("Overwriting conflict " + resPath + ", expect old TS " + oldTS + ", but it is " + realTime);
                            }
                            pstat2 = null;
                            try {
                                pstat2 = connection.prepareStatement(JDBCResourceDAO.this.getUpdateContentSql(tableName));
                                if (JDBCResourceDAO.this.isContentOverflow(content, resPath)) {
                                    logger.debug("Overflow! resource path: {}, content size: {}", (Object)resPath, (Object)content.length);
                                    pstat2.setNull(1, 2004);
                                    pstat2.setString(2, resPath);
                                    JDBCResourceDAO.this.writeLargeCellToHdfs(resPath, content);
                                    try {
                                        int result2 = pstat2.executeUpdate();
                                        if (result2 != 1) {
                                            throw new SQLException();
                                        }
                                    }
                                    catch (SQLException e) {
                                        JDBCResourceDAO.this.rollbackLargeCellFromHdfs(resPath);
                                        throw e;
                                    }
                                    JDBCResourceDAO.this.cleanOldLargeCellFromHdfs(resPath);
                                    break block15;
                                }
                                pstat2.setBinaryStream(1, new BufferedInputStream(new ByteArrayInputStream(content)));
                                pstat2.setString(2, resPath);
                                pstat2.executeUpdate();
                            }
                            catch (Throwable throwable) {
                                JDBCConnectionManager.closeQuietly(pstat2);
                                throw throwable;
                            }
                        }
                        JDBCConnectionManager.closeQuietly(pstat2);
                    }
                    return;
                }
            }
        });
    }

    private byte[] getResourceDataBytes(JDBCResource resource) throws SQLException {
        ByteArrayOutputStream bout = null;
        try {
            bout = new ByteArrayOutputStream();
            IOUtils.copy((InputStream)resource.getContent(), (OutputStream)bout);
            byte[] byArray = bout.toByteArray();
            return byArray;
        }
        catch (Throwable e) {
            throw new SQLException(e);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)bout);
        }
    }

    private boolean isContentOverflow(byte[] content, String resPath) throws SQLException {
        if (this.kylinConfig.isJsonAlwaysSmallCell() && this.isJsonMetadata(resPath)) {
            int smallCellMetadataWarningThreshold = this.kylinConfig.getSmallCellMetadataWarningThreshold();
            int smallCellMetadataErrorThreshold = this.kylinConfig.getSmallCellMetadataErrorThreshold();
            if (content.length > smallCellMetadataWarningThreshold) {
                logger.warn("A JSON metadata entry's size is not supposed to exceed kylin.metadata.jdbc.small-cell-meta-size-warning-threshold(" + smallCellMetadataWarningThreshold + "), resPath: " + resPath + ", actual size: " + content.length);
            }
            if (content.length > smallCellMetadataErrorThreshold) {
                throw new SQLException(new IllegalArgumentException("A JSON metadata entry's size is not supposed to exceed kylin.metadata.jdbc.small-cell-meta-size-error-threshold(" + smallCellMetadataErrorThreshold + "), resPath: " + resPath + ", actual size: " + content.length));
            }
            return false;
        }
        int maxSize = this.kylinConfig.getJdbcResourceStoreMaxCellSize();
        return content.length > maxSize;
    }

    private void createTableIfNeeded(final String tableName) throws SQLException {
        this.executeSql(new SqlOperation(){

            @Override
            public void execute(Connection connection) throws SQLException {
                if (this.checkTableExists(tableName, connection)) {
                    logger.info("Table [{}] already exists", (Object)tableName);
                    return;
                }
                this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getCreateIfNeededSql(tableName));
                this.pstat.executeUpdate();
                logger.info("Create table [{}] success", (Object)tableName);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private boolean checkTableExists(String tableName2, Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(JDBCResourceDAO.this.getCheckTableExistsSql(tableName2));
                ResultSet rs = ps.executeQuery();
                try {
                    while (rs.next()) {
                        if (!tableName2.equals(rs.getString(1))) continue;
                        boolean bl = true;
                        return bl;
                    }
                }
                finally {
                    DBUtils.closeQuietly(rs);
                    DBUtils.closeQuietly(ps);
                }
                return false;
            }
        });
    }

    private void createIndex(final String indexName, final String tableName, final String colName) {
        try {
            this.executeSql(new SqlOperation(){

                @Override
                public void execute(Connection connection) throws SQLException {
                    this.pstat = connection.prepareStatement(JDBCResourceDAO.this.getCreateIndexSql(indexName, tableName, colName));
                    this.pstat.executeUpdate();
                }
            });
        }
        catch (SQLException ex) {
            logger.info("Create index failed with message: " + ex.getLocalizedMessage());
        }
    }

    private void executeSql(SqlOperation operation) throws SQLException {
        Connection connection = null;
        try {
            connection = this.connectionManager.getConn();
            operation.execute(connection);
            ++this.queriedSqlNum;
        }
        finally {
            JDBCConnectionManager.closeQuietly(operation.rs);
            JDBCConnectionManager.closeQuietly(operation.pstat);
            JDBCConnectionManager.closeQuietly(connection);
        }
    }

    private String getCheckTableExistsSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getCheckTableExistsSql(), tableName);
        return sql;
    }

    private String getCreateIfNeededSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getCreateIfNeedSql(), tableName, META_TABLE_KEY, META_TABLE_TS, META_TABLE_CONTENT);
        return sql;
    }

    private String getCreateIndexSql(String indexName, String tableName, String indexCol) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getCreateIndexSql(), indexName, tableName, indexCol);
        return sql;
    }

    private String getKeyEqualSqlString(String tableName, boolean fetchContent, boolean fetchTimestamp) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getKeyEqualsSql(), this.getSelectList(fetchContent, fetchTimestamp), tableName, META_TABLE_KEY);
        return sql;
    }

    private String getDeletePstatSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getDeletePstatSql(), tableName, META_TABLE_KEY);
        return sql;
    }

    private String getListResourceSqlString(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getListResourceSql(), META_TABLE_KEY, tableName, META_TABLE_KEY);
        return sql;
    }

    private String getAllResourceSqlString(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getAllResourceSql(), this.getSelectList(true, true), tableName, META_TABLE_KEY, META_TABLE_TS, META_TABLE_TS);
        return sql;
    }

    private String getReplaceSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getReplaceSql(), tableName, META_TABLE_TS, META_TABLE_CONTENT, META_TABLE_KEY);
        return sql;
    }

    private String getInsertSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getInsertSql(), tableName, META_TABLE_KEY, META_TABLE_TS, META_TABLE_CONTENT);
        return sql;
    }

    private String getReplaceSqlWithoutContent(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getReplaceSqlWithoutContent(), tableName, META_TABLE_TS, META_TABLE_KEY);
        return sql;
    }

    private String getInsertSqlWithoutContent(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getInsertSqlWithoutContent(), tableName, META_TABLE_KEY, META_TABLE_TS);
        return sql;
    }

    private String getUpdateSqlWithoutContent(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getUpdateSqlWithoutContent(), tableName, META_TABLE_TS, META_TABLE_KEY, META_TABLE_TS);
        return sql;
    }

    private String getUpdateContentSql(String tableName) {
        String sql = MessageFormat.format(this.jdbcSqlQueryFormat.getUpdateContentSql(), tableName, META_TABLE_CONTENT, META_TABLE_KEY);
        return sql;
    }

    private String getSelectList(boolean fetchContent, boolean fetchTimestamp) {
        StringBuilder sb = new StringBuilder();
        sb.append(META_TABLE_KEY);
        if (fetchTimestamp) {
            sb.append(",META_TABLE_TS");
        }
        if (fetchContent) {
            sb.append(",META_TABLE_CONTENT");
        }
        return sb.toString();
    }

    private InputStream getInputStream(String resPath, ResultSet rs) throws SQLException, IOException {
        InputStream inputStream;
        if (rs == null) {
            return null;
        }
        InputStream inputStream2 = inputStream = rs.getBlob(META_TABLE_CONTENT) == null ? null : rs.getBlob(META_TABLE_CONTENT).getBinaryStream();
        if (inputStream != null) {
            return inputStream;
        }
        Path redirectPath = this.bigCellHDFSPath(resPath);
        return this.redirectFileSystem.open(redirectPath);
    }

    private Path writeLargeCellToHdfs(String resPath, byte[] largeColumn) throws SQLException {
        Path path;
        FSDataOutputStream out = null;
        Path redirectPath = this.bigCellHDFSPath(resPath);
        Path oldPath = new Path(redirectPath.toString() + "_old");
        try {
            boolean isResourceExist = this.redirectFileSystem.exists(redirectPath);
            if (isResourceExist) {
                FileUtil.copy((FileSystem)this.redirectFileSystem, (Path)redirectPath, (FileSystem)this.redirectFileSystem, (Path)oldPath, (boolean)false, (Configuration)HadoopUtil.getCurrentConfiguration());
                this.redirectFileSystem.delete(redirectPath, true);
                logger.debug("a copy of hdfs file {} is made", (Object)redirectPath);
            }
            out = this.redirectFileSystem.create(redirectPath);
            out.write(largeColumn);
            path = redirectPath;
        }
        catch (Throwable e) {
            try {
                try {
                    this.rollbackLargeCellFromHdfs(resPath);
                }
                catch (Throwable ex) {
                    logger.error("fail to roll back resource " + resPath + " in hdfs", ex);
                }
                throw new SQLException(e);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(out);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((OutputStream)out);
        return path;
    }

    public void rollbackLargeCellFromHdfs(String resPath) throws SQLException {
        Path redirectPath = this.bigCellHDFSPath(resPath);
        Path oldPath = new Path(redirectPath.toString() + "_old");
        try {
            if (this.redirectFileSystem.exists(oldPath)) {
                FileUtil.copy((FileSystem)this.redirectFileSystem, (Path)oldPath, (FileSystem)this.redirectFileSystem, (Path)redirectPath, (boolean)true, (boolean)true, (Configuration)HadoopUtil.getCurrentConfiguration());
                logger.info("roll back hdfs file {}", (Object)resPath);
            } else {
                this.redirectFileSystem.delete(redirectPath, true);
                logger.warn("no backup for hdfs file {} is found, clean it", (Object)resPath);
            }
        }
        catch (Throwable e) {
            try {
                this.redirectFileSystem.delete(redirectPath, true);
            }
            catch (Throwable ex) {
                logger.error("fail to delete resource " + redirectPath + " in hdfs", ex);
            }
            throw new SQLException(e);
        }
    }

    private void cleanOldLargeCellFromHdfs(String resPath) throws SQLException {
        Path redirectPath = this.bigCellHDFSPath(resPath);
        Path oldPath = new Path(redirectPath.toString() + "_old");
        try {
            if (this.redirectFileSystem.exists(oldPath)) {
                this.redirectFileSystem.delete(oldPath, true);
            }
        }
        catch (Throwable e) {
            logger.warn("error cleaning the backup file for " + redirectPath + ", leave it as garbage", e);
        }
    }

    public Path bigCellHDFSPath(String resPath) {
        String hdfsWorkingDirectory = this.kylinConfig.getHdfsWorkingDirectory();
        Path redirectPath = new Path(hdfsWorkingDirectory, "resources-jdbc" + resPath);
        redirectPath = Path.getPathWithoutSchemeAndAuthority((Path)redirectPath);
        return redirectPath;
    }

    public long getQueriedSqlNum() {
        return this.queriedSqlNum;
    }

    public String getMetaTableName(String resPath) {
        if (resPath.startsWith("/bad_query") || resPath.startsWith("/execute_output") || resPath.startsWith("/temp_statement")) {
            return this.tableNames[1];
        }
        return this.tableNames[0];
    }

    static abstract class SqlOperation {
        PreparedStatement pstat = null;
        ResultSet rs = null;

        SqlOperation() {
        }

        public abstract void execute(Connection var1) throws SQLException;
    }
}

