/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.storage.hbase;

import com.google.common.collect.Lists;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.NavigableSet;
import java.util.TreeSet;
import java.util.UUID;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.hadoop.hbase.filter.KeyOnlyFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueFilter;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.StorageURL;
import org.apache.kylin.common.persistence.RawResource;
import org.apache.kylin.common.persistence.ResourceStore;
import org.apache.kylin.common.persistence.StringEntity;
import org.apache.kylin.common.util.Bytes;
import org.apache.kylin.common.util.BytesUtil;
import org.apache.kylin.storage.hbase.HBaseConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HBaseResourceStore
extends ResourceStore {
    private static final Logger logger = LoggerFactory.getLogger(HBaseResourceStore.class);
    private static final String FAMILY = "f";
    private static final byte[] B_FAMILY = Bytes.toBytes("f");
    private static final String COLUMN = "c";
    private static final byte[] B_COLUMN = Bytes.toBytes("c");
    private static final String COLUMN_TS = "t";
    private static final byte[] B_COLUMN_TS = Bytes.toBytes("t");
    final String tableName;
    final StorageURL metadataUrl;

    Connection getConnection() throws IOException {
        return HBaseConnection.get(this.metadataUrl);
    }

    public HBaseResourceStore(KylinConfig kylinConfig) throws IOException {
        super(kylinConfig);
        this.metadataUrl = this.buildMetadataUrl(kylinConfig);
        this.tableName = this.metadataUrl.getIdentifier();
        this.createHTableIfNeeded(this.tableName);
    }

    private StorageURL buildMetadataUrl(KylinConfig kylinConfig) throws IOException {
        StorageURL url = kylinConfig.getMetadataUrl();
        if (!url.getScheme().equals("hbase")) {
            throw new IOException("Cannot create HBaseResourceStore. Url not match. Url: " + url);
        }
        LinkedHashMap<String, String> newParams = new LinkedHashMap<String, String>();
        newParams.put("hbase.client.scanner.timeout.period", "10000");
        newParams.put("hbase.rpc.timeout", "5000");
        newParams.put("hbase.client.retries.number", "1");
        newParams.putAll(url.getAllParameters());
        return url.copy(newParams);
    }

    private void createHTableIfNeeded(String tableName) throws IOException {
        HBaseConnection.createHTableIfNeeded(this.getConnection(), tableName, FAMILY);
    }

    @Override
    protected boolean existsImpl(String resPath) throws IOException {
        Result r = this.getFromHTable(resPath, false, false);
        return r != null;
    }

    @Override
    protected NavigableSet<String> listResourcesImpl(String folderPath) throws IOException {
        final TreeSet<String> result = new TreeSet<String>();
        this.visitFolder(folderPath, (Filter)new KeyOnlyFilter(), new FolderVisitor(){

            @Override
            public void visit(String childPath, String fullPath, Result hbaseResult) {
                result.add(childPath);
            }
        });
        return result.isEmpty() ? null : result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public String createMetaStoreUUID() throws IOException {
        try (Admin hbaseAdmin = HBaseConnection.get(this.metadataUrl).getAdmin();){
            String metaStoreName = this.metadataUrl.getIdentifier();
            HTableDescriptor desc = hbaseAdmin.getTableDescriptor(TableName.valueOf((String)metaStoreName));
            String uuid = desc.getValue("UUID");
            if (uuid != null) {
                String string2 = uuid;
                return string2;
            }
            String string = UUID.randomUUID().toString();
            return string;
        }
        catch (Exception e) {
            return null;
        }
    }

    @Override
    public String getMetaStoreUUID() throws IOException {
        if (!this.exists("/UUID")) {
            this.putResource("/UUID", new StringEntity(this.createMetaStoreUUID()), 0L, StringEntity.serializer);
        }
        StringEntity entity = this.getResource("/UUID", StringEntity.class, StringEntity.serializer);
        return entity.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void visitFolder(String folderPath, Filter filter, FolderVisitor visitor) throws IOException {
        assert (folderPath.startsWith("/"));
        String lookForPrefix = folderPath.endsWith("/") ? folderPath : folderPath + "/";
        byte[] startRow = Bytes.toBytes(lookForPrefix);
        byte[] endRow = Bytes.toBytes(lookForPrefix);
        int n = endRow.length - 1;
        endRow[n] = (byte)(endRow[n] + 1);
        Table table = this.getConnection().getTable(TableName.valueOf((String)this.tableName));
        Scan scan = new Scan(startRow, endRow);
        if (!(filter != null && filter instanceof KeyOnlyFilter)) {
            scan.addColumn(B_FAMILY, B_COLUMN_TS);
            scan.addColumn(B_FAMILY, B_COLUMN);
        }
        if (filter != null) {
            scan.setFilter(filter);
        }
        this.tuneScanParameters(scan);
        try {
            ResultScanner scanner = table.getScanner(scan);
            for (Result r : scanner) {
                String path = Bytes.toString(r.getRow());
                assert (path.startsWith(lookForPrefix));
                int cut = path.indexOf(47, lookForPrefix.length());
                String child = cut < 0 ? path : path.substring(0, cut);
                visitor.visit(child, path, r);
            }
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    private void tuneScanParameters(Scan scan) {
        scan.setCaching(this.kylinConfig.getHBaseScanCacheRows());
        scan.setMaxResultSize((long)this.kylinConfig.getHBaseScanMaxResultSize());
        scan.setCacheBlocks(true);
    }

    @Override
    protected List<RawResource> getAllResourcesImpl(String folderPath, long timeStart, long timeEndExclusive) throws IOException {
        FilterList filter = this.generateTimeFilterList(timeStart, timeEndExclusive);
        final ArrayList result = Lists.newArrayList();
        try {
            this.visitFolder(folderPath, (Filter)filter, new FolderVisitor(){

                @Override
                public void visit(String childPath, String fullPath, Result hbaseResult) throws IOException {
                    if (childPath.equals(fullPath)) {
                        result.add(new RawResource(HBaseResourceStore.this.getInputStream(childPath, hbaseResult), HBaseResourceStore.this.getTimestamp(hbaseResult)));
                    }
                }
            });
        }
        catch (IOException e) {
            for (RawResource rawResource : result) {
                IOUtils.closeQuietly((InputStream)rawResource.inputStream);
            }
            throw e;
        }
        return result;
    }

    private FilterList generateTimeFilterList(long timeStart, long timeEndExclusive) {
        FilterList filterList = new FilterList(FilterList.Operator.MUST_PASS_ALL);
        if (timeStart != Long.MIN_VALUE) {
            SingleColumnValueFilter timeStartFilter = new SingleColumnValueFilter(B_FAMILY, B_COLUMN_TS, CompareFilter.CompareOp.GREATER_OR_EQUAL, Bytes.toBytes(timeStart));
            filterList.addFilter((Filter)timeStartFilter);
        }
        if (timeEndExclusive != Long.MAX_VALUE) {
            SingleColumnValueFilter timeEndFilter = new SingleColumnValueFilter(B_FAMILY, B_COLUMN_TS, CompareFilter.CompareOp.LESS, Bytes.toBytes(timeEndExclusive));
            filterList.addFilter((Filter)timeEndFilter);
        }
        return filterList.getFilters().size() == 0 ? null : filterList;
    }

    private InputStream getInputStream(String resPath, Result r) throws IOException {
        if (r == null) {
            return null;
        }
        byte[] value = r.getValue(B_FAMILY, B_COLUMN);
        if (value.length == 0) {
            Path redirectPath = this.bigCellHDFSPath(resPath);
            FileSystem fileSystem = FileSystem.get((Configuration)HBaseConnection.getCurrentHBaseConfiguration());
            try {
                return fileSystem.open(redirectPath);
            }
            catch (IOException ex) {
                throw new IOException("Failed to read resource at " + resPath, ex);
            }
        }
        return new ByteArrayInputStream(value);
    }

    private long getTimestamp(Result r) {
        if (r == null || r.getValue(B_FAMILY, B_COLUMN_TS) == null) {
            return 0L;
        }
        return Bytes.toLong(r.getValue(B_FAMILY, B_COLUMN_TS));
    }

    @Override
    protected RawResource getResourceImpl(String resPath) throws IOException {
        Result r = this.getFromHTable(resPath, true, true);
        if (r == null) {
            return null;
        }
        return new RawResource(this.getInputStream(resPath, r), this.getTimestamp(r));
    }

    @Override
    protected long getResourceTimestampImpl(String resPath) throws IOException {
        return this.getTimestamp(this.getFromHTable(resPath, false, true));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void putResourceImpl(String resPath, InputStream content, long ts) throws IOException {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        IOUtils.copy((InputStream)content, (OutputStream)bout);
        bout.close();
        Table table = this.getConnection().getTable(TableName.valueOf((String)this.tableName));
        try {
            byte[] row = Bytes.toBytes(resPath);
            Put put = this.buildPut(resPath, ts, row, bout.toByteArray(), table);
            table.put(put);
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected long checkAndPutResourceImpl(String resPath, byte[] content, long oldTS, long newTS) throws IOException, IllegalStateException {
        Table table = this.getConnection().getTable(TableName.valueOf((String)this.tableName));
        try {
            byte[] row = Bytes.toBytes(resPath);
            byte[] bOldTS = oldTS == 0L ? null : Bytes.toBytes(oldTS);
            Put put = this.buildPut(resPath, newTS, row, content, table);
            boolean ok = table.checkAndPut(row, B_FAMILY, B_COLUMN_TS, bOldTS, put);
            logger.trace("Update row " + resPath + " from oldTs: " + oldTS + ", to newTs: " + newTS + ", operation result: " + ok);
            if (!ok) {
                long real = this.getResourceTimestampImpl(resPath);
                throw new IllegalStateException("Overwriting conflict " + resPath + ", expect old TS " + oldTS + ", but it is " + real);
            }
            long l = newTS;
            return l;
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void deleteResourceImpl(String resPath) throws IOException {
        Table table = this.getConnection().getTable(TableName.valueOf((String)this.tableName));
        try {
            byte[] value;
            boolean hdfsResourceExist = false;
            Result result = this.internalGetFromHTable(table, resPath, true, false);
            if (result != null && (value = result.getValue(B_FAMILY, B_COLUMN)) != null && value.length == 0) {
                hdfsResourceExist = true;
            }
            Delete del = new Delete(Bytes.toBytes(resPath));
            table.delete(del);
            if (hdfsResourceExist) {
                Path redirectPath = this.bigCellHDFSPath(resPath);
                FileSystem fileSystem = FileSystem.get((Configuration)HBaseConnection.getCurrentHBaseConfiguration());
                if (fileSystem.exists(redirectPath)) {
                    fileSystem.delete(redirectPath, true);
                }
            }
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    @Override
    protected String getReadableResourcePathImpl(String resPath) {
        return this.tableName + "(key='" + resPath + "')@" + this.kylinConfig.getMetadataUrl();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Result getFromHTable(String path, boolean fetchContent, boolean fetchTimestamp) throws IOException {
        Table table = this.getConnection().getTable(TableName.valueOf((String)this.tableName));
        try {
            Result result = this.internalGetFromHTable(table, path, fetchContent, fetchTimestamp);
            return result;
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    private Result internalGetFromHTable(Table table, String path, boolean fetchContent, boolean fetchTimestamp) throws IOException {
        byte[] rowkey = Bytes.toBytes(path);
        Get get = new Get(rowkey);
        if (!fetchContent && !fetchTimestamp) {
            get.setCheckExistenceOnly(true);
        } else {
            if (fetchContent) {
                get.addColumn(B_FAMILY, B_COLUMN);
            }
            if (fetchTimestamp) {
                get.addColumn(B_FAMILY, B_COLUMN_TS);
            }
        }
        Result result = table.get(get);
        boolean exists = result != null && (!result.isEmpty() || result.getExists() != null && result.getExists() != false);
        return exists ? result : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Path writeLargeCellToHdfs(String resPath, byte[] largeColumn, Table table) throws IOException {
        Path redirectPath = this.bigCellHDFSPath(resPath);
        FileSystem fileSystem = FileSystem.get((Configuration)HBaseConnection.getCurrentHBaseConfiguration());
        if (fileSystem.exists(redirectPath)) {
            fileSystem.delete(redirectPath, true);
        }
        FSDataOutputStream out = fileSystem.create(redirectPath);
        try {
            out.write(largeColumn);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)out);
        }
        return redirectPath;
    }

    public Path bigCellHDFSPath(String resPath) {
        String hdfsWorkingDirectory = this.kylinConfig.getHdfsWorkingDirectory();
        Path redirectPath = new Path(hdfsWorkingDirectory, "resources" + resPath);
        return redirectPath;
    }

    private Put buildPut(String resPath, long ts, byte[] row, byte[] content, Table table) throws IOException {
        int kvSizeLimit = Integer.parseInt(this.getConnection().getConfiguration().get("hbase.client.keyvalue.maxsize", "10485760"));
        if (content.length > kvSizeLimit) {
            this.writeLargeCellToHdfs(resPath, content, table);
            content = BytesUtil.EMPTY_BYTE_ARRAY;
        }
        Put put = new Put(row);
        put.addColumn(B_FAMILY, B_COLUMN, content);
        put.addColumn(B_FAMILY, B_COLUMN_TS, Bytes.toBytes(ts));
        return put;
    }

    public String toString() {
        return this.tableName + "@hbase";
    }

    static interface FolderVisitor {
        public void visit(String var1, String var2, Result var3) throws IOException;
    }
}

