/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.dict.global;

import com.google.common.base.Preconditions;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.util.ClassUtil;
import org.apache.kylin.common.util.HadoopUtil;
import org.apache.kylin.dict.BytesConverter;
import org.apache.kylin.dict.global.AppendDictNode;
import org.apache.kylin.dict.global.AppendDictSlice;
import org.apache.kylin.dict.global.AppendDictSliceKey;
import org.apache.kylin.dict.global.GlobalDictMetadata;
import org.apache.kylin.dict.global.GlobalDictStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GlobalDictHDFSStore
extends GlobalDictStore {
    static final Logger logger = LoggerFactory.getLogger(GlobalDictHDFSStore.class);
    static final String V1_INDEX_NAME = ".index";
    static final String V2_INDEX_NAME = ".index_v2";
    static final String VERSION_PREFIX = "version_";
    static final int BUFFER_SIZE = 0x800000;
    private final Path basePath;
    private final Configuration conf;
    private final FileSystem fileSystem;

    public GlobalDictHDFSStore(String baseDir) throws IOException {
        super(baseDir);
        this.basePath = new Path(baseDir);
        this.conf = HadoopUtil.getCurrentConfiguration();
        this.fileSystem = HadoopUtil.getFileSystem(baseDir);
        if (!this.fileSystem.exists(this.basePath)) {
            logger.info("Global dict at {} doesn't exist, create a new one", (Object)this.basePath);
            this.fileSystem.mkdirs(this.basePath);
        }
        this.migrateOldLayout();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void migrateOldLayout() throws IOException {
        FileStatus[] sliceFiles = this.fileSystem.listStatus(this.basePath, new PathFilter(){

            public boolean accept(Path path) {
                return path.getName().startsWith("cached_");
            }
        });
        Path indexFile = new Path(this.basePath, V1_INDEX_NAME);
        if (this.fileSystem.exists(indexFile) && sliceFiles.length > 0) {
            long version = System.currentTimeMillis();
            Path tempDir = new Path(this.basePath, "tmp_version_" + version);
            Path versionDir = this.getVersionDir(version);
            logger.info("Convert global dict at {} to new layout with version {}", (Object)this.basePath, (Object)version);
            this.fileSystem.mkdirs(tempDir);
            try {
                FileUtil.copy((FileSystem)this.fileSystem, (Path)indexFile, (FileSystem)this.fileSystem, (Path)tempDir, (boolean)false, (Configuration)this.conf);
                for (FileStatus sliceFile : sliceFiles) {
                    FileUtil.copy((FileSystem)this.fileSystem, (Path)sliceFile.getPath(), (FileSystem)this.fileSystem, (Path)tempDir, (boolean)false, (Configuration)this.conf);
                }
                this.fileSystem.rename(tempDir, versionDir);
                this.fileSystem.delete(indexFile, false);
                for (FileStatus sliceFile : sliceFiles) {
                    this.fileSystem.delete(sliceFile.getPath(), true);
                }
            }
            finally {
                if (this.fileSystem.exists(tempDir)) {
                    this.fileSystem.delete(tempDir, true);
                }
            }
        }
    }

    @Override
    void prepareForWrite(String workingDir) throws IOException {
        Long[] versions;
        Path working = new Path(workingDir);
        if (this.fileSystem.exists(working)) {
            this.fileSystem.delete(working, true);
            logger.info("Working directory {} exits, delete it first", (Object)working);
        }
        if ((versions = this.listAllVersions()).length > 0) {
            Path latestVersion = this.getVersionDir(versions[versions.length - 1]);
            FileUtil.copy((FileSystem)this.fileSystem, (Path)latestVersion, (FileSystem)this.fileSystem, (Path)working, (boolean)false, (boolean)true, (Configuration)this.conf);
        } else {
            this.fileSystem.mkdirs(working);
        }
    }

    @Override
    public Long[] listAllVersions() throws IOException {
        FileStatus[] versionDirs = this.fileSystem.listStatus(this.basePath, new PathFilter(){

            public boolean accept(Path path) {
                return path.getName().startsWith(GlobalDictHDFSStore.VERSION_PREFIX);
            }
        });
        TreeSet<Long> versions = new TreeSet<Long>();
        for (int i = 0; i < versionDirs.length; ++i) {
            Path path = versionDirs[i].getPath();
            versions.add(Long.parseLong(path.getName().substring(VERSION_PREFIX.length())));
        }
        return versions.toArray(new Long[versions.size()]);
    }

    @Override
    public Path getVersionDir(long version) {
        return new Path(this.basePath, VERSION_PREFIX + version);
    }

    @Override
    public GlobalDictMetadata getMetadata(long version) throws IOException {
        IndexFormatV1 format;
        Path versionDir = this.getVersionDir(version);
        Object[] indexFiles = this.fileSystem.listStatus(versionDir, new PathFilter(){

            public boolean accept(Path path) {
                return path.getName().startsWith(GlobalDictHDFSStore.V1_INDEX_NAME);
            }
        });
        Preconditions.checkState((indexFiles.length == 1 ? 1 : 0) != 0, (String)"zero or more than one index file found: %s", (Object[])new Object[]{Arrays.toString(indexFiles)});
        String indexFile = indexFiles[0].getPath().getName();
        if (V2_INDEX_NAME.equals(indexFile)) {
            format = new IndexFormatV2(this.fileSystem, this.conf);
        } else if (V1_INDEX_NAME.equals(indexFile)) {
            format = new IndexFormatV1(this.fileSystem, this.conf);
        } else {
            throw new RuntimeException("Unknown index file: " + indexFile);
        }
        return format.readIndexFile(versionDir);
    }

    @Override
    public AppendDictSlice readSlice(String directory, String sliceFileName) throws IOException {
        Path path = new Path(directory, sliceFileName);
        logger.info("read slice from {}", (Object)path);
        try (FSDataInputStream input = this.fileSystem.open(path, 0x800000);){
            AppendDictSlice appendDictSlice = AppendDictSlice.deserializeFrom((DataInput)input);
            return appendDictSlice;
        }
    }

    @Override
    public String writeSlice(String workingDir, AppendDictSliceKey key, AppendDictNode slice) throws IOException {
        String sliceFile = IndexFormatV2.sliceFileName(key);
        Path path = new Path(workingDir, sliceFile);
        logger.info("write slice with key {} into file {}", (Object)key, (Object)path);
        try (FSDataOutputStream out = this.fileSystem.create(path, true, 0x800000);){
            byte[] bytes = slice.buildTrieBytes();
            out.write(bytes);
        }
        return sliceFile;
    }

    @Override
    public void deleteSlice(String workingDir, String sliceFileName) throws IOException {
        Path path = new Path(workingDir, sliceFileName);
        logger.info("delete slice at {}", (Object)path);
        if (this.fileSystem.exists(path)) {
            this.fileSystem.delete(path, false);
        }
    }

    @Override
    public void commit(String workingDir, GlobalDictMetadata metadata) throws IOException {
        Path workingPath = new Path(workingDir);
        Path oldIndexFile = new Path(workingPath, V1_INDEX_NAME);
        if (this.fileSystem.exists(oldIndexFile)) {
            this.fileSystem.delete(oldIndexFile, false);
        }
        IndexFormatV2 index = new IndexFormatV2(this.fileSystem, this.conf);
        index.writeIndexFile(workingPath, metadata);
        index.sanityCheck(workingPath, metadata);
        Path newVersionPath = new Path(this.basePath, VERSION_PREFIX + System.currentTimeMillis());
        this.fileSystem.rename(workingPath, newVersionPath);
        this.cleanUp();
    }

    private void cleanUp() throws IOException {
        Long[] versions = this.listAllVersions();
        long timestamp = System.currentTimeMillis();
        for (int i = 0; i < versions.length - this.maxVersions; ++i) {
            if (versions[i] + (long)this.versionTTL >= timestamp) continue;
            this.fileSystem.delete(this.getVersionDir(versions[i]), true);
        }
    }

    @Override
    public String copyToAnotherMeta(KylinConfig srcConfig, KylinConfig dstConfig) throws IOException {
        Preconditions.checkArgument((boolean)this.baseDir.startsWith(srcConfig.getHdfsWorkingDirectory()), (String)"Please check why current directory {} doesn't belong to source working directory {}", (Object[])new Object[]{this.baseDir, srcConfig.getHdfsWorkingDirectory()});
        String dstBaseDir = this.baseDir.replaceFirst(srcConfig.getHdfsWorkingDirectory(), dstConfig.getHdfsWorkingDirectory());
        Long[] versions = this.listAllVersions();
        if (versions.length == 0) {
            return dstBaseDir;
        }
        Path srcVersionDir = this.getVersionDir(versions[versions.length - 1]);
        Path dstVersionDir = new Path(srcVersionDir.toString().replaceFirst(srcConfig.getHdfsWorkingDirectory(), dstConfig.getHdfsWorkingDirectory()));
        FileSystem dstFS = dstVersionDir.getFileSystem(this.conf);
        if (dstFS.exists(dstVersionDir)) {
            dstFS.delete(dstVersionDir, true);
        }
        FileUtil.copy((FileSystem)this.fileSystem, (Path)srcVersionDir, (FileSystem)dstFS, (Path)dstVersionDir, (boolean)false, (boolean)true, (Configuration)this.conf);
        return dstBaseDir;
    }

    public static class IndexFormatV2
    extends IndexFormatV1 {
        static final String SLICE_PREFIX = "cached_";
        static final int MINOR_VERSION_V1 = 1;

        protected IndexFormatV2(FileSystem fs, Configuration conf) {
            super(fs, conf);
        }

        @Override
        public GlobalDictMetadata readIndexFile(Path dir) throws IOException {
            Path indexFile = new Path(dir, GlobalDictHDFSStore.V2_INDEX_NAME);
            try (FSDataInputStream in = this.fs.open(indexFile);){
                BytesConverter converter;
                byte minorVersion = in.readByte();
                if (minorVersion != 1) {
                    throw new RuntimeException("Unsupported minor version " + minorVersion);
                }
                int baseId = in.readInt();
                int maxId = in.readInt();
                int maxValueLength = in.readInt();
                int nValues = in.readInt();
                String converterName = in.readUTF();
                try {
                    converter = ClassUtil.forName(converterName, BytesConverter.class).newInstance();
                }
                catch (Exception e) {
                    throw new RuntimeException("Fail to instantiate BytesConverter: " + converterName, e);
                }
                int nSlices = in.readInt();
                TreeMap<AppendDictSliceKey, String> sliceFileMap = new TreeMap<AppendDictSliceKey, String>();
                for (int i = 0; i < nSlices; ++i) {
                    AppendDictSliceKey key = new AppendDictSliceKey();
                    key.readFields((DataInput)in);
                    String sliceFileName = in.readUTF();
                    sliceFileMap.put(key, sliceFileName);
                }
                GlobalDictMetadata globalDictMetadata = new GlobalDictMetadata(baseId, maxId, maxValueLength, nValues, converter, sliceFileMap);
                return globalDictMetadata;
            }
        }

        @Override
        public void writeIndexFile(Path dir, GlobalDictMetadata metadata) throws IOException {
            Path indexFile = new Path(dir, GlobalDictHDFSStore.V2_INDEX_NAME);
            try (FSDataOutputStream out = this.fs.create(indexFile, true);){
                out.writeByte(1);
                out.writeInt(metadata.baseId);
                out.writeInt(metadata.maxId);
                out.writeInt(metadata.maxValueLength);
                out.writeInt(metadata.nValues);
                out.writeUTF(metadata.bytesConverter.getClass().getName());
                out.writeInt(metadata.sliceFileMap.size());
                for (Map.Entry<AppendDictSliceKey, String> entry : metadata.sliceFileMap.entrySet()) {
                    entry.getKey().write((DataOutput)out);
                    out.writeUTF(entry.getValue());
                }
            }
        }

        @Override
        public void sanityCheck(Path dir, GlobalDictMetadata metadata) throws IOException {
            for (Map.Entry<AppendDictSliceKey, String> entry : metadata.sliceFileMap.entrySet()) {
                if (this.fs.exists(new Path(dir, entry.getValue()))) continue;
                throw new RuntimeException("The slice file " + entry.getValue() + " for the key: " + entry.getKey() + " must be existed!");
            }
        }

        public static String sliceFileName(AppendDictSliceKey key) {
            return String.format("%s%d_%d", SLICE_PREFIX, System.currentTimeMillis(), key.hashCode());
        }
    }

    public static class IndexFormatV1
    implements IndexFormat {
        static final String SLICE_PREFIX = "cached_";
        protected final FileSystem fs;
        protected final Configuration conf;

        protected IndexFormatV1(FileSystem fs, Configuration conf) {
            this.fs = fs;
            this.conf = conf;
        }

        @Override
        public GlobalDictMetadata readIndexFile(Path dir) throws IOException {
            Path indexFile = new Path(dir, GlobalDictHDFSStore.V1_INDEX_NAME);
            try (FSDataInputStream in = this.fs.open(indexFile);){
                BytesConverter converter;
                int baseId = in.readInt();
                int maxId = in.readInt();
                int maxValueLength = in.readInt();
                int nValues = in.readInt();
                String converterName = in.readUTF();
                try {
                    converter = ClassUtil.forName(converterName, BytesConverter.class).newInstance();
                }
                catch (Exception e) {
                    throw new RuntimeException("Fail to instantiate BytesConverter: " + converterName, e);
                }
                int nSlices = in.readInt();
                TreeMap<AppendDictSliceKey, String> sliceFileMap = new TreeMap<AppendDictSliceKey, String>();
                for (int i = 0; i < nSlices; ++i) {
                    AppendDictSliceKey key = new AppendDictSliceKey();
                    key.readFields((DataInput)in);
                    sliceFileMap.put(key, IndexFormatV1.sliceFileName(key));
                }
                String firstFile = (String)sliceFileMap.remove(sliceFileMap.firstKey());
                sliceFileMap.put(AppendDictSliceKey.START_KEY, firstFile);
                GlobalDictMetadata globalDictMetadata = new GlobalDictMetadata(baseId, maxId, maxValueLength, nValues, converter, sliceFileMap);
                return globalDictMetadata;
            }
        }

        @Override
        public void writeIndexFile(Path dir, GlobalDictMetadata metadata) throws IOException {
            Path indexFile = new Path(dir, GlobalDictHDFSStore.V1_INDEX_NAME);
            try (FSDataOutputStream out = this.fs.create(indexFile, true);){
                out.writeInt(metadata.baseId);
                out.writeInt(metadata.maxId);
                out.writeInt(metadata.maxValueLength);
                out.writeInt(metadata.nValues);
                out.writeUTF(metadata.bytesConverter.getClass().getName());
                out.writeInt(metadata.sliceFileMap.size());
                for (Map.Entry<AppendDictSliceKey, String> entry : metadata.sliceFileMap.entrySet()) {
                    entry.getKey().write((DataOutput)out);
                }
            }
        }

        @Override
        public void sanityCheck(Path dir, GlobalDictMetadata metadata) throws IOException {
            throw new UnsupportedOperationException("sanityCheck V1 format is no longer supported");
        }

        public static String sliceFileName(AppendDictSliceKey key) {
            return SLICE_PREFIX + key;
        }
    }

    public static interface IndexFormat {
        public GlobalDictMetadata readIndexFile(Path var1) throws IOException;

        public void writeIndexFile(Path var1, GlobalDictMetadata var2) throws IOException;

        public void sanityCheck(Path var1, GlobalDictMetadata var2) throws IOException;
    }
}

