/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Properties;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.header.internals.RecordHeader;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.tools.LineMessageReader;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class LineMessageReaderTest {
    @Test
    public void testLineReader() {
        String input = "key0\tvalue0\nkey1\tvalue1";
        Properties props = this.defaultTestProps();
        props.put("parse.headers", "false");
        this.runTest(props, input, this.record("key0", "value0"), this.record("key1", "value1"));
    }

    @Test
    public void testLineReaderHeader() {
        String input = "headerKey0:headerValue0,headerKey1:headerValue1\tkey0\tvalue0\n";
        ProducerRecord<String, String> expected = this.record("key0", "value0", Arrays.asList(new RecordHeader("headerKey0", "headerValue0".getBytes(StandardCharsets.UTF_8)), new RecordHeader("headerKey1", "headerValue1".getBytes(StandardCharsets.UTF_8))));
        this.runTest(this.defaultTestProps(), input, expected);
    }

    @Test
    public void testMinimalValidInputWithHeaderKeyAndValue() {
        this.runTest(this.defaultTestProps(), ":\t\t", this.record("", "", Collections.singletonList(new RecordHeader("", "".getBytes(StandardCharsets.UTF_8)))));
    }

    @Test
    public void testKeyMissingValue() {
        Properties props = this.defaultTestProps();
        props.put("parse.headers", "false");
        this.runTest(props, "key\t", this.record("key", ""));
    }

    @Test
    public void testDemarcationsLongerThanOne() {
        Properties props = this.defaultTestProps();
        props.put("key.separator", "\t\t");
        props.put("headers.delimiter", "\t\t");
        props.put("headers.separator", "---");
        props.put("headers.key.separator", "::::");
        this.runTest(props, "headerKey0.0::::headerValue0.0---headerKey1.0::::\t\tkey\t\tvalue", this.record("key", "value", Arrays.asList(new RecordHeader("headerKey0.0", "headerValue0.0".getBytes(StandardCharsets.UTF_8)), new RecordHeader("headerKey1.0", "".getBytes(StandardCharsets.UTF_8)))));
    }

    @Test
    public void testLineReaderHeaderNoKey() {
        String input = "headerKey:headerValue\tvalue\n";
        Properties props = this.defaultTestProps();
        props.put("parse.key", "false");
        this.runTest(props, input, this.record(null, "value", Collections.singletonList(new RecordHeader("headerKey", "headerValue".getBytes(StandardCharsets.UTF_8)))));
    }

    @Test
    public void testLineReaderOnlyValue() {
        Properties props = this.defaultTestProps();
        props.put("parse.key", "false");
        props.put("parse.headers", "false");
        this.runTest(props, "value\n", this.record(null, "value"));
    }

    @Test
    public void testParseHeaderEnabledWithCustomDelimiterAndVaryingNumberOfKeyValueHeaderPairs() {
        Properties props = this.defaultTestProps();
        props.put("key.separator", "#");
        props.put("headers.delimiter", "!");
        props.put("headers.separator", "&");
        props.put("headers.key.separator", ":");
        String input = "headerKey0.0:headerValue0.0&headerKey0.1:headerValue0.1!key0#value0\nheaderKey1.0:headerValue1.0!key1#value1";
        ProducerRecord<String, String> record0 = this.record("key0", "value0", Arrays.asList(new RecordHeader("headerKey0.0", "headerValue0.0".getBytes(StandardCharsets.UTF_8)), new RecordHeader("headerKey0.1", "headerValue0.1".getBytes(StandardCharsets.UTF_8))));
        ProducerRecord<String, String> record1 = this.record("key1", "value1", Collections.singletonList(new RecordHeader("headerKey1.0", "headerValue1.0".getBytes(StandardCharsets.UTF_8))));
        this.runTest(props, input, record0, record1);
    }

    @Test
    public void testMissingKeySeparator() {
        LineMessageReader lineReader = new LineMessageReader();
        String input = "headerKey0.0:headerValue0.0,headerKey0.1:headerValue0.1\tkey0\tvalue0\nheaderKey1.0:headerValue1.0\tkey1[MISSING-DELIMITER]value1";
        lineReader.configure(Utils.propsToStringMap((Properties)this.defaultTestProps()));
        Iterator iter = lineReader.readRecords((InputStream)new ByteArrayInputStream(input.getBytes()));
        iter.next();
        KafkaException expectedException = (KafkaException)Assertions.assertThrows(KafkaException.class, iter::next);
        Assertions.assertEquals((Object)"No key separator found on line number 2: 'headerKey1.0:headerValue1.0\tkey1[MISSING-DELIMITER]value1'", (Object)expectedException.getMessage());
    }

    @Test
    public void testMissingHeaderKeySeparator() {
        LineMessageReader lineReader = new LineMessageReader();
        String input = "key[MISSING-DELIMITER]val\tkey0\tvalue0\n";
        lineReader.configure(Utils.propsToStringMap((Properties)this.defaultTestProps()));
        Iterator iter = lineReader.readRecords((InputStream)new ByteArrayInputStream(input.getBytes()));
        KafkaException expectedException = (KafkaException)Assertions.assertThrows(KafkaException.class, iter::next);
        Assertions.assertEquals((Object)"No header key separator found in pair 'key[MISSING-DELIMITER]val' on line number 1", (Object)expectedException.getMessage());
    }

    @Test
    public void testHeaderDemarcationCollision() {
        Properties props = this.defaultTestProps();
        props.put("headers.delimiter", "\t");
        props.put("headers.separator", "\t");
        props.put("headers.key.separator", "\t");
        this.assertThrowsOnInvalidPatternConfig(props, "headers.delimiter and headers.separator may not be equal");
        props.put("headers.separator", ",");
        this.assertThrowsOnInvalidPatternConfig(props, "headers.delimiter and headers.key.separator may not be equal");
        props.put("headers.key.separator", ",");
        this.assertThrowsOnInvalidPatternConfig(props, "headers.separator and headers.key.separator may not be equal");
    }

    @Test
    public void testIgnoreErrorInInput() {
        String input = "headerKey0.0:headerValue0.0\tkey0\tvalue0\nheaderKey1.0:headerValue1.0,headerKey1.1:headerValue1.1[MISSING-HEADER-DELIMITER]key1\tvalue1\nheaderKey2.0:headerValue2.0\tkey2[MISSING-KEY-DELIMITER]value2\nheaderKey3.0:headerValue3.0[MISSING-HEADER-DELIMITER]key3[MISSING-KEY-DELIMITER]value3\n";
        Properties props = this.defaultTestProps();
        props.put("ignore.error", "true");
        ProducerRecord<String, String> validRecord = this.record("key0", "value0", Collections.singletonList(new RecordHeader("headerKey0.0", "headerValue0.0".getBytes(StandardCharsets.UTF_8))));
        ProducerRecord<Object, String> missingHeaderDelimiter = this.record(null, "value1", Arrays.asList(new RecordHeader("headerKey1.0", "headerValue1.0".getBytes(StandardCharsets.UTF_8)), new RecordHeader("headerKey1.1", "headerValue1.1[MISSING-HEADER-DELIMITER]key1".getBytes(StandardCharsets.UTF_8))));
        ProducerRecord<Object, String> missingKeyDelimiter = this.record(null, "key2[MISSING-KEY-DELIMITER]value2", Collections.singletonList(new RecordHeader("headerKey2.0", "headerValue2.0".getBytes(StandardCharsets.UTF_8))));
        ProducerRecord<Object, String> missingKeyHeaderDelimiter = this.record(null, "headerKey3.0:headerValue3.0[MISSING-HEADER-DELIMITER]key3[MISSING-KEY-DELIMITER]value3", Collections.emptyList());
        this.runTest(props, input, validRecord, missingHeaderDelimiter, missingKeyDelimiter, missingKeyHeaderDelimiter);
    }

    @Test
    public void testMalformedHeaderIgnoreError() {
        String input = "key-val\tkey0\tvalue0\n";
        Properties props = this.defaultTestProps();
        props.put("ignore.error", "true");
        ProducerRecord<String, String> expected = this.record("key0", "value0", Collections.singletonList(new RecordHeader("key-val", null)));
        this.runTest(props, input, expected);
    }

    @Test
    public void testNullMarker() {
        String input = "key\t\nkey\t<NULL>\nkey\t<NULL>value\n<NULL>\tvalue\n<NULL>\t<NULL>";
        Properties props = this.defaultTestProps();
        props.put("null.marker", "<NULL>");
        props.put("parse.headers", "false");
        this.runTest(props, input, this.record("key", ""), this.record("key", null), this.record("key", "<NULL>value"), this.record(null, "value"), this.record(null, null));
        props.remove("null.marker");
        this.runTest(props, input, this.record("key", ""), this.record("key", "<NULL>"), this.record("key", "<NULL>value"), this.record("<NULL>", "value"), this.record("<NULL>", "<NULL>"));
    }

    @Test
    public void testNullMarkerWithHeaders() {
        String input = "h0:v0,h1:v1\t<NULL>\tvalue\n<NULL>\tkey\t<NULL>\nh0:,h1:v1\t<NULL>\t<NULL>\nh0:<NULL>,h1:v1\tkey\t<NULL>\nh0:<NULL>,h1:<NULL>value\tkey\t<NULL>\n";
        RecordHeader header = new RecordHeader("h1", "v1".getBytes(StandardCharsets.UTF_8));
        Properties props = this.defaultTestProps();
        props.put("null.marker", "<NULL>");
        this.runTest(props, input, this.record(null, "value", Arrays.asList(new RecordHeader("h0", "v0".getBytes(StandardCharsets.UTF_8)), header)), this.record("key", null), this.record(null, null, Arrays.asList(new RecordHeader("h0", "".getBytes(StandardCharsets.UTF_8)), header)), this.record("key", null, Arrays.asList(new RecordHeader("h0", null), header)), this.record("key", null, Arrays.asList(new RecordHeader("h0", null), new RecordHeader("h1", "<NULL>value".getBytes(StandardCharsets.UTF_8)))));
        LineMessageReader lineReader = new LineMessageReader();
        props.remove("null.marker");
        lineReader.configure(Utils.propsToStringMap((Properties)props));
        Iterator iter = lineReader.readRecords((InputStream)new ByteArrayInputStream(input.getBytes()));
        this.assertRecordEquals(this.record("<NULL>", "value", Arrays.asList(new RecordHeader("h0", "v0".getBytes(StandardCharsets.UTF_8)), header)), (ProducerRecord<byte[], byte[]>)((ProducerRecord)iter.next()));
        KafkaException expectedException = (KafkaException)Assertions.assertThrows(KafkaException.class, iter::next);
        Assertions.assertEquals((Object)"No header key separator found in pair '<NULL>' on line number 2", (Object)expectedException.getMessage());
        this.assertRecordEquals(this.record("<NULL>", "<NULL>", Arrays.asList(new RecordHeader("h0", "".getBytes(StandardCharsets.UTF_8)), header)), (ProducerRecord<byte[], byte[]>)((ProducerRecord)iter.next()));
        this.assertRecordEquals(this.record("key", "<NULL>", Arrays.asList(new RecordHeader("h0", "<NULL>".getBytes(StandardCharsets.UTF_8)), header)), (ProducerRecord<byte[], byte[]>)((ProducerRecord)iter.next()));
        this.assertRecordEquals(this.record("key", "<NULL>", Arrays.asList(new RecordHeader("h0", "<NULL>".getBytes(StandardCharsets.UTF_8)), new RecordHeader("h1", "<NULL>value".getBytes(StandardCharsets.UTF_8)))), (ProducerRecord<byte[], byte[]>)((ProducerRecord)iter.next()));
    }

    @Test
    public void testNullMarkerHeaderKeyThrows() {
        String input = "<NULL>:v0,h1:v1\tkey\tvalue\n";
        Properties props = this.defaultTestProps();
        props.put("null.marker", "<NULL>");
        LineMessageReader lineReader = new LineMessageReader();
        lineReader.configure(Utils.propsToStringMap((Properties)props));
        Iterator iter = lineReader.readRecords((InputStream)new ByteArrayInputStream(input.getBytes()));
        KafkaException expectedException = (KafkaException)Assertions.assertThrows(KafkaException.class, iter::next);
        Assertions.assertEquals((Object)"Header keys should not be equal to the null marker '<NULL>' as they can't be null", (Object)expectedException.getMessage());
        props.remove("null.marker");
        this.runTest(props, input, this.record("key", "value", Arrays.asList(new RecordHeader("<NULL>", "v0".getBytes(StandardCharsets.UTF_8)), new RecordHeader("h1", "v1".getBytes(StandardCharsets.UTF_8)))));
    }

    @Test
    public void testInvalidNullMarker() {
        Properties props = this.defaultTestProps();
        props.put("headers.delimiter", "-");
        props.put("headers.separator", ":");
        props.put("headers.key.separator", "/");
        props.put("null.marker", "-");
        this.assertThrowsOnInvalidPatternConfig(props, "null.marker and headers.delimiter may not be equal");
        props.put("null.marker", ":");
        this.assertThrowsOnInvalidPatternConfig(props, "null.marker and headers.separator may not be equal");
        props.put("null.marker", "/");
        this.assertThrowsOnInvalidPatternConfig(props, "null.marker and headers.key.separator may not be equal");
    }

    private Properties defaultTestProps() {
        Properties props = new Properties();
        props.put("topic", "topic");
        props.put("parse.key", "true");
        props.put("parse.headers", "true");
        return props;
    }

    private void assertThrowsOnInvalidPatternConfig(Properties props, String expectedMessage) {
        KafkaException exception = (KafkaException)Assertions.assertThrows(KafkaException.class, () -> new LineMessageReader().configure(Utils.propsToStringMap((Properties)props)));
        Assertions.assertEquals((Object)expectedMessage, (Object)exception.getMessage());
    }

    @SafeVarargs
    private final void runTest(Properties props, String input, ProducerRecord<String, String> ... expectedRecords) {
        LineMessageReader lineReader = new LineMessageReader();
        lineReader.configure(Utils.propsToStringMap((Properties)props));
        Iterator iter = lineReader.readRecords((InputStream)new ByteArrayInputStream(input.getBytes()));
        for (ProducerRecord<String, String> record : expectedRecords) {
            this.assertRecordEquals(record, (ProducerRecord<byte[], byte[]>)((ProducerRecord)iter.next()));
        }
        Assertions.assertFalse((boolean)iter.hasNext());
        Assertions.assertThrows(NoSuchElementException.class, iter::next);
    }

    private <K, V> void assertRecordEquals(ProducerRecord<K, V> expected, ProducerRecord<byte[], byte[]> actual) {
        Assertions.assertEquals((Object)expected.key(), (Object)(actual.key() == null ? null : new String((byte[])actual.key())));
        Assertions.assertEquals((Object)expected.value(), (Object)(actual.value() == null ? null : new String((byte[])actual.value())));
        Assertions.assertArrayEquals((Object[])expected.headers().toArray(), (Object[])actual.headers().toArray());
    }

    private <K, V> ProducerRecord<K, V> record(K key, V value, List<Header> headers) {
        ProducerRecord record = new ProducerRecord("topic", key, value);
        headers.forEach(h -> record.headers().add(h.key(), h.value()));
        return record;
    }

    private <K, V> ProducerRecord<K, V> record(K key, V value) {
        return new ProducerRecord("topic", key, value);
    }
}

