/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.io.File;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.ProcessorStateException;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.StateDirectory;
import org.apache.kafka.test.TestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class StateDirectoryTest {
    private final MockTime time = new MockTime();
    private File stateDir;
    private String applicationId = "applicationId";
    private StateDirectory directory;
    private File appDir;

    @Before
    public void before() {
        this.stateDir = new File(TestUtils.IO_TMP_DIR, TestUtils.randomString((int)5));
        this.directory = new StateDirectory(new StreamsConfig((Map)new Properties(){
            {
                this.put("application.id", StateDirectoryTest.this.applicationId);
                this.put("bootstrap.servers", "dummy:1234");
                this.put("state.dir", StateDirectoryTest.this.stateDir.getPath());
            }
        }), (Time)this.time);
        this.appDir = new File(this.stateDir, this.applicationId);
    }

    @After
    public void cleanup() throws IOException {
        Utils.delete((File)this.stateDir);
    }

    @Test
    public void shouldCreateBaseDirectory() {
        Assert.assertTrue((boolean)this.stateDir.exists());
        Assert.assertTrue((boolean)this.stateDir.isDirectory());
        Assert.assertTrue((boolean)this.appDir.exists());
        Assert.assertTrue((boolean)this.appDir.isDirectory());
    }

    @Test
    public void shouldCreateTaskStateDirectory() {
        TaskId taskId = new TaskId(0, 0);
        File taskDirectory = this.directory.directoryForTask(taskId);
        Assert.assertTrue((boolean)taskDirectory.exists());
        Assert.assertTrue((boolean)taskDirectory.isDirectory());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void shouldLockTaskStateDirectory() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        File taskDirectory = this.directory.directoryForTask(taskId);
        this.directory.lock(taskId);
        try (FileChannel channel = FileChannel.open(new File(taskDirectory, ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            channel.tryLock();
            Assert.fail((String)"shouldn't be able to lock already locked directory");
        }
        catch (OverlappingFileLockException overlappingFileLockException) {
        }
        finally {
            this.directory.unlock(taskId);
        }
    }

    @Test
    public void shouldBeTrueIfAlreadyHoldsLock() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        this.directory.directoryForTask(taskId);
        this.directory.lock(taskId);
        try {
            Assert.assertTrue((boolean)this.directory.lock(taskId));
        }
        finally {
            this.directory.unlock(taskId);
        }
    }

    @Test(expected=ProcessorStateException.class)
    public void shouldThrowProcessorStateException() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        Utils.delete((File)this.stateDir);
        this.directory.directoryForTask(taskId);
    }

    @Test
    public void shouldNotLockDeletedDirectory() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        Utils.delete((File)this.stateDir);
        Assert.assertFalse((boolean)this.directory.lock(taskId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void shouldLockMulitpleTaskDirectories() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        File task1Dir = this.directory.directoryForTask(taskId);
        TaskId taskId2 = new TaskId(1, 0);
        File task2Dir = this.directory.directoryForTask(taskId2);
        try (FileChannel channel1 = FileChannel.open(new File(task1Dir, ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);
             FileChannel channel2 = FileChannel.open(new File(task2Dir, ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            this.directory.lock(taskId);
            this.directory.lock(taskId2);
            channel1.tryLock();
            channel2.tryLock();
            Assert.fail((String)"shouldn't be able to lock already locked directory");
        }
        catch (OverlappingFileLockException overlappingFileLockException) {
        }
        finally {
            this.directory.unlock(taskId);
            this.directory.unlock(taskId2);
        }
    }

    @Test
    public void shouldReleaseTaskStateDirectoryLock() throws IOException {
        TaskId taskId = new TaskId(0, 0);
        File taskDirectory = this.directory.directoryForTask(taskId);
        this.directory.lock(taskId);
        this.directory.unlock(taskId);
        try (FileChannel channel = FileChannel.open(new File(taskDirectory, ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            channel.tryLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void shouldCleanUpTaskStateDirectoriesThatAreNotCurrentlyLocked() throws IOException {
        TaskId task0 = new TaskId(0, 0);
        TaskId task1 = new TaskId(1, 0);
        try {
            this.directory.lock(task0);
            this.directory.lock(task1);
            this.directory.directoryForTask(new TaskId(2, 0));
            List<File> files = Arrays.asList(this.appDir.listFiles());
            Assert.assertEquals((long)3L, (long)files.size());
            this.time.sleep(1000L);
            this.directory.cleanRemovedTasks(0L);
            files = Arrays.asList(this.appDir.listFiles());
            Assert.assertEquals((long)2L, (long)files.size());
            Assert.assertTrue((boolean)files.contains(new File(this.appDir, task0.toString())));
            Assert.assertTrue((boolean)files.contains(new File(this.appDir, task1.toString())));
        }
        finally {
            this.directory.unlock(task0);
            this.directory.unlock(task1);
        }
    }

    @Test
    public void shouldCleanupStateDirectoriesWhenLastModifiedIsLessThanNowMinusCleanupDelay() {
        File dir = this.directory.directoryForTask(new TaskId(2, 0));
        int cleanupDelayMs = 60000;
        this.directory.cleanRemovedTasks(60000L);
        Assert.assertTrue((boolean)dir.exists());
        this.time.sleep(61000L);
        this.directory.cleanRemovedTasks(60000L);
        Assert.assertFalse((boolean)dir.exists());
    }

    @Test
    public void shouldNotRemoveNonTaskDirectoriesAndFiles() {
        File otherDir = TestUtils.tempDirectory((Path)this.stateDir.toPath(), (String)"foo");
        this.directory.cleanRemovedTasks(0L);
        Assert.assertTrue((boolean)otherDir.exists());
    }

    @Test
    public void shouldListAllTaskDirectories() {
        TestUtils.tempDirectory((Path)this.stateDir.toPath(), (String)"foo");
        File taskDir1 = this.directory.directoryForTask(new TaskId(0, 0));
        File taskDir2 = this.directory.directoryForTask(new TaskId(0, 1));
        List<File> dirs = Arrays.asList(this.directory.listTaskDirectories());
        Assert.assertEquals((long)2L, (long)dirs.size());
        Assert.assertTrue((boolean)dirs.contains(taskDir1));
        Assert.assertTrue((boolean)dirs.contains(taskDir2));
    }

    @Test
    public void shouldCreateDirectoriesIfParentDoesntExist() {
        File tempDir = TestUtils.tempDirectory();
        final File stateDir = new File(new File(tempDir, "foo"), "state-dir");
        StateDirectory stateDirectory = new StateDirectory(new StreamsConfig((Map)new Properties(){
            {
                this.put("application.id", StateDirectoryTest.this.applicationId);
                this.put("bootstrap.servers", "dummy:1234");
                this.put("state.dir", stateDir.getPath());
            }
        }), (Time)this.time);
        File taskDir = stateDirectory.directoryForTask(new TaskId(0, 0));
        Assert.assertTrue((boolean)stateDir.exists());
        Assert.assertTrue((boolean)taskDir.exists());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expected=OverlappingFileLockException.class)
    public void shouldLockGlobalStateDirectory() throws IOException {
        this.directory.lockGlobalState();
        try (FileChannel channel = FileChannel.open(new File(this.directory.globalStateDir(), ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            channel.lock();
        }
        finally {
            this.directory.unlockGlobalState();
        }
    }

    @Test
    public void shouldUnlockGlobalStateDirectory() throws IOException {
        this.directory.lockGlobalState();
        this.directory.unlockGlobalState();
        try (FileChannel channel = FileChannel.open(new File(this.directory.globalStateDir(), ".lock").toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            channel.lock();
        }
    }

    @Test
    public void shouldNotLockStateDirLockedByAnotherThread() throws IOException, InterruptedException {
        final TaskId taskId = new TaskId(0, 0);
        final AtomicReference exceptionOnThread = new AtomicReference();
        Thread thread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    StateDirectoryTest.this.directory.lock(taskId);
                }
                catch (IOException e) {
                    exceptionOnThread.set(e);
                }
            }
        });
        thread.start();
        thread.join(30000L);
        Assert.assertNull((String)"should not have had an exception during locking on other thread", exceptionOnThread.get());
        Assert.assertFalse((boolean)this.directory.lock(taskId));
    }

    @Test
    public void shouldNotUnLockStateDirLockedByAnotherThread() throws IOException, InterruptedException {
        final TaskId taskId = new TaskId(0, 0);
        final CountDownLatch lockLatch = new CountDownLatch(1);
        final CountDownLatch unlockLatch = new CountDownLatch(1);
        final AtomicReference exceptionOnThread = new AtomicReference();
        Thread thread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    StateDirectoryTest.this.directory.lock(taskId);
                    lockLatch.countDown();
                    unlockLatch.await();
                    StateDirectoryTest.this.directory.unlock(taskId);
                }
                catch (Exception e) {
                    exceptionOnThread.set(e);
                }
            }
        });
        thread.start();
        lockLatch.await(5L, TimeUnit.SECONDS);
        Assert.assertNull((String)"should not have had an exception on other thread", exceptionOnThread.get());
        this.directory.unlock(taskId);
        Assert.assertFalse((boolean)this.directory.lock(taskId));
        unlockLatch.countDown();
        thread.join(30000L);
        Assert.assertNull((String)"should not have had an exception on other thread", exceptionOnThread.get());
        Assert.assertTrue((boolean)this.directory.lock(taskId));
    }

    @Test
    public void shouldCleanupAllTaskDirectoriesIncludingGlobalOne() {
        this.directory.directoryForTask(new TaskId(1, 0));
        this.directory.globalStateDir();
        List<File> files = Arrays.asList(this.appDir.listFiles());
        Assert.assertEquals((long)2L, (long)files.size());
        this.directory.clean();
        files = Arrays.asList(this.appDir.listFiles());
        Assert.assertEquals((long)0L, (long)files.size());
    }
}

