/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.integration;

import java.io.File;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.IntegerDeserializer;
import org.apache.kafka.common.serialization.IntegerSerializer;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.Topology;
import org.apache.kafka.streams.integration.utils.EmbeddedKafkaCluster;
import org.apache.kafka.streams.integration.utils.IntegrationTestUtils;
import org.apache.kafka.streams.kstream.Consumed;
import org.apache.kafka.streams.kstream.ForeachAction;
import org.apache.kafka.streams.kstream.KStream;
import org.apache.kafka.streams.kstream.Materialized;
import org.apache.kafka.streams.kstream.Reducer;
import org.apache.kafka.streams.processor.Processor;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.processor.ProcessorSupplier;
import org.apache.kafka.streams.processor.StateRestoreListener;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.StateDirectory;
import org.apache.kafka.streams.state.KeyValueBytesStoreSupplier;
import org.apache.kafka.streams.state.KeyValueStore;
import org.apache.kafka.streams.state.StoreBuilder;
import org.apache.kafka.streams.state.Stores;
import org.apache.kafka.streams.state.internals.KeyValueStoreBuilder;
import org.apache.kafka.streams.state.internals.OffsetCheckpoint;
import org.apache.kafka.test.IntegrationTest;
import org.apache.kafka.test.TestUtils;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.core.IsEqual;
import org.junit.After;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={IntegrationTest.class})
public class RestoreIntegrationTest {
    private static final int NUM_BROKERS = 1;
    private static final String APPID = "restore-test";
    @ClassRule
    public static final EmbeddedKafkaCluster CLUSTER = new EmbeddedKafkaCluster(1);
    private static final String INPUT_STREAM = "input-stream";
    private static final String INPUT_STREAM_2 = "input-stream-2";
    private final int numberOfKeys = 10000;
    private KafkaStreams kafkaStreams;

    @BeforeClass
    public static void createTopics() throws InterruptedException {
        CLUSTER.createTopic(INPUT_STREAM, 2, 1);
        CLUSTER.createTopic(INPUT_STREAM_2, 2, 1);
        CLUSTER.createTopic("restore-test-store-changelog", 2, 1);
    }

    private Properties props(String applicationId) {
        Properties streamsConfiguration = new Properties();
        streamsConfiguration.put("application.id", applicationId);
        streamsConfiguration.put("bootstrap.servers", CLUSTER.bootstrapServers());
        streamsConfiguration.put("cache.max.bytes.buffering", (Object)0);
        streamsConfiguration.put("state.dir", TestUtils.tempDirectory((String)applicationId).getPath());
        streamsConfiguration.put("default.key.serde", Serdes.Integer().getClass());
        streamsConfiguration.put("default.value.serde", Serdes.Integer().getClass());
        streamsConfiguration.put("commit.interval.ms", (Object)1000);
        streamsConfiguration.put("auto.offset.reset", "earliest");
        streamsConfiguration.put("internal.leave.group.on.close", (Object)true);
        return streamsConfiguration;
    }

    @After
    public void shutdown() {
        if (this.kafkaStreams != null) {
            this.kafkaStreams.close(Duration.ofSeconds(30L));
        }
    }

    @Test
    public void shouldRestoreStateFromSourceTopic() throws Exception {
        final AtomicInteger numReceived = new AtomicInteger(0);
        StreamsBuilder builder = new StreamsBuilder();
        Properties props = this.props(APPID);
        props.put("topology.optimization", "all");
        int offsetLimitDelta = 1000;
        int offsetCheckpointed = 1000;
        this.createStateForRestoration(INPUT_STREAM);
        this.setCommittedOffset(INPUT_STREAM, 1000);
        StateDirectory stateDirectory = new StateDirectory(new StreamsConfig((Map)props), (Time)new MockTime());
        new OffsetCheckpoint(new File(stateDirectory.directoryForTask(new TaskId(0, 0)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(INPUT_STREAM, 0), 1000L));
        new OffsetCheckpoint(new File(stateDirectory.directoryForTask(new TaskId(0, 1)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(INPUT_STREAM, 1), 1000L));
        final CountDownLatch startupLatch = new CountDownLatch(1);
        final CountDownLatch shutdownLatch = new CountDownLatch(1);
        builder.table(INPUT_STREAM, Consumed.with((Serde)Serdes.Integer(), (Serde)Serdes.Integer())).toStream().foreach((ForeachAction)new ForeachAction<Integer, Integer>(){

            public void apply(Integer key, Integer value) {
                if (numReceived.incrementAndGet() == 2000) {
                    shutdownLatch.countDown();
                }
            }
        });
        this.kafkaStreams = new KafkaStreams(builder.build(), props);
        this.kafkaStreams.setStateListener(new KafkaStreams.StateListener(){

            public void onChange(KafkaStreams.State newState, KafkaStreams.State oldState) {
                if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                    startupLatch.countDown();
                }
            }
        });
        final AtomicLong restored = new AtomicLong(0L);
        this.kafkaStreams.setGlobalStateRestoreListener(new StateRestoreListener(){

            public void onRestoreStart(TopicPartition topicPartition, String storeName, long startingOffset, long endingOffset) {
            }

            public void onBatchRestored(TopicPartition topicPartition, String storeName, long batchEndOffset, long numRestored) {
            }

            public void onRestoreEnd(TopicPartition topicPartition, String storeName, long totalRestored) {
                restored.addAndGet(totalRestored);
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)restored.get(), (Matcher)IsEqual.equalTo((Object)6000L));
        Assert.assertTrue((boolean)shutdownLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)numReceived.get(), (Matcher)IsEqual.equalTo((Object)2000));
    }

    @Test
    public void shouldRestoreStateFromChangelogTopic() throws Exception {
        final AtomicInteger numReceived = new AtomicInteger(0);
        StreamsBuilder builder = new StreamsBuilder();
        Properties props = this.props(APPID);
        int offsetCheckpointed = 1000;
        this.createStateForRestoration("restore-test-store-changelog");
        this.createStateForRestoration(INPUT_STREAM);
        StateDirectory stateDirectory = new StateDirectory(new StreamsConfig((Map)props), (Time)new MockTime());
        new OffsetCheckpoint(new File(stateDirectory.directoryForTask(new TaskId(0, 0)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition("restore-test-store-changelog", 0), 1000L));
        new OffsetCheckpoint(new File(stateDirectory.directoryForTask(new TaskId(0, 1)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition("restore-test-store-changelog", 1), 1000L));
        final CountDownLatch startupLatch = new CountDownLatch(1);
        final CountDownLatch shutdownLatch = new CountDownLatch(1);
        builder.table(INPUT_STREAM, Consumed.with((Serde)Serdes.Integer(), (Serde)Serdes.Integer()), Materialized.as((String)"store")).toStream().foreach((ForeachAction)new ForeachAction<Integer, Integer>(){

            public void apply(Integer key, Integer value) {
                if (numReceived.incrementAndGet() == 10000) {
                    shutdownLatch.countDown();
                }
            }
        });
        this.kafkaStreams = new KafkaStreams(builder.build(), props);
        this.kafkaStreams.setStateListener(new KafkaStreams.StateListener(){

            public void onChange(KafkaStreams.State newState, KafkaStreams.State oldState) {
                if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                    startupLatch.countDown();
                }
            }
        });
        final AtomicLong restored = new AtomicLong(0L);
        this.kafkaStreams.setGlobalStateRestoreListener(new StateRestoreListener(){

            public void onRestoreStart(TopicPartition topicPartition, String storeName, long startingOffset, long endingOffset) {
            }

            public void onBatchRestored(TopicPartition topicPartition, String storeName, long batchEndOffset, long numRestored) {
            }

            public void onRestoreEnd(TopicPartition topicPartition, String storeName, long totalRestored) {
                restored.addAndGet(totalRestored);
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)restored.get(), (Matcher)IsEqual.equalTo((Object)8000L));
        Assert.assertTrue((boolean)shutdownLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)numReceived.get(), (Matcher)IsEqual.equalTo((Object)10000));
    }

    @Test
    public void shouldSuccessfullyStartWhenLoggingDisabled() throws InterruptedException {
        StreamsBuilder builder = new StreamsBuilder();
        KStream stream = builder.stream(INPUT_STREAM);
        stream.groupByKey().reduce((Reducer)new Reducer<Integer>(){

            public Integer apply(Integer value1, Integer value2) {
                return value1 + value2;
            }
        }, Materialized.as((String)"reduce-store").withLoggingDisabled());
        final CountDownLatch startupLatch = new CountDownLatch(1);
        this.kafkaStreams = new KafkaStreams(builder.build(), this.props(APPID));
        this.kafkaStreams.setStateListener(new KafkaStreams.StateListener(){

            public void onChange(KafkaStreams.State newState, KafkaStreams.State oldState) {
                if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                    startupLatch.countDown();
                }
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
    }

    @Test
    public void shouldProcessDataFromStoresWithLoggingDisabled() throws InterruptedException, ExecutionException {
        IntegrationTestUtils.produceKeyValuesSynchronously(INPUT_STREAM_2, Arrays.asList(KeyValue.pair((Object)1, (Object)1), KeyValue.pair((Object)2, (Object)2), KeyValue.pair((Object)3, (Object)3)), TestUtils.producerConfig((String)CLUSTER.bootstrapServers(), IntegerSerializer.class, IntegerSerializer.class), (Time)RestoreIntegrationTest.CLUSTER.time);
        KeyValueBytesStoreSupplier lruMapSupplier = Stores.lruMap((String)INPUT_STREAM_2, (int)10);
        StoreBuilder storeBuilder = new KeyValueStoreBuilder(lruMapSupplier, Serdes.Integer(), Serdes.Integer(), (Time)RestoreIntegrationTest.CLUSTER.time).withLoggingDisabled();
        StreamsBuilder streamsBuilder = new StreamsBuilder();
        streamsBuilder.addStateStore(storeBuilder);
        KStream stream = streamsBuilder.stream(INPUT_STREAM_2);
        final CountDownLatch processorLatch = new CountDownLatch(3);
        stream.process((ProcessorSupplier)new ProcessorSupplier<Integer, Integer>(){

            public Processor<Integer, Integer> get() {
                return new KeyValueStoreProcessor(RestoreIntegrationTest.INPUT_STREAM_2, processorLatch);
            }
        }, new String[]{INPUT_STREAM_2});
        Topology topology = streamsBuilder.build();
        this.kafkaStreams = new KafkaStreams(topology, this.props("restore-test-logging-disabled"));
        final CountDownLatch latch = new CountDownLatch(1);
        this.kafkaStreams.setStateListener(new KafkaStreams.StateListener(){

            public void onChange(KafkaStreams.State newState, KafkaStreams.State oldState) {
                if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                    latch.countDown();
                }
            }
        });
        this.kafkaStreams.start();
        latch.await(30L, TimeUnit.SECONDS);
        Assert.assertTrue((boolean)processorLatch.await(30L, TimeUnit.SECONDS));
    }

    private void createStateForRestoration(String changelogTopic) {
        Properties producerConfig = new Properties();
        producerConfig.put("bootstrap.servers", CLUSTER.bootstrapServers());
        try (KafkaProducer producer = new KafkaProducer(producerConfig, (Serializer)new IntegerSerializer(), (Serializer)new IntegerSerializer());){
            for (int i = 0; i < 10000; ++i) {
                producer.send(new ProducerRecord(changelogTopic, (Object)i, (Object)i));
            }
        }
    }

    private void setCommittedOffset(String topic, int limitDelta) {
        Properties consumerConfig = new Properties();
        consumerConfig.put("bootstrap.servers", CLUSTER.bootstrapServers());
        consumerConfig.put("group.id", APPID);
        consumerConfig.put("client.id", "commit-consumer");
        consumerConfig.put("key.deserializer", IntegerDeserializer.class);
        consumerConfig.put("value.deserializer", IntegerDeserializer.class);
        KafkaConsumer consumer = new KafkaConsumer(consumerConfig);
        List<TopicPartition> partitions = Arrays.asList(new TopicPartition(topic, 0), new TopicPartition(topic, 1));
        consumer.assign(partitions);
        consumer.seekToEnd(partitions);
        for (TopicPartition partition : partitions) {
            long position = consumer.position(partition);
            consumer.seek(partition, position - (long)limitDelta);
        }
        consumer.commitSync();
        consumer.close();
    }

    public static class KeyValueStoreProcessor
    implements Processor<Integer, Integer> {
        private String topic;
        private final CountDownLatch processorLatch;
        private KeyValueStore<Integer, Integer> store;

        public KeyValueStoreProcessor(String topic, CountDownLatch processorLatch) {
            this.topic = topic;
            this.processorLatch = processorLatch;
        }

        public void init(ProcessorContext context) {
            this.store = (KeyValueStore)context.getStateStore(this.topic);
        }

        public void process(Integer key, Integer value) {
            if (key != null) {
                this.store.put((Object)key, (Object)value);
                this.processorLatch.countDown();
            }
        }

        public void close() {
        }
    }
}

