/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tiered.storage.utils;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Timer;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.storage.internals.log.LogFileUtils;

public final class BrokerLocalStorage {
    private final Integer brokerId;
    private final Set<File> brokerStorageDirectories;
    private final Integer storageWaitTimeoutSec;
    private final int storagePollPeriodSec = 1;
    private final Time time = Time.SYSTEM;

    public BrokerLocalStorage(Integer brokerId, Set<String> storageDirnames, Integer storageWaitTimeoutSec) {
        this.brokerId = brokerId;
        this.brokerStorageDirectories = storageDirnames.stream().map(File::new).collect(Collectors.toSet());
        this.storageWaitTimeoutSec = storageWaitTimeoutSec;
    }

    public Integer getBrokerId() {
        return this.brokerId;
    }

    public Set<File> getBrokerStorageDirectories() {
        return this.brokerStorageDirectories;
    }

    public void waitForEarliestLocalOffset(TopicPartition topicPartition, Long offset) {
        Function<OffsetHolder, Optional<String>> relativePosFunc = offsetHolder -> {
            Optional<Object> result = Optional.empty();
            if (((OffsetHolder)offsetHolder).firstLogFileBaseOffset < offset && !this.isOffsetPresentInFirstLocalSegment(topicPartition, ((OffsetHolder)offsetHolder).firstLogFileBaseOffset, offset)) {
                result = Optional.of("smaller than");
            } else if (((OffsetHolder)offsetHolder).firstLogFileBaseOffset > offset) {
                result = Optional.of("ahead of");
            }
            return result;
        };
        this.waitForOffset(topicPartition, offset, relativePosFunc);
    }

    public void waitForAtLeastEarliestLocalOffset(TopicPartition topicPartition, Long offset) {
        Function<OffsetHolder, Optional<String>> relativePosFunc = offsetHolder -> {
            Optional<Object> result = Optional.empty();
            if (((OffsetHolder)offsetHolder).firstLogFileBaseOffset < offset && !this.isOffsetPresentInFirstLocalSegment(topicPartition, ((OffsetHolder)offsetHolder).firstLogFileBaseOffset, offset)) {
                result = Optional.of("smaller than");
            }
            return result;
        };
        this.waitForOffset(topicPartition, offset, relativePosFunc);
    }

    private void waitForOffset(TopicPartition topicPartition, Long offset, Function<OffsetHolder, Optional<String>> relativePosFunc) {
        Timer timer = this.time.timer(TimeUnit.SECONDS.toMillis(this.storageWaitTimeoutSec.intValue()));
        OffsetHolder offsetHolder = new OffsetHolder(0L, Collections.emptyList());
        while (timer.notExpired() && offsetHolder.firstLogFileBaseOffset < offset) {
            timer.sleep(TimeUnit.SECONDS.toMillis(1L));
            offsetHolder = this.getEarliestLocalOffset(topicPartition);
        }
        Optional<String> relativePos = relativePosFunc.apply(offsetHolder);
        if (relativePos.isPresent()) {
            String pos = relativePos.get();
            String message = String.format("[BrokerId=%d] The base offset of the first log segment of %s in the log directory is %d which is %s the expected offset %s. The directory of %s is made of the following files: %s", this.brokerId, topicPartition, offsetHolder.firstLogFileBaseOffset, pos, offset, topicPartition, String.join((CharSequence)System.lineSeparator(), offsetHolder.partitionFiles));
            throw new AssertionError((Object)message);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isOffsetPresentInFirstLocalSegment(TopicPartition topicPartition, Long firstLogFileBaseOffset, Long offsetToSearch) {
        if (offsetToSearch < firstLogFileBaseOffset) {
            return false;
        }
        if (offsetToSearch.equals(firstLogFileBaseOffset)) {
            return true;
        }
        File logDir = this.brokerStorageDirectories.stream().filter(dir -> this.dirContainsTopicPartition(topicPartition, (File)dir)).findFirst().orElseThrow(() -> new IllegalArgumentException(String.format("[BrokerId=%d] Directory for the topic-partition %s was not found", this.brokerId, topicPartition)));
        File partitionDir = new File(logDir.getAbsolutePath(), topicPartition.toString());
        File firstSegmentFile = new File(partitionDir.getAbsolutePath(), LogFileUtils.filenamePrefixFromOffset((long)firstLogFileBaseOffset) + ".log");
        try (FileRecords fileRecords = FileRecords.open((File)firstSegmentFile, (boolean)false);){
            FileLogInputStream.FileChannelRecordBatch batch;
            Iterator iterator = fileRecords.batches().iterator();
            do {
                if (!iterator.hasNext()) return false;
            } while ((batch = (FileLogInputStream.FileChannelRecordBatch)iterator.next()).baseOffset() > offsetToSearch || batch.lastOffset() < offsetToSearch);
            boolean bl = true;
            return bl;
        }
        catch (IOException ex) {
            return false;
        }
    }

    public void eraseStorage(FilenameFilter filter) throws IOException {
        for (File brokerDir : this.brokerStorageDirectories) {
            for (File file : Objects.requireNonNull(brokerDir.listFiles(filter))) {
                Utils.delete((File)file);
            }
        }
    }

    private OffsetHolder getEarliestLocalOffset(TopicPartition topicPartition) {
        List<String> partitionFiles = this.getTopicPartitionFileNames(topicPartition);
        Optional<String> firstLogFile = partitionFiles.stream().filter(filename -> filename.endsWith(".log")).sorted().findFirst();
        if (!firstLogFile.isPresent()) {
            throw new IllegalArgumentException(String.format("[BrokerId=%d] No log file found for the topic-partition %s", this.brokerId, topicPartition));
        }
        return new OffsetHolder(LogFileUtils.offsetFromFileName((String)firstLogFile.get()), partitionFiles);
    }

    public boolean dirContainsTopicPartition(TopicPartition topicPartition, File logDir) {
        File[] files = this.getTopicPartitionFiles(topicPartition, Collections.singleton(logDir));
        return files != null && files.length > 0;
    }

    private File[] getTopicPartitionFiles(TopicPartition topicPartition) {
        return this.getTopicPartitionFiles(topicPartition, this.brokerStorageDirectories);
    }

    private File[] getTopicPartitionFiles(TopicPartition topicPartition, Set<File> logDirs) {
        File brokerDir;
        File[] files = null;
        Iterator<File> iterator = logDirs.iterator();
        while (iterator.hasNext() && ((files = (brokerDir = iterator.next()).listFiles((dir, name) -> name.equals(topicPartition.toString()))) == null || files.length == 0)) {
        }
        return files;
    }

    private List<String> getTopicPartitionFileNames(TopicPartition topicPartition) {
        File[] files = this.getTopicPartitionFiles(topicPartition);
        if (files == null || files.length == 0) {
            throw new IllegalArgumentException(String.format("[BrokerId=%d] Directory for the topic-partition %s was not found", this.brokerId, topicPartition));
        }
        File topicPartitionDir = files[0];
        return Arrays.stream((Object[])Objects.requireNonNull(topicPartitionDir.listFiles())).map(File::getName).collect(Collectors.toList());
    }

    private static final class OffsetHolder {
        private final long firstLogFileBaseOffset;
        private final List<String> partitionFiles;

        public OffsetHolder(long firstLogFileBaseOffset, List<String> partitionFiles) {
            this.firstLogFileBaseOffset = firstLogFileBaseOffset;
            this.partitionFiles = partitionFiles;
        }
    }
}

