/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.log.remote.storage;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.server.log.remote.storage.LogSegmentData;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentId;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentMetadata;
import org.apache.kafka.server.log.remote.storage.RemoteTopicPartitionDirectory;
import org.apache.kafka.server.log.remote.storage.Transferer;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RemoteLogSegmentFileset {
    private static final Pattern FILENAME_FORMAT = Pattern.compile("(\\d+-)([a-zA-Z0-9_-]{22})(\\.[a-z_]+)");
    private static final int GROUP_UUID = 2;
    private static final int GROUP_FILE_TYPE = 3;
    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteLogSegmentFileset.class);
    private final RemoteTopicPartitionDirectory partitionDirectory;
    private final RemoteLogSegmentId remoteLogSegmentId;
    private final Map<RemoteLogSegmentFileType, File> files;

    public static RemoteLogSegmentFileset openFileset(File storageDir, RemoteLogSegmentMetadata metadata) {
        RemoteTopicPartitionDirectory tpDir = RemoteTopicPartitionDirectory.openTopicPartitionDirectory(metadata.remoteLogSegmentId().topicIdPartition(), storageDir);
        File partitionDirectory = tpDir.getDirectory();
        Uuid uuid = metadata.remoteLogSegmentId().id();
        String startOffset = LogFileUtils.filenamePrefixFromOffset((long)metadata.startOffset());
        Map<RemoteLogSegmentFileType, File> files = Arrays.stream(RemoteLogSegmentFileType.values()).collect(Collectors.toMap(Function.identity(), type -> new File(partitionDirectory, type.toFilename(startOffset, uuid))));
        return new RemoteLogSegmentFileset(tpDir, metadata.remoteLogSegmentId(), files);
    }

    public static RemoteLogSegmentFileset openExistingFileset(RemoteTopicPartitionDirectory tpDirectory, Uuid uuid) {
        try {
            Map<RemoteLogSegmentFileType, File> files = Files.list(tpDirectory.getDirectory().toPath()).filter(path -> path.getFileName().toString().contains(uuid.toString())).collect(Collectors.toMap(path -> RemoteLogSegmentFileType.getFileType(path.getFileName().toString()), Path::toFile));
            Set expectedFileTypes = Arrays.stream(RemoteLogSegmentFileType.values()).filter(x -> !x.isOptional()).collect(Collectors.toSet());
            if (!files.keySet().containsAll(expectedFileTypes)) {
                expectedFileTypes.removeAll(files.keySet());
                throw new IllegalStateException(String.format("Invalid fileset, missing files: %s", expectedFileTypes));
            }
            RemoteLogSegmentId id = new RemoteLogSegmentId(tpDirectory.getTopicIdPartition(), uuid);
            return new RemoteLogSegmentFileset(tpDirectory, id, files);
        }
        catch (IOException ex) {
            throw new RuntimeException(String.format("Unable to list the files in the directory '%s'", tpDirectory.getDirectory()), ex);
        }
    }

    public RemoteLogSegmentId getRemoteLogSegmentId() {
        return this.remoteLogSegmentId;
    }

    public File getFile(RemoteLogSegmentFileType type) {
        return this.files.get((Object)type);
    }

    public boolean delete() {
        return RemoteLogSegmentFileset.deleteFilesOnly(this.files.values());
    }

    public List<Record> getRecords() throws IOException {
        return StreamSupport.stream(FileRecords.open((File)this.files.get((Object)RemoteLogSegmentFileType.SEGMENT)).records().spliterator(), false).collect(Collectors.toList());
    }

    public void copy(Transferer transferer, LogSegmentData data) throws IOException {
        transferer.transfer(data.logSegment().toFile(), this.files.get((Object)RemoteLogSegmentFileType.SEGMENT));
        transferer.transfer(data.offsetIndex().toFile(), this.files.get((Object)RemoteLogSegmentFileType.OFFSET_INDEX));
        transferer.transfer(data.timeIndex().toFile(), this.files.get((Object)RemoteLogSegmentFileType.TIME_INDEX));
        if (data.transactionIndex().isPresent()) {
            transferer.transfer(((Path)data.transactionIndex().get()).toFile(), this.files.get((Object)RemoteLogSegmentFileType.TRANSACTION_INDEX));
        }
        transferer.transfer(data.leaderEpochIndex(), this.files.get((Object)RemoteLogSegmentFileType.LEADER_EPOCH_CHECKPOINT));
        transferer.transfer(data.producerSnapshotIndex().toFile(), this.files.get((Object)RemoteLogSegmentFileType.PRODUCER_SNAPSHOT));
    }

    public String toString() {
        String ls = this.files.values().stream().map(file -> "\t" + file.getName() + "\n").reduce("", (s1, s2) -> s1 + s2);
        return String.format("%s/\n%s", this.partitionDirectory.getDirectory().getName(), ls);
    }

    public static boolean deleteFilesOnly(Collection<File> files) {
        Optional<File> notAFile = files.stream().filter(f -> f.exists() && !f.isFile()).findAny();
        if (notAFile.isPresent()) {
            LOGGER.warn(String.format("Found unexpected directory %s. Will not delete.", notAFile.get().getAbsolutePath()));
            return false;
        }
        return files.stream().map(RemoteLogSegmentFileset::deleteQuietly).reduce(true, Boolean::logicalAnd);
    }

    public static boolean deleteQuietly(File file) {
        try {
            LOGGER.trace("Deleting " + file.getAbsolutePath());
            if (!file.exists()) {
                return true;
            }
            return file.delete();
        }
        catch (Exception e) {
            LOGGER.error(String.format("Encountered error while deleting %s", file.getAbsolutePath()));
            return false;
        }
    }

    RemoteLogSegmentFileset(RemoteTopicPartitionDirectory topicPartitionDirectory, RemoteLogSegmentId remoteLogSegmentId, Map<RemoteLogSegmentFileType, File> files) {
        this.partitionDirectory = Objects.requireNonNull(topicPartitionDirectory);
        this.remoteLogSegmentId = Objects.requireNonNull(remoteLogSegmentId);
        this.files = Collections.unmodifiableMap(files);
    }

    public static enum RemoteLogSegmentFileType {
        SEGMENT(false, ".log"),
        OFFSET_INDEX(false, ".index"),
        TIME_INDEX(false, ".timeindex"),
        TRANSACTION_INDEX(true, ".txnindex"),
        LEADER_EPOCH_CHECKPOINT(false, ".leader_epoch_checkpoint"),
        PRODUCER_SNAPSHOT(false, ".snapshot");

        private final boolean optional;
        private final String suffix;

        private RemoteLogSegmentFileType(boolean optional, String suffix) {
            this.optional = optional;
            this.suffix = suffix;
        }

        public String toFilename(String startOffset, Uuid uuid) {
            return startOffset + "-" + uuid.toString() + this.suffix;
        }

        public static RemoteLogSegmentFileType getFileType(String filename) {
            String fileSuffix = RemoteLogSegmentFileType.substr(filename, 3);
            for (RemoteLogSegmentFileType fileType : RemoteLogSegmentFileType.values()) {
                if (!fileType.getSuffix().equals(fileSuffix)) continue;
                return fileType;
            }
            throw new IllegalArgumentException(String.format("Not a remote log segment file: %s", filename));
        }

        public static Uuid getUuid(String filename) {
            return Uuid.fromString((String)RemoteLogSegmentFileType.substr(filename, 2));
        }

        static String substr(String filename, int group) {
            Matcher m = FILENAME_FORMAT.matcher(filename);
            if (!m.matches()) {
                throw new IllegalArgumentException(String.format("Not a remote log segment file: %s", filename));
            }
            return m.group(group);
        }

        public boolean isOptional() {
            return this.optional;
        }

        public String getSuffix() {
            return this.suffix;
        }
    }
}

