/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.log.remote.storage;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.GatheringByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.compress.Compression;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.server.log.remote.storage.LocalTieredStorage;
import org.apache.kafka.server.log.remote.storage.LocalTieredStorageSnapshot;
import org.apache.kafka.server.log.remote.storage.LocalTieredStorageTraverser;
import org.apache.kafka.server.log.remote.storage.LogSegmentData;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentFileset;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentId;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentMetadata;
import org.apache.kafka.server.log.remote.storage.RemoteResourceNotFoundException;
import org.apache.kafka.server.log.remote.storage.RemoteStorageException;
import org.apache.kafka.server.log.remote.storage.RemoteStorageManager;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LocalTieredStorageTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(LocalTieredStorageTest.class);
    private final LocalLogSegments localLogSegments = new LocalLogSegments();
    private final TopicPartition topicPartition = new TopicPartition("my-topic", 1);
    private final TopicIdPartition topicIdPartition = new TopicIdPartition(Uuid.randomUuid(), this.topicPartition);
    private LocalTieredStorage tieredStorage;
    private Verifier remoteStorageVerifier;
    private String storageDir;
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd-HH:mm:ss");

    private void init(Map<String, Object> extraConfig, String testName) {
        this.tieredStorage = new LocalTieredStorage();
        this.remoteStorageVerifier = new Verifier(this.tieredStorage, this.topicIdPartition);
        this.storageDir = this.generateStorageId(testName);
        HashMap<String, Object> config = new HashMap<String, Object>();
        config.put("dir", this.storageDir);
        config.put("delete.on.close", "true");
        config.put("broker.id", 1);
        config.putAll(extraConfig);
        this.tieredStorage.configure(config);
    }

    @BeforeEach
    public void before(TestInfo testInfo) {
        this.init(Collections.emptyMap(), testInfo.getDisplayName());
    }

    @AfterEach
    public void after() throws IOException {
        this.tieredStorage.clear();
        this.localLogSegments.deleteAll();
        Files.deleteIfExists(Paths.get(this.storageDir, new String[0]));
    }

    @Test
    public void copyEmptyLogSegment() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
        this.tieredStorage.copyLogSegmentData(metadata, segment);
        this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata);
    }

    @Test
    public void copyDataFromLogSegment() throws RemoteStorageException {
        byte[] data = new byte[]{0, 1, 2};
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
        LogSegmentData segment = this.localLogSegments.nextSegment(new byte[][]{data});
        this.tieredStorage.copyLogSegmentData(metadata, segment);
        this.remoteStorageVerifier.verifyRemoteLogSegmentMatchesLocal(metadata, segment);
    }

    @Test
    public void fetchLogSegment() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment(new byte[][]{{0, 1, 2}});
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyFetchedLogSegment(id, 0, new byte[]{0, 1, 2});
    }

    @Test
    public void fetchOffsetIndex() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyFetchedOffsetIndex(id, LocalLogSegments.OFFSET_FILE_BYTES);
    }

    @Test
    public void fetchTimeIndex() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyFetchedTimeIndex(id, LocalLogSegments.TIME_FILE_BYTES);
    }

    @Test
    public void fetchTransactionIndex() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyFetchedTransactionIndex(id, LocalLogSegments.TXN_FILE_BYTES);
    }

    @Test
    public void fetchLeaderEpochCheckpoint() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyLeaderEpochCheckpoint(id, LocalLogSegments.LEADER_EPOCH_CHECKPOINT_FILE_BYTES);
    }

    @Test
    public void fetchProducerSnapshot() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyProducerSnapshot(id, LocalLogSegments.PRODUCER_SNAPSHOT_FILE_BYTES);
    }

    @Test
    public void deleteLogSegment() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
        LogSegmentData segment = this.localLogSegments.nextSegment();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata);
        this.tieredStorage.deleteLogSegmentData(this.newRemoteLogSegmentMetadata(id));
        this.remoteStorageVerifier.verifyLogSegmentFilesAbsent(metadata);
    }

    @Test
    public void deletePartition() throws RemoteStorageException {
        int segmentCount = 10;
        ArrayList<RemoteLogSegmentMetadata> segmentMetadatas = new ArrayList<RemoteLogSegmentMetadata>();
        for (int i = 0; i < segmentCount; ++i) {
            RemoteLogSegmentId id = this.newRemoteLogSegmentId();
            RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
            LogSegmentData segment = this.localLogSegments.nextSegment();
            this.tieredStorage.copyLogSegmentData(metadata, segment);
            this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata);
            segmentMetadatas.add(metadata);
        }
        this.tieredStorage.deletePartition(this.topicIdPartition);
        this.remoteStorageVerifier.assertFileDoesNotExist(this.remoteStorageVerifier.expectedPartitionPath());
        for (RemoteLogSegmentMetadata segmentMetadata : segmentMetadatas) {
            this.remoteStorageVerifier.verifyLogSegmentFilesAbsent(segmentMetadata);
        }
    }

    @Test
    public void deleteLogSegmentWithoutOptionalFiles() throws RemoteStorageException {
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
        LogSegmentData segment = this.localLogSegments.nextSegment();
        ((Path)segment.transactionIndex().get()).toFile().delete();
        this.tieredStorage.copyLogSegmentData(metadata, segment);
        this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata, path -> {
            String fileName = path.getFileName().toString();
            if (!fileName.contains(".txnindex")) {
                this.remoteStorageVerifier.assertFileExists(path);
            }
        });
        this.tieredStorage.deleteLogSegmentData(this.newRemoteLogSegmentMetadata(id));
        this.remoteStorageVerifier.verifyLogSegmentFilesAbsent(metadata);
    }

    @Test
    public void segmentsAreNotDeletedIfDeleteApiIsDisabled(TestInfo testInfo) throws RemoteStorageException {
        this.init(Collections.singletonMap("delete.enable", "false"), testInfo.getDisplayName());
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment();
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(id);
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata);
        this.tieredStorage.deleteLogSegmentData(this.newRemoteLogSegmentMetadata(id));
        this.remoteStorageVerifier.verifyContainsLogSegmentFiles(metadata);
    }

    @Test
    public void traverseSingleOffloadedRecord() throws RemoteStorageException {
        final byte[] bytes = new byte[]{0, 1, 2};
        final RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        LogSegmentData segment = this.localLogSegments.nextSegment(new byte[][]{bytes});
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), segment);
        this.tieredStorage.traverse(new LocalTieredStorageTraverser(){

            @Override
            public void visitTopicIdPartition(TopicIdPartition topicIdPartition) {
                Assertions.assertEquals((Object)LocalTieredStorageTest.this.topicPartition, (Object)topicIdPartition.topicPartition());
            }

            @Override
            public void visitSegment(RemoteLogSegmentFileset fileset) {
                Assertions.assertEquals((Object)id, (Object)fileset.getRemoteLogSegmentId());
                try {
                    FileRecords records = FileRecords.open((File)fileset.getFile(RemoteLogSegmentFileset.RemoteLogSegmentFileType.SEGMENT));
                    Iterator it = records.records().iterator();
                    Assertions.assertEquals((Object)ByteBuffer.wrap(bytes), (Object)((Record)it.next()).value());
                }
                catch (IOException e) {
                    throw new AssertionError((Object)e);
                }
            }
        });
    }

    @Test
    public void traverseMultipleOffloadedRecordsInOneSegment() throws RemoteStorageException, IOException {
        byte[] record1 = new byte[]{0, 1, 2};
        byte[] record2 = new byte[]{3, 4, 5};
        RemoteLogSegmentId id = this.newRemoteLogSegmentId();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(id), this.localLogSegments.nextSegment(record1, record2));
        LocalTieredStorageSnapshot snapshot = LocalTieredStorageSnapshot.takeSnapshot(this.tieredStorage);
        Assertions.assertEquals(Arrays.asList(this.topicPartition), snapshot.getTopicPartitions());
        Assertions.assertEquals(Arrays.asList(ByteBuffer.wrap(record1), ByteBuffer.wrap(record2)), LocalTieredStorageTest.extractRecordsValue(snapshot, id));
    }

    @Test
    public void traverseMultipleOffloadedRecordsInTwoSegments() throws RemoteStorageException, IOException {
        byte[] record1a = new byte[]{0, 1, 2};
        byte[] record2a = new byte[]{3, 4, 5};
        byte[] record1b = new byte[]{6, 7, 8};
        byte[] record2b = new byte[]{9, 10, 11};
        RemoteLogSegmentId idA = this.newRemoteLogSegmentId();
        RemoteLogSegmentId idB = this.newRemoteLogSegmentId();
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(idA), this.localLogSegments.nextSegment(record1a, record2a));
        this.tieredStorage.copyLogSegmentData(this.newRemoteLogSegmentMetadata(idB), this.localLogSegments.nextSegment(record1b, record2b));
        LocalTieredStorageSnapshot snapshot = LocalTieredStorageSnapshot.takeSnapshot(this.tieredStorage);
        HashMap<RemoteLogSegmentId, List<ByteBuffer>> expected = new HashMap<RemoteLogSegmentId, List<ByteBuffer>>();
        expected.put(idA, Arrays.asList(ByteBuffer.wrap(record1a), ByteBuffer.wrap(record2a)));
        expected.put(idB, Arrays.asList(ByteBuffer.wrap(record1b), ByteBuffer.wrap(record2b)));
        HashMap<RemoteLogSegmentId, List<ByteBuffer>> actual = new HashMap<RemoteLogSegmentId, List<ByteBuffer>>();
        actual.put(idA, LocalTieredStorageTest.extractRecordsValue(snapshot, idA));
        actual.put(idB, LocalTieredStorageTest.extractRecordsValue(snapshot, idB));
        Assertions.assertEquals(Arrays.asList(this.topicPartition), snapshot.getTopicPartitions());
        Assertions.assertEquals(expected, actual);
    }

    @Test
    public void fetchThrowsIfDataDoesNotExist() {
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(this.newRemoteLogSegmentId());
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchLogSegment(metadata, 0, metadata.segmentSizeInBytes()));
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchIndex(metadata, RemoteStorageManager.IndexType.OFFSET));
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchIndex(metadata, RemoteStorageManager.IndexType.TIMESTAMP));
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchIndex(metadata, RemoteStorageManager.IndexType.LEADER_EPOCH));
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchIndex(metadata, RemoteStorageManager.IndexType.PRODUCER_SNAPSHOT));
        Assertions.assertThrows(RemoteResourceNotFoundException.class, () -> this.tieredStorage.fetchIndex(metadata, RemoteStorageManager.IndexType.TRANSACTION));
    }

    @Test
    public void assertStartAndEndPositionConsistency() {
        RemoteLogSegmentMetadata metadata = this.newRemoteLogSegmentMetadata(this.newRemoteLogSegmentId());
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.tieredStorage.fetchLogSegment(metadata, -1, Integer.MAX_VALUE));
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.tieredStorage.fetchLogSegment(metadata, 1, -1));
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.tieredStorage.fetchLogSegment(metadata, 2, 1));
    }

    private RemoteLogSegmentMetadata newRemoteLogSegmentMetadata(RemoteLogSegmentId id) {
        return new RemoteLogSegmentMetadata(id, 0L, 0L, -1L, -1, 1000L, 1024, Collections.singletonMap(0, 0L));
    }

    private RemoteLogSegmentId newRemoteLogSegmentId() {
        return new RemoteLogSegmentId(this.topicIdPartition, Uuid.randomUuid());
    }

    private static List<ByteBuffer> extractRecordsValue(LocalTieredStorageSnapshot snapshot, RemoteLogSegmentId id) throws IOException {
        FileRecords records = FileRecords.open((File)snapshot.getFile(id, RemoteLogSegmentFileset.RemoteLogSegmentFileType.SEGMENT));
        ArrayList<ByteBuffer> buffers = new ArrayList<ByteBuffer>();
        for (Record record : records.records()) {
            buffers.add(record.value());
        }
        return buffers;
    }

    private String generateStorageId(String testName) {
        return String.format("kafka-tiered-storage/%s-%s-%s", this.getClass().getSimpleName(), testName, DATE_TIME_FORMATTER.format(LocalDateTime.now()));
    }

    private static final class LocalLogSegments {
        private static final byte[] OFFSET_FILE_BYTES = "offset".getBytes();
        private static final byte[] TIME_FILE_BYTES = "time".getBytes();
        private static final byte[] TXN_FILE_BYTES = "txn".getBytes();
        private static final byte[] PRODUCER_SNAPSHOT_FILE_BYTES = "pid".getBytes();
        private static final byte[] LEADER_EPOCH_CHECKPOINT_FILE_BYTES = "0\n2\n0 0\n2 12".getBytes();
        private final Path segmentPath = Paths.get("local-segments", new String[0]);
        private long baseOffset = 0L;

        LocalLogSegments() {
            if (Files.notExists(this.segmentPath, new LinkOption[0])) {
                try {
                    Files.createDirectories(this.segmentPath, new FileAttribute[0]);
                }
                catch (IOException ex) {
                    LOGGER.error("Failed to create directory: {}", (Object)this.segmentPath, (Object)ex);
                }
            }
        }

        LogSegmentData nextSegment() {
            return this.nextSegment(new byte[][]{new byte[0]});
        }

        LogSegmentData nextSegment(byte[] ... data) {
            String offset = LogFileUtils.filenamePrefixFromOffset((long)this.baseOffset);
            try {
                FileChannel channel = FileChannel.open(this.segmentPath.resolve(offset + ".log"), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
                ByteBuffer buffer = ByteBuffer.allocate(128);
                int magic = 2;
                MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (Compression)Compression.NONE, (TimestampType)TimestampType.CREATE_TIME, (long)this.baseOffset);
                for (byte[] value : data) {
                    builder.append(System.currentTimeMillis(), null, value);
                }
                builder.build().writeFullyTo((GatheringByteChannel)channel);
                channel.force(true);
                Path segment = this.segmentPath.resolve(offset + ".log");
                Path offsetIdx = this.segmentPath.resolve(offset + ".index");
                Path timeIdx = this.segmentPath.resolve(offset + ".timeindex");
                Path txnIdx = this.segmentPath.resolve(offset + ".txnindex");
                Path producerIdSnapshot = this.segmentPath.resolve(offset + ".snapshot");
                Files.write(offsetIdx, OFFSET_FILE_BYTES, new OpenOption[0]);
                Files.write(timeIdx, TIME_FILE_BYTES, new OpenOption[0]);
                Files.write(txnIdx, TXN_FILE_BYTES, new OpenOption[0]);
                Files.write(producerIdSnapshot, PRODUCER_SNAPSHOT_FILE_BYTES, new OpenOption[0]);
                this.baseOffset += (long)data.length;
                return new LogSegmentData(segment, offsetIdx, timeIdx, Optional.of(txnIdx), producerIdSnapshot, ByteBuffer.wrap(LEADER_EPOCH_CHECKPOINT_FILE_BYTES));
            }
            catch (IOException e) {
                throw new AssertionError((Object)e);
            }
        }

        void deleteAll() throws IOException {
            List paths = Files.list(this.segmentPath).collect(Collectors.toList());
            for (Path path : paths) {
                Files.delete(path);
            }
            Files.delete(this.segmentPath);
        }
    }

    private static interface Function<A, B> {
        public B apply(A var1) throws RemoteStorageException;
    }

    public static final class Verifier {
        private final LocalTieredStorage remoteStorage;
        private final TopicIdPartition topicIdPartition;

        public Verifier(LocalTieredStorage remoteStorage, TopicIdPartition topicIdPartition) {
            this.remoteStorage = Objects.requireNonNull(remoteStorage);
            this.topicIdPartition = Objects.requireNonNull(topicIdPartition);
        }

        private List<Path> expectedPaths(RemoteLogSegmentMetadata metadata) {
            String rootPath = this.getStorageRootDirectory();
            TopicPartition tp = this.topicIdPartition.topicPartition();
            String topicPartitionSubpath = String.format("%s-%d-%s", tp.topic(), tp.partition(), this.topicIdPartition.topicId());
            String uuid = metadata.remoteLogSegmentId().id().toString();
            String startOffset = LogFileUtils.filenamePrefixFromOffset((long)metadata.startOffset());
            return Arrays.asList(Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + ".log"), Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + ".index"), Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + ".timeindex"), Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + ".txnindex"), Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + RemoteLogSegmentFileset.RemoteLogSegmentFileType.LEADER_EPOCH_CHECKPOINT.getSuffix()), Paths.get(rootPath, topicPartitionSubpath, startOffset + "-" + uuid + ".snapshot"));
        }

        public Path expectedPartitionPath() {
            String rootPath = this.getStorageRootDirectory();
            TopicPartition tp = this.topicIdPartition.topicPartition();
            String topicPartitionSubpath = String.format("%s-%d-%s", this.topicIdPartition.topicId(), tp.partition(), tp.topic());
            return Paths.get(rootPath, topicPartitionSubpath);
        }

        public void verifyContainsLogSegmentFiles(RemoteLogSegmentMetadata metadata, Consumer<Path> action) {
            this.expectedPaths(metadata).forEach(action);
        }

        public void verifyContainsLogSegmentFiles(RemoteLogSegmentMetadata metadata) {
            this.expectedPaths(metadata).forEach(this::assertFileExists);
        }

        public void verifyLogSegmentFilesAbsent(RemoteLogSegmentMetadata metadata) {
            this.expectedPaths(metadata).forEach(this::assertFileDoesNotExist);
        }

        public void verifyRemoteLogSegmentMatchesLocal(RemoteLogSegmentMetadata metadata, LogSegmentData seg) {
            Path remoteSegmentPath = this.expectedPaths(metadata).get(0);
            this.assertFileDataEquals(remoteSegmentPath, seg.logSegment());
        }

        public void verifyFetchedLogSegment(RemoteLogSegmentId id, int startPosition, byte[] expected) {
            try {
                InputStream in = this.remoteStorage.fetchLogSegment(this.newMetadata(id), startPosition);
                ByteBuffer buffer = ByteBuffer.wrap(this.readFully(in));
                Iterator records = MemoryRecords.readableRecords((ByteBuffer)buffer).records().iterator();
                Assertions.assertTrue((boolean)records.hasNext());
                Assertions.assertEquals((Object)ByteBuffer.wrap(expected), (Object)((Record)records.next()).value());
            }
            catch (IOException | RemoteStorageException e) {
                throw new AssertionError((Object)e);
            }
        }

        public void verifyFetchedOffsetIndex(RemoteLogSegmentId id, byte[] expected) {
            this.verifyFileContents(metadata -> this.remoteStorage.fetchIndex((RemoteLogSegmentMetadata)metadata, RemoteStorageManager.IndexType.OFFSET), id, expected);
        }

        public void verifyFetchedTimeIndex(RemoteLogSegmentId id, byte[] expected) {
            this.verifyFileContents(metadata -> this.remoteStorage.fetchIndex((RemoteLogSegmentMetadata)metadata, RemoteStorageManager.IndexType.TIMESTAMP), id, expected);
        }

        public void verifyFetchedTransactionIndex(RemoteLogSegmentId id, byte[] expected) {
            this.verifyFileContents(metadata -> this.remoteStorage.fetchIndex((RemoteLogSegmentMetadata)metadata, RemoteStorageManager.IndexType.TRANSACTION), id, expected);
        }

        public void verifyLeaderEpochCheckpoint(RemoteLogSegmentId id, byte[] expected) {
            this.verifyFileContents(metadata -> this.remoteStorage.fetchIndex((RemoteLogSegmentMetadata)metadata, RemoteStorageManager.IndexType.LEADER_EPOCH), id, expected);
        }

        public void verifyProducerSnapshot(RemoteLogSegmentId id, byte[] expected) {
            this.verifyFileContents(metadata -> this.remoteStorage.fetchIndex((RemoteLogSegmentMetadata)metadata, RemoteStorageManager.IndexType.PRODUCER_SNAPSHOT), id, expected);
        }

        private void verifyFileContents(Function<RemoteLogSegmentMetadata, InputStream> actual, RemoteLogSegmentId id, byte[] expected) {
            try {
                InputStream in = actual.apply(this.newMetadata(id));
                Assertions.assertArrayEquals((byte[])expected, (byte[])this.readFully(in));
            }
            catch (IOException | RemoteStorageException e) {
                throw new AssertionError((Object)e);
            }
        }

        private RemoteLogSegmentMetadata newMetadata(RemoteLogSegmentId id) {
            return new RemoteLogSegmentMetadata(id, 0L, 0L, -1L, -1, 1000L, 1024, Collections.singletonMap(0, 0L));
        }

        private String getStorageRootDirectory() {
            try {
                return this.remoteStorage.getStorageDirectoryRoot();
            }
            catch (RemoteStorageException e) {
                throw new RuntimeException(e);
            }
        }

        private void assertFileExists(Path path) {
            Assertions.assertTrue((boolean)path.toFile().exists(), (String)String.format("File %s does not exist", path));
        }

        private void assertFileDoesNotExist(Path path) {
            Assertions.assertFalse((boolean)path.toFile().exists(), (String)String.format("File %s should not exist", path));
        }

        private void assertFileDataEquals(Path path1, Path path2) {
            try {
                this.assertFileExists(path1);
                Assertions.assertArrayEquals((byte[])Files.readAllBytes(path1), (byte[])Files.readAllBytes(path2));
            }
            catch (IOException e) {
                throw new AssertionError((Object)e);
            }
        }

        private byte[] readFully(InputStream in) throws IOException {
            int len;
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            while ((len = in.read(buffer)) != -1) {
                out.write(buffer, 0, len);
            }
            return out.toByteArray();
        }
    }
}

