/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.internals.Topic;
import org.apache.kafka.common.metadata.ConfigRecord;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.ConfigurationValidator;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.ResultOrError;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.policy.AlterConfigPolicy;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class ConfigurationControlManager {
    private final Logger log;
    private final SnapshotRegistry snapshotRegistry;
    private final Map<ConfigResource.Type, ConfigDef> configDefs;
    private final Optional<AlterConfigPolicy> alterConfigPolicy;
    private final ConfigurationValidator validator;
    private final TimelineHashMap<ConfigResource, TimelineHashMap<String, String>> configData;

    ConfigurationControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, Map<ConfigResource.Type, ConfigDef> configDefs, Optional<AlterConfigPolicy> alterConfigPolicy, ConfigurationValidator validator) {
        this.log = logContext.logger(ConfigurationControlManager.class);
        this.snapshotRegistry = snapshotRegistry;
        this.configDefs = configDefs;
        this.configData = new TimelineHashMap(snapshotRegistry, 0);
        this.alterConfigPolicy = alterConfigPolicy;
        this.validator = validator;
    }

    ControllerResult<Map<ConfigResource, ApiError>> incrementalAlterConfigs(Map<ConfigResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>>> configChanges) {
        ArrayList<ApiMessageAndVersion> outputRecords = new ArrayList<ApiMessageAndVersion>();
        HashMap<ConfigResource, ApiError> outputResults = new HashMap<ConfigResource, ApiError>();
        for (Map.Entry<ConfigResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>>> resourceEntry : configChanges.entrySet()) {
            this.incrementalAlterConfigResource(resourceEntry.getKey(), resourceEntry.getValue(), outputRecords, outputResults);
        }
        return ControllerResult.atomicOf(outputRecords, outputResults);
    }

    private void incrementalAlterConfigResource(ConfigResource configResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>> keysToOps, List<ApiMessageAndVersion> outputRecords, Map<ConfigResource, ApiError> outputResults) {
        ApiError error = ConfigurationControlManager.checkConfigResource(configResource);
        if (error.isFailure()) {
            outputResults.put(configResource, error);
            return;
        }
        ArrayList<ApiMessageAndVersion> newRecords = new ArrayList<ApiMessageAndVersion>();
        for (Map.Entry<String, Map.Entry<AlterConfigOp.OpType, String>> keysToOpsEntry : keysToOps.entrySet()) {
            String key = keysToOpsEntry.getKey();
            String currentValue = null;
            TimelineHashMap<String, String> currentConfigs = this.configData.get(configResource);
            if (currentConfigs != null) {
                currentValue = currentConfigs.get(key);
            }
            String newValue = currentValue;
            Map.Entry<AlterConfigOp.OpType, String> opTypeAndNewValue = keysToOpsEntry.getValue();
            AlterConfigOp.OpType opType = opTypeAndNewValue.getKey();
            String opValue = opTypeAndNewValue.getValue();
            switch (opType) {
                case SET: {
                    newValue = opValue;
                    break;
                }
                case DELETE: {
                    newValue = null;
                    break;
                }
                case APPEND: 
                case SUBTRACT: {
                    if (!this.isSplittable(configResource.type(), key)) {
                        outputResults.put(configResource, new ApiError(Errors.INVALID_CONFIG, "Can't " + opType + " to key " + key + " because its type is not LIST."));
                        return;
                    }
                    List<String> newValueParts = this.getParts(newValue, key, configResource);
                    if (opType == AlterConfigOp.OpType.APPEND) {
                        if (!newValueParts.contains(opValue)) {
                            newValueParts.add(opValue);
                        }
                        newValue = String.join((CharSequence)",", newValueParts);
                        break;
                    }
                    if (!newValueParts.remove(opValue)) break;
                    newValue = String.join((CharSequence)",", newValueParts);
                }
            }
            if (Objects.equals(currentValue, newValue)) continue;
            newRecords.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(newValue), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        error = this.validateAlterConfig(configResource, newRecords);
        if (error.isFailure()) {
            outputResults.put(configResource, error);
            return;
        }
        outputRecords.addAll(newRecords);
        outputResults.put(configResource, ApiError.NONE);
    }

    private ApiError validateAlterConfig(ConfigResource configResource, List<ApiMessageAndVersion> newRecords) {
        HashMap<String, String> newConfigs = new HashMap<String, String>();
        TimelineHashMap<String, String> existingConfigs = this.configData.get(configResource);
        if (existingConfigs != null) {
            newConfigs.putAll(existingConfigs);
        }
        for (ApiMessageAndVersion newRecord : newRecords) {
            ConfigRecord configRecord = (ConfigRecord)newRecord.message();
            if (configRecord.value() == null) {
                newConfigs.remove(configRecord.name());
                continue;
            }
            newConfigs.put(configRecord.name(), configRecord.value());
        }
        try {
            this.validator.validate(configResource, newConfigs);
            if (this.alterConfigPolicy.isPresent()) {
                this.alterConfigPolicy.get().validate(new AlterConfigPolicy.RequestMetadata(configResource, newConfigs));
            }
        }
        catch (ConfigException e) {
            return new ApiError(Errors.INVALID_CONFIG, e.getMessage());
        }
        catch (Throwable e) {
            return ApiError.fromThrowable((Throwable)e);
        }
        return ApiError.NONE;
    }

    ControllerResult<Map<ConfigResource, ApiError>> legacyAlterConfigs(Map<ConfigResource, Map<String, String>> newConfigs) {
        ArrayList<ApiMessageAndVersion> outputRecords = new ArrayList<ApiMessageAndVersion>();
        HashMap<ConfigResource, ApiError> outputResults = new HashMap<ConfigResource, ApiError>();
        for (Map.Entry<ConfigResource, Map<String, String>> resourceEntry : newConfigs.entrySet()) {
            this.legacyAlterConfigResource(resourceEntry.getKey(), resourceEntry.getValue(), outputRecords, outputResults);
        }
        return ControllerResult.atomicOf(outputRecords, outputResults);
    }

    private void legacyAlterConfigResource(ConfigResource configResource, Map<String, String> newConfigs, List<ApiMessageAndVersion> outputRecords, Map<ConfigResource, ApiError> outputResults) {
        ApiError error = ConfigurationControlManager.checkConfigResource(configResource);
        if (error.isFailure()) {
            outputResults.put(configResource, error);
            return;
        }
        ArrayList<ApiMessageAndVersion> newRecords = new ArrayList<ApiMessageAndVersion>();
        Map currentConfigs = this.configData.get(configResource);
        if (currentConfigs == null) {
            currentConfigs = Collections.emptyMap();
        }
        for (Map.Entry<String, String> entry : newConfigs.entrySet()) {
            String currentValue;
            String key = entry.getKey();
            String newValue = entry.getValue();
            if (Objects.equals(newValue, currentValue = (String)currentConfigs.get(key))) continue;
            newRecords.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(newValue), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        for (String key : currentConfigs.keySet()) {
            if (newConfigs.containsKey(key)) continue;
            newRecords.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(null), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        error = this.validateAlterConfig(configResource, newRecords);
        if (error.isFailure()) {
            outputResults.put(configResource, error);
            return;
        }
        outputRecords.addAll(newRecords);
        outputResults.put(configResource, ApiError.NONE);
    }

    private List<String> getParts(String value, String key, ConfigResource configResource) {
        String[] splitValues;
        if (value == null) {
            value = this.getConfigValueDefault(configResource.type(), key);
        }
        ArrayList<String> parts = new ArrayList<String>();
        if (value == null) {
            return parts;
        }
        for (String splitValue : splitValues = value.split(",")) {
            if (splitValue.isEmpty()) continue;
            parts.add(splitValue);
        }
        return parts;
    }

    static ApiError checkConfigResource(ConfigResource configResource) {
        switch (configResource.type()) {
            case BROKER_LOGGER: {
                return new ApiError(Errors.INVALID_REQUEST, "Unsupported configuration resource type BROKER_LOGGER ");
            }
            case BROKER: {
                if (!configResource.name().isEmpty()) {
                    try {
                        int brokerId = Integer.parseInt(configResource.name());
                        if (brokerId < 0) {
                            return new ApiError(Errors.INVALID_REQUEST, "Illegal negative broker ID in BROKER resource.");
                        }
                    }
                    catch (NumberFormatException e) {
                        return new ApiError(Errors.INVALID_REQUEST, "Illegal non-integral BROKER resource type name.");
                    }
                }
                return ApiError.NONE;
            }
            case TOPIC: {
                try {
                    Topic.validate((String)configResource.name());
                }
                catch (Exception e) {
                    return new ApiError(Errors.INVALID_REQUEST, "Illegal topic name.");
                }
                return ApiError.NONE;
            }
            case UNKNOWN: {
                return new ApiError(Errors.INVALID_REQUEST, "Unsupported configuration resource type UNKNOWN.");
            }
        }
        return new ApiError(Errors.INVALID_REQUEST, "Unsupported unexpected resource type");
    }

    boolean isSplittable(ConfigResource.Type type, String key) {
        ConfigDef configDef = this.configDefs.get(type);
        if (configDef == null) {
            return false;
        }
        ConfigDef.ConfigKey configKey = (ConfigDef.ConfigKey)configDef.configKeys().get(key);
        if (configKey == null) {
            return false;
        }
        return configKey.type == ConfigDef.Type.LIST;
    }

    String getConfigValueDefault(ConfigResource.Type type, String key) {
        ConfigDef configDef = this.configDefs.get(type);
        if (configDef == null) {
            return null;
        }
        ConfigDef.ConfigKey configKey = (ConfigDef.ConfigKey)configDef.configKeys().get(key);
        if (configKey == null || !configKey.hasDefault()) {
            return null;
        }
        return ConfigDef.convertToString((Object)configKey.defaultValue, (ConfigDef.Type)configKey.type);
    }

    public void replay(ConfigRecord record) {
        ConfigResource.Type type = ConfigResource.Type.forId((byte)record.resourceType());
        ConfigResource configResource = new ConfigResource(type, record.resourceName());
        TimelineHashMap<String, String> configs = this.configData.get(configResource);
        if (configs == null) {
            configs = new TimelineHashMap(this.snapshotRegistry, 0);
            this.configData.put(configResource, configs);
        }
        if (record.value() == null) {
            configs.remove(record.name());
        } else {
            configs.put(record.name(), record.value());
        }
        if (configs.isEmpty()) {
            this.configData.remove(configResource);
        }
        this.log.info("{}: set configuration {} to {}", new Object[]{configResource, record.name(), record.value()});
    }

    Map<String, String> getConfigs(ConfigResource configResource) {
        Map map = this.configData.get(configResource);
        if (map == null) {
            return Collections.emptyMap();
        }
        return Collections.unmodifiableMap(new HashMap(map));
    }

    public Map<ConfigResource, ResultOrError<Map<String, String>>> describeConfigs(long lastCommittedOffset, Map<ConfigResource, Collection<String>> resources) {
        HashMap<ConfigResource, ResultOrError<Map<String, String>>> results = new HashMap<ConfigResource, ResultOrError<Map<String, String>>>();
        for (Map.Entry<ConfigResource, Collection<String>> resourceEntry : resources.entrySet()) {
            ConfigResource resource = resourceEntry.getKey();
            ApiError error = ConfigurationControlManager.checkConfigResource(resource);
            if (error.isFailure()) {
                results.put(resource, new ResultOrError(error));
                continue;
            }
            HashMap<String, String> foundConfigs = new HashMap<String, String>();
            TimelineHashMap<String, String> configs = this.configData.get(resource, lastCommittedOffset);
            if (configs != null) {
                Collection<String> targetConfigs = resourceEntry.getValue();
                if (targetConfigs.isEmpty()) {
                    for (Map.Entry<String, String> entry : configs.entrySet(lastCommittedOffset)) {
                        foundConfigs.put(entry.getKey(), entry.getValue());
                    }
                } else {
                    for (String key : targetConfigs) {
                        String value = configs.get(key, lastCommittedOffset);
                        if (value == null) continue;
                        foundConfigs.put(key, value);
                    }
                }
            }
            results.put(resource, new ResultOrError(foundConfigs));
        }
        return results;
    }

    void deleteTopicConfigs(String name) {
        this.configData.remove(new ConfigResource(ConfigResource.Type.TOPIC, name));
    }

    boolean uncleanLeaderElectionEnabledForTopic(String name) {
        return false;
    }

    ConfigurationControlIterator iterator(long epoch) {
        return new ConfigurationControlIterator(epoch);
    }

    class ConfigurationControlIterator
    implements Iterator<List<ApiMessageAndVersion>> {
        private final long epoch;
        private final Iterator<Map.Entry<ConfigResource, TimelineHashMap<String, String>>> iterator;

        ConfigurationControlIterator(long epoch) {
            this.epoch = epoch;
            this.iterator = ConfigurationControlManager.this.configData.entrySet(epoch).iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public List<ApiMessageAndVersion> next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
            Map.Entry<ConfigResource, TimelineHashMap<String, String>> entry = this.iterator.next();
            ConfigResource resource = entry.getKey();
            for (Map.Entry<String, String> configEntry : entry.getValue().entrySet(this.epoch)) {
                records.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceName(resource.name()).setResourceType(resource.type().id()).setName(configEntry.getKey()).setValue(configEntry.getValue()), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
            }
            return records;
        }
    }
}

