/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.common.runtime;

import com.yammer.metrics.core.Histogram;
import com.yammer.metrics.core.MetricName;
import com.yammer.metrics.core.MetricsRegistry;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.common.utils.ThreadUtils;
import org.apache.kafka.coordinator.common.runtime.HdrHistogram;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class HdrHistogramTest {
    private static final double[] QUANTILES = new double[]{0.5, 0.75, 0.95, 0.98, 0.99, 0.999};
    private static final long MAX_VALUE = TimeUnit.MINUTES.toMillis(1L);
    private static final long REGULAR_VALUE = 100L;
    private static final int NUM_SIGNIFICANT_DIGITS = 3;

    @Test
    public void testHdrVsYammerUniform() {
        ArrayList<Long> samples = new ArrayList<Long>();
        for (long i = 0L; i <= MAX_VALUE; ++i) {
            samples.add(i);
        }
        this.testHdrHistogramVsYammerHistogram(samples, "uniform");
    }

    @Test
    public void testHdrVsYammerNormal() {
        ArrayList<Long> samples = new ArrayList<Long>();
        for (long i = 0L; i <= MAX_VALUE; ++i) {
            samples.add(ThreadLocalRandom.current().nextLong(MAX_VALUE));
        }
        this.testHdrHistogramVsYammerHistogram(samples, "normal");
    }

    @Test
    public void testHdrVsYammerBimodal() {
        ArrayList<Long> samples = new ArrayList<Long>();
        for (long i = 0L; i <= MAX_VALUE; ++i) {
            if (i % 500L == 0L) {
                samples.add(ThreadLocalRandom.current().nextLong(MAX_VALUE));
                continue;
            }
            samples.add(ThreadLocalRandom.current().nextLong(100L));
        }
        this.testHdrHistogramVsYammerHistogram(samples, "bimodal");
    }

    private void testHdrHistogramVsYammerHistogram(List<Long> samples, String distributionLabel) {
        HdrHistogram hdrHistogram = new HdrHistogram(MAX_VALUE, 3);
        Histogram yammerHistogram = new MetricsRegistry().newHistogram(new MetricName("", "", ""), true);
        Collections.sort(samples);
        double[] expectedQuantileValues = new double[QUANTILES.length];
        for (int i = 0; i < QUANTILES.length; ++i) {
            int quantileIndex = (int)Math.ceil((double)samples.size() * QUANTILES[i]) - 1;
            expectedQuantileValues[i] = samples.get(quantileIndex).longValue();
        }
        Collections.shuffle(samples);
        for (long sample : samples) {
            hdrHistogram.record(sample);
            yammerHistogram.update(sample);
        }
        System.out.printf("Testing HdrHistogram vs Yammer histogram for %s distribution%n", distributionLabel);
        long now = System.currentTimeMillis();
        int numYammerWins = 0;
        for (int i = 0; i < QUANTILES.length; ++i) {
            double quantile = QUANTILES[i];
            double hdrHistogramValue = hdrHistogram.measurePercentile(now, quantile * 100.0);
            double yammerHistogramValue = yammerHistogram.getSnapshot().getValue(quantile);
            double expectedValue = expectedQuantileValues[i];
            System.out.printf("Values for quantile %f: HdrHistogram: %f, Yammer histogram: %f, Expected: %f%n", quantile, hdrHistogramValue, yammerHistogramValue, expectedValue);
            if (!(Math.abs(expectedValue - hdrHistogramValue) > Math.abs(expectedValue - yammerHistogramValue))) continue;
            ++numYammerWins;
        }
        System.out.printf("HdrHistogram was more accurate: %d out of %d times%n", QUANTILES.length - numYammerWins, QUANTILES.length);
        Assertions.assertTrue((numYammerWins <= QUANTILES.length / 2 ? 1 : 0) != 0);
    }

    @Test
    public void testCount() throws Exception {
        int numUpdates = 100000;
        HdrHistogram hdrHistogram = new HdrHistogram(MAX_VALUE, 3);
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        for (int i = 0; i < numUpdates; ++i) {
            executorService.submit(() -> hdrHistogram.record(1L));
        }
        executorService.shutdown();
        Assertions.assertTrue((boolean)executorService.awaitTermination(1L, TimeUnit.SECONDS));
        Assertions.assertEquals((long)numUpdates, (long)hdrHistogram.count(System.currentTimeMillis()));
    }

    @Test
    public void testMax() throws Exception {
        int numSignificantDigits = 5;
        int numUpdates = 30000;
        double expectedMax = 0.0;
        long[] values = new long[numUpdates];
        for (int i = 0; i < numUpdates; ++i) {
            long value;
            values[i] = value = ThreadLocalRandom.current().nextLong(MAX_VALUE);
            expectedMax = Math.max(expectedMax, (double)value);
        }
        HdrHistogram hdrHistogram = new HdrHistogram(MAX_VALUE, numSignificantDigits);
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        for (int i = 0; i < numUpdates; ++i) {
            long value = values[i];
            executorService.submit(() -> hdrHistogram.record(value));
        }
        executorService.shutdown();
        Assertions.assertTrue((boolean)executorService.awaitTermination(1L, TimeUnit.SECONDS));
        double histogramMax = hdrHistogram.max(System.currentTimeMillis());
        Assertions.assertEquals((double)expectedMax, (double)histogramMax);
    }

    @Test
    public void testHistogramDataReset() {
        long maxSnapshotAgeMs = 10L;
        HdrHistogram hdrHistogram = new HdrHistogram(maxSnapshotAgeMs, MAX_VALUE, 3);
        int numEventsInFirstCycle = 1000;
        for (int i = 0; i < numEventsInFirstCycle; ++i) {
            hdrHistogram.record((long)i);
        }
        long now = System.currentTimeMillis();
        Assertions.assertEquals((long)numEventsInFirstCycle, (long)hdrHistogram.count(now));
        int numEventsInSecondCycle = 2000;
        for (int i = 0; i < numEventsInSecondCycle; ++i) {
            hdrHistogram.record((long)i);
        }
        Assertions.assertEquals((long)numEventsInFirstCycle, (long)hdrHistogram.count(now));
        Assertions.assertEquals((long)numEventsInFirstCycle, (long)hdrHistogram.count(now + maxSnapshotAgeMs));
        Assertions.assertEquals((long)numEventsInSecondCycle, (long)hdrHistogram.count(now + 1L + maxSnapshotAgeMs));
    }

    @Test
    public void testLatestHistogramRace() throws InterruptedException, ExecutionException {
        long maxSnapshotAgeMs = 10L;
        long now = System.currentTimeMillis();
        HdrHistogram hdrHistogram = new HdrHistogram(maxSnapshotAgeMs, MAX_VALUE, 1);
        ExecutorService countExecutor = Executors.newFixedThreadPool(2);
        for (int i = 1; i < 10000; ++i) {
            long moreThanMaxAge;
            int numEvents = 2;
            for (int j = 0; j < numEvents; ++j) {
                hdrHistogram.record((long)i);
            }
            now = moreThanMaxAge = now + maxSnapshotAgeMs + 1L;
            CountDownLatch latch = new CountDownLatch(1);
            Callable<Long> countTask = () -> {
                try {
                    Assertions.assertTrue((boolean)latch.await(500L, TimeUnit.MILLISECONDS));
                    return hdrHistogram.count(moreThanMaxAge);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            };
            Future<Long> t1Future = countExecutor.submit(countTask);
            Future<Long> t2Future = countExecutor.submit(countTask);
            latch.countDown();
            long t1Count = t1Future.get();
            long t2Count = t2Future.get();
            Assertions.assertTrue(((long)numEvents == t1Count && (long)numEvents == t2Count ? 1 : 0) != 0, (String)String.format("Expected %d events in both threads, got %d in T1 and %d in T2", numEvents, t1Count, t2Count));
        }
        ThreadUtils.shutdownExecutorServiceQuietly((ExecutorService)countExecutor, (long)500L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    @Test
    public void testRecordLimit() {
        long highestTrackableValue = 10L;
        HdrHistogram hdrHistogram = new HdrHistogram(10L, highestTrackableValue, 3);
        hdrHistogram.record(highestTrackableValue + 1000L);
        Assertions.assertEquals((long)highestTrackableValue, (long)hdrHistogram.max(System.currentTimeMillis()));
    }
}

