/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.kafka.common.requests;

import org.apache.kafka.common.acl.AccessControlEntry;
import org.apache.kafka.common.acl.AclBinding;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.types.Struct;
import org.apache.kafka.common.resource.Resource;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class DeleteAclsResponse extends AbstractResponse {
    public static final Logger log = LoggerFactory.getLogger(DeleteAclsResponse.class);
    private final static String FILTER_RESPONSES = "filter_responses";
    private final static String MATCHING_ACLS = "matching_acls";

    public static class AclDeletionResult {
        private final ApiError error;
        private final AclBinding acl;

        public AclDeletionResult(ApiError error, AclBinding acl) {
            this.error = error;
            this.acl = acl;
        }

        public AclDeletionResult(AclBinding acl) {
            this(ApiError.NONE, acl);
        }

        public ApiError error() {
            return error;
        }

        public AclBinding acl() {
            return acl;
        }

        @Override
        public String toString() {
            return "(error=" + error + ", acl=" + acl + ")";
        }
    }

    public static class AclFilterResponse {
        private final ApiError error;
        private final Collection<AclDeletionResult> deletions;

        public AclFilterResponse(ApiError error, Collection<AclDeletionResult> deletions) {
            this.error = error;
            this.deletions = deletions;
        }

        public AclFilterResponse(Collection<AclDeletionResult> deletions) {
            this(ApiError.NONE, deletions);
        }

        public ApiError error() {
            return error;
        }

        public Collection<AclDeletionResult> deletions() {
            return deletions;
        }

        @Override
        public String toString() {
            return "(error=" + error + ", deletions=" + Utils.join(deletions, ",") + ")";
        }
    }

    private final int throttleTimeMs;

    private final List<AclFilterResponse> responses;

    public DeleteAclsResponse(int throttleTimeMs, List<AclFilterResponse> responses) {
        this.throttleTimeMs = throttleTimeMs;
        this.responses = responses;
    }

    public DeleteAclsResponse(Struct struct) {
        this.throttleTimeMs = struct.getInt(THROTTLE_TIME_KEY_NAME);
        this.responses = new ArrayList<>();
        for (Object responseStructObj : struct.getArray(FILTER_RESPONSES)) {
            Struct responseStruct = (Struct) responseStructObj;
            ApiError error = new ApiError(responseStruct);
            List<AclDeletionResult> deletions = new ArrayList<>();
            for (Object matchingAclStructObj : responseStruct.getArray(MATCHING_ACLS)) {
                Struct matchingAclStruct = (Struct) matchingAclStructObj;
                ApiError matchError = new ApiError(matchingAclStruct);
                AccessControlEntry entry = RequestUtils.aceFromStructFields(matchingAclStruct);
                Resource resource = RequestUtils.resourceFromStructFields(matchingAclStruct);
                deletions.add(new AclDeletionResult(matchError, new AclBinding(resource, entry)));
            }
            this.responses.add(new AclFilterResponse(error, deletions));
        }
    }

    @Override
    protected Struct toStruct(short version) {
        Struct struct = new Struct(ApiKeys.DELETE_ACLS.responseSchema(version));
        struct.set(THROTTLE_TIME_KEY_NAME, throttleTimeMs);
        List<Struct> responseStructs = new ArrayList<>();
        for (AclFilterResponse response : responses) {
            Struct responseStruct = struct.instance(FILTER_RESPONSES);
            response.error.write(responseStruct);
            List<Struct> deletionStructs = new ArrayList<>();
            for (AclDeletionResult deletion : response.deletions()) {
                Struct deletionStruct = responseStruct.instance(MATCHING_ACLS);
                deletion.error.write(deletionStruct);
                RequestUtils.resourceSetStructFields(deletion.acl().resource(), deletionStruct);
                RequestUtils.aceSetStructFields(deletion.acl().entry(), deletionStruct);
                deletionStructs.add(deletionStruct);
            }
            responseStruct.set(MATCHING_ACLS, deletionStructs.toArray(new Struct[0]));
            responseStructs.add(responseStruct);
        }
        struct.set(FILTER_RESPONSES, responseStructs.toArray());
        return struct;
    }

    public int throttleTimeMs() {
        return throttleTimeMs;
    }

    public List<AclFilterResponse> responses() {
        return responses;
    }

    public static DeleteAclsResponse parse(ByteBuffer buffer, short version) {
        return new DeleteAclsResponse(ApiKeys.DELETE_ACLS.responseSchema(version).read(buffer));
    }

    public String toString() {
        return "(responses=" + Utils.join(responses, ",") + ")";
    }

}
