// ***************************************************************************************************************************
// * Licensed to the Apache Software Foundation (ASF) under one or more contributor license agreements.  See the NOTICE file *
// * distributed with this work for additional information regarding copyright ownership.  The ASF licenses this file        *
// * to you under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance            *
// * with the License.  You may obtain a copy of the License at                                                              *
// *                                                                                                                         *
// *  http://www.apache.org/licenses/LICENSE-2.0                                                                             *
// *                                                                                                                         *
// * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an  *
// * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the        *
// * specific language governing permissions and limitations under the License.                                              *
// ***************************************************************************************************************************
package org.apache.juneau.dto.atom;

import static org.apache.juneau.xml.annotation.XmlFormat.*;

import java.net.URI;

import org.apache.juneau.annotation.*;
import org.apache.juneau.xml.annotation.*;

/**
 * Represents an <code>atomCategory</code> construct in the RFC4287 specification.
 * <p>
 * <h6 class='figure'>Schema</h6>
 * <p class='bcode'>
 * 	atomCategory =
 * 		element atom:category {
 * 			atomCommonAttributes,
 * 			attribute term { text },
 * 			attribute scheme { atomUri }?,
 * 			attribute label { text }?,
 * 			undefinedContent
 * 		}
 * </p>
 * <p>
 * Refer to <a class='doclink' href='package-summary.html#TOC'>org.apache.juneau.dto.atom</a> for further information about ATOM support.
 */
@Bean(typeName="category")
@SuppressWarnings("hiding")
public class Category extends Common {

	private String term;
	private URI scheme;
	private String label;

	/**
	 * Normal constructor.
	 * @param term The category term.
	 */
	public Category(String term) {
		term(term);
	}

	/** Bean constructor. */
	public Category() {}


	//--------------------------------------------------------------------------------
	// Bean properties
	//--------------------------------------------------------------------------------

	/**
	 * @return The category term.
	 */
	@Xml(format=ATTR)
	public String getTerm() {
		return term;
	}

	/**
	 * Sets the category term.
	 *
	 * @param term The category term.
	 * @return This object (for method chaining).
	 */
	@BeanProperty(name="term")
	public Category term(String term) {
		this.term = term;
		return this;
	}

	/**
	 * Returns the category scheme.
	 *
	 * @return The category scheme.
	 */
	@Xml(format=ATTR)
	public URI getScheme() {
		return scheme;
	}

	/**
	 * Sets the category scheme.
	 *
	 * @param scheme The category scheme.
	 * @return This object (for method chaining).
	 */
	@BeanProperty(name="scheme")
	public Category scheme(URI scheme) {
		this.scheme = scheme;
		return this;
	}

	/**
	 * Returns the category label.
	 *
	 * @return The category label.
	 */
	@Xml(format=ATTR)
	public String getLabel() {
		return label;
	}

	/**
	 * Sets the category label.
	 *
	 * @param label The category label.
	 * @return This object (for method chaining).
	 */
	@BeanProperty(name="label")
	public Category label(String label) {
		this.label = label;
		return this;
	}


	//--------------------------------------------------------------------------------
	// Overridden setters (to simplify method chaining)
	//--------------------------------------------------------------------------------

	@Override /* Common */
	public Category base(URI base) {
		super.base(base);
		return this;
	}

	@Override /* Common */
	public Category lang(String lang) {
		super.lang(lang);
		return this;
	}
}
