/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.util.mime;

import com.google.common.base.Charsets;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Optional;
import org.apache.james.mime4j.dom.Entity;
import org.apache.james.mime4j.dom.Header;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.Multipart;
import org.apache.james.mime4j.field.Fields;
import org.apache.james.mime4j.message.BasicBodyFactory;
import org.apache.james.mime4j.message.BodyPart;
import org.apache.james.mime4j.message.BodyPartBuilder;
import org.apache.james.mime4j.message.HeaderImpl;
import org.apache.james.mime4j.message.MessageBuilder;
import org.apache.james.mime4j.message.MultipartBuilder;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.util.ByteSequence;
import org.apache.james.util.mime.MessageContentExtractor;
import org.assertj.core.api.Assertions;
import org.junit.Before;
import org.junit.Test;

public class MessageContentExtractorTest {
    private static final String BINARY_CONTENT = "binary";
    private static final String TEXT_CONTENT = "text content";
    private static final String HTML_CONTENT = "<b>html</b> content";
    private static final String TEXT_CONTENT2 = "other text content";
    private static final String HTML_CONTENT2 = "other <b>html</b> content";
    private static final String ATTACHMENT_CONTENT = "attachment content";
    private static final String ANY_VALUE = "anyValue";
    private static final Field CONTENT_ID_FIELD = new Field(){

        public String getName() {
            return "Content-ID";
        }

        public String getBody() {
            return MessageContentExtractorTest.ANY_VALUE;
        }

        public ByteSequence getRaw() {
            return ByteSequence.EMPTY;
        }
    };
    private MessageContentExtractor testee;
    private BodyPart htmlPart;
    private BodyPart textPart;
    private BodyPart textAttachment;
    private BodyPart inlineText;
    private BodyPart inlineImage;

    @Before
    public void setup() throws IOException {
        this.testee = new MessageContentExtractor();
        this.textPart = BodyPartBuilder.create().setBody(TEXT_CONTENT, "plain", Charsets.UTF_8).build();
        this.htmlPart = BodyPartBuilder.create().setBody(HTML_CONTENT, "html", Charsets.UTF_8).build();
        this.textAttachment = BodyPartBuilder.create().setBody(ATTACHMENT_CONTENT, "plain", Charsets.UTF_8).setContentDisposition("attachment").build();
        this.inlineText = BodyPartBuilder.create().setBody(ATTACHMENT_CONTENT, "plain", Charsets.UTF_8).setContentDisposition("inline").build();
        this.inlineImage = BodyPartBuilder.create().setBody(new byte[0], "image/png").setContentDisposition("inline").build();
    }

    @Test
    public void extractShouldReturnEmptyWhenBinaryContentOnly() throws IOException {
        Message message = MessageBuilder.create().setBody(BasicBodyFactory.INSTANCE.binaryBody(BINARY_CONTENT, Charsets.UTF_8)).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).isEmpty();
    }

    @Test
    public void extractShouldReturnTextOnlyWhenTextOnlyBody() throws IOException {
        Message message = MessageBuilder.create().setBody(TEXT_CONTENT, Charsets.UTF_8).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).isEmpty();
    }

    @Test
    public void extractShouldReturnHtmlOnlyWhenHtmlOnlyBody() throws IOException {
        Message message = MessageBuilder.create().setBody(HTML_CONTENT, "html", Charsets.UTF_8).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnHtmlAndTextWhenMultipartAlternative() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.textPart).addBodyPart((Entity)this.htmlPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnHtmlWhenMultipartAlternativeWithoutPlainPart() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.htmlPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnTextWhenMultipartAlternativeWithoutHtmlPart() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.textPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).isEmpty();
    }

    @Test
    public void extractShouldReturnFirstNonAttachmentPartWhenMultipartMixed() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"mixed").addBodyPart((Entity)this.textAttachment).addBodyPart((Entity)this.htmlPart).addBodyPart((Entity)this.textPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
    }

    @Test
    public void extractShouldReturnInlinedTextBodyWithoutCIDWhenNoOtherValidParts() throws IOException {
        String textBody = "body 1";
        Multipart multipart = MultipartBuilder.create((String)"report").addBodyPart((Entity)BodyPartBuilder.create().setBody(textBody, "plain", Charsets.UTF_8).setContentDisposition("inline").build()).addBodyPart((Entity)BodyPartBuilder.create().setBody("body 2", "rfc822-headers", Charsets.UTF_8).setContentDisposition("inline").build()).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)textBody);
    }

    @Test
    public void extractShouldReturnEmptyWhenMultipartMixedAndFirstPartIsATextAttachment() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"mixed").addBodyPart((Entity)this.textAttachment).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).isEmpty();
    }

    @Test
    public void extractShouldReturnFirstPartOnlyWhenMultipartMixedAndFirstPartIsHtml() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"mixed").addBodyPart((Entity)this.htmlPart).addBodyPart((Entity)this.textPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnHtmlAndTextWhenMultipartMixedAndFirstPartIsMultipartAlternative() throws IOException {
        BodyPart multipartAlternative = BodyPartBuilder.create().setBody(MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.htmlPart).addBodyPart((Entity)this.textPart).build()).build();
        Multipart multipartMixed = MultipartBuilder.create((String)"mixed").addBodyPart((Entity)multipartAlternative).build();
        Message message = MessageBuilder.create().setBody(multipartMixed).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnHtmlWhenMultipartRelated() throws IOException {
        Multipart multipart = MultipartBuilder.create((String)"related").addBodyPart((Entity)this.htmlPart).build();
        Message message = MessageBuilder.create().setBody(multipart).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldReturnHtmlAndTextWhenMultipartAlternativeAndFirstPartIsMultipartRelated() throws IOException {
        BodyPart multipartRelated = BodyPartBuilder.create().setBody(MultipartBuilder.create((String)"related").addBodyPart((Entity)this.htmlPart).build()).build();
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)multipartRelated).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldRetrieveHtmlBodyWithOneInlinedHTMLAttachmentWithoutCid() throws IOException {
        BodyPart inlinedHTMLPart = BodyPartBuilder.create().setBody(HTML_CONTENT, "html", Charsets.UTF_8).build();
        HeaderImpl inlinedHeader = new HeaderImpl();
        inlinedHeader.addField((Field)Fields.contentDisposition((String)"inline"));
        inlinedHeader.addField((Field)Fields.contentType((String)"text/html; charset=utf-8"));
        inlinedHTMLPart.setHeader((Header)inlinedHeader);
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)inlinedHTMLPart).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldNotRetrieveHtmlBodyWithOneInlinedHTMLAttachmentWithCid() throws IOException {
        BodyPart inlinedHTMLPart = BodyPartBuilder.create().setBody(HTML_CONTENT, "html", Charsets.UTF_8).build();
        HeaderImpl inlinedHeader = new HeaderImpl();
        inlinedHeader.addField((Field)Fields.contentDisposition((String)"inline"));
        inlinedHeader.addField((Field)Fields.contentType((String)"text/html; charset=utf-8"));
        inlinedHeader.addField(CONTENT_ID_FIELD);
        inlinedHTMLPart.setHeader((Header)inlinedHeader);
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)inlinedHTMLPart).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getHtmlBody()).isEmpty();
    }

    @Test
    public void extractShouldRetrieveTextBodyWithOneInlinedTextAttachmentWithoutCid() throws IOException {
        BodyPart inlinedTextPart = BodyPartBuilder.create().setBody(TEXT_CONTENT, "text", Charsets.UTF_8).build();
        HeaderImpl inlinedHeader = new HeaderImpl();
        inlinedHeader.addField((Field)Fields.contentDisposition((String)"inline"));
        inlinedHeader.addField((Field)Fields.contentType((String)"text/plain; charset=utf-8"));
        inlinedTextPart.setHeader((Header)inlinedHeader);
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)inlinedTextPart).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
    }

    @Test
    public void extractShouldNotRetrieveTextBodyWithOneInlinedTextAttachmentWithCid() throws IOException {
        BodyPart inlinedTextPart = BodyPartBuilder.create().setBody(TEXT_CONTENT, "text", Charsets.UTF_8).build();
        HeaderImpl inlinedHeader = new HeaderImpl();
        inlinedHeader.addField((Field)Fields.contentDisposition((String)"inline"));
        inlinedHeader.addField((Field)Fields.contentType((String)"text/plain; charset=utf-8"));
        inlinedHeader.addField(CONTENT_ID_FIELD);
        inlinedTextPart.setHeader((Header)inlinedHeader);
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)inlinedTextPart).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).isEmpty();
    }

    @Test
    public void extractShouldRetrieveTextAndHtmlBodyWhenOneInlinedTextAttachmentAndMainContentInMultipart() throws IOException {
        BodyPart multipartAlternative = BodyPartBuilder.create().setBody(MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.textPart).addBodyPart((Entity)this.htmlPart).build()).build();
        Multipart multipartMixed = MultipartBuilder.create((String)"mixed").addBodyPart((Entity)multipartAlternative).addBodyPart((Entity)this.inlineText).build();
        Message message = MessageBuilder.create().setBody(multipartMixed).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void extractShouldRetrieveTextBodyAndHtmlBodyWhenTextBodyInMainMultipartAndHtmlBodyInInnerMultipart() throws IOException {
        BodyPart multipartRelated = BodyPartBuilder.create().setBody(MultipartBuilder.create((String)"related").addBodyPart((Entity)this.htmlPart).addBodyPart((Entity)this.inlineImage).build()).build();
        Multipart multipartAlternative = MultipartBuilder.create((String)"alternative").addBodyPart((Entity)this.textPart).addBodyPart((Entity)multipartRelated).build();
        Message message = MessageBuilder.create().setBody(multipartAlternative).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)TEXT_CONTENT);
        Assertions.assertThat((Optional)actual.getHtmlBody()).contains((Object)HTML_CONTENT);
    }

    @Test
    public void mergeMessageContentShouldReturnEmptyWhenAllEmpty() {
        MessageContentExtractor.MessageContent messageContent1 = MessageContentExtractor.MessageContent.empty();
        MessageContentExtractor.MessageContent messageContent2 = MessageContentExtractor.MessageContent.empty();
        MessageContentExtractor.MessageContent expected = MessageContentExtractor.MessageContent.empty();
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void mergeMessageContentShouldReturnFirstWhenSecondEmpty() {
        MessageContentExtractor.MessageContent messageContent1 = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT), Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent messageContent2 = MessageContentExtractor.MessageContent.empty();
        MessageContentExtractor.MessageContent expected = messageContent1;
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void mergeMessageContentShouldReturnSecondWhenFirstEmpty() {
        MessageContentExtractor.MessageContent messageContent2;
        MessageContentExtractor.MessageContent messageContent1 = MessageContentExtractor.MessageContent.empty();
        MessageContentExtractor.MessageContent expected = messageContent2 = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT), Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void mergeMessageContentShouldReturnMixWhenFirstTextOnlyAndSecondHtmlOnly() {
        MessageContentExtractor.MessageContent messageContent1 = MessageContentExtractor.MessageContent.ofTextOnly(Optional.of(TEXT_CONTENT));
        MessageContentExtractor.MessageContent messageContent2 = MessageContentExtractor.MessageContent.ofHtmlOnly(Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent expected = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT), Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void mergeMessageContentShouldReturnMixWhenFirstHtmlOnlyAndSecondTextOnly() {
        MessageContentExtractor.MessageContent messageContent1 = MessageContentExtractor.MessageContent.ofHtmlOnly(Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent messageContent2 = MessageContentExtractor.MessageContent.ofTextOnly(Optional.of(TEXT_CONTENT));
        MessageContentExtractor.MessageContent expected = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT), Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void mergeMessageContentShouldReturnFirstWhenTwiceAreComplete() {
        MessageContentExtractor.MessageContent messageContent1 = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT), Optional.of(HTML_CONTENT));
        MessageContentExtractor.MessageContent messageContent2 = new MessageContentExtractor.MessageContent(Optional.of(TEXT_CONTENT2), Optional.of(HTML_CONTENT2));
        MessageContentExtractor.MessageContent expected = messageContent1;
        MessageContentExtractor.MessageContent actual = messageContent1.merge(messageContent2);
        Assertions.assertThat((Object)actual).isEqualTo((Object)expected);
    }

    @Test
    public void extractShouldRespectCharsetWhenOtherThanUTF8() throws IOException {
        String text = "\u00e9\u00e9\u00e9\u00e9\r\n\u00e8\u00e8\u00e8\u00e8\r\n\u00e0\u00e0\u00e0\u00e0";
        Message message = MessageBuilder.create().setBody(text, Charset.forName("windows-1252")).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)text);
    }

    @Test
    public void extractShouldRespectCharsetWhenUTF8() throws IOException {
        String text = "\u00e9\u00e9\u00e9\u00e9\r\n\u00e8\u00e8\u00e8\u00e8\r\n\u00e0\u00e0\u00e0\u00e0";
        Message message = MessageBuilder.create().setBody(text, Charsets.UTF_8).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)text);
    }

    @Test
    public void extractShouldUseUSASCIIWhenNoCharset() throws IOException {
        String text = "\u00e9\u00e9\u00e9\u00e9\r\n\u00e8\u00e8\u00e8\u00e8\r\n\u00e0\u00e0\u00e0\u00e0";
        Message message = MessageBuilder.create().setBody(text, null).build();
        MessageContentExtractor.MessageContent actual = this.testee.extract(message);
        Assertions.assertThat((Optional)actual.getTextBody()).contains((Object)"????\r\n????\r\n????");
    }
}

