/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.filesystem.api;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import junitparams.JUnitParamsRunner;
import junitparams.Parameters;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.james.filesystem.api.FileSystem;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import spark.Service;

@RunWith(value=JUnitParamsRunner.class)
public abstract class AbstractFileSystemTest {
    private static final int RANDOM_PORT = 0;
    private static final String FAKE_DIRECTORY = "b7b73e3a-5234-11e5-87f2-9b171f273b49/";
    private static final String FAKE_FILE = "d9091ae6-521f-11e5-b666-bb11fef67c2a";
    private static final String EXISTING_CLASSPATH_FILE = "classpathTest.txt";
    private static final String EXISTING_CLASSPATH_FILE_WITH_SPACES = "class path Test.txt";
    @Rule
    public TemporaryFolder tmpFolder = new TemporaryFolder();
    protected FileSystem fileSystem;
    private Service httpServer;
    private File rootDirectory;

    protected abstract FileSystem buildFileSystem(String var1);

    @Before
    public void setUp() throws Exception {
        this.httpServer = Service.ignite().port(0);
        this.httpServer.get("/", (request, response) -> "content");
        this.httpServer.awaitInitialization();
        this.rootDirectory = this.tmpFolder.getRoot();
        this.createSubFolderWithAFileIn("conf", "conf.txt", "confcontent");
        this.createSubFolderWithAFileIn("var", "var.txt", "varcontent");
        this.fileSystem = this.buildFileSystem(this.rootDirectory.getAbsolutePath());
    }

    private void createSubFolderWithAFileIn(String folderName, String fileName, String fileContent) throws IOException {
        File folder = this.tmpFolder.newFolder(folderName);
        File file = new File(folder.getAbsolutePath() + "/" + fileName);
        FileUtils.writeStringToFile((File)file, (String)fileContent, (Charset)StandardCharsets.UTF_8);
    }

    @After
    public void tearDown() throws Exception {
        this.httpServer.stop();
    }

    @Test
    public final void getBaseDirShouldReturnParentDir() throws Exception {
        File basedir = this.fileSystem.getBasedir();
        Assertions.assertThat((String)basedir.getPath()).isEqualTo((Object)this.rootDirectory.getAbsolutePath());
    }

    @Test(expected=NullPointerException.class)
    public final void nullInputShouldThrowNullPointerException() throws Exception {
        this.fileSystem.getFile(null);
    }

    public final void emptyInputShouldThrowReturnEmptyPathFile() throws Exception {
        File file = this.fileSystem.getFile("");
        Assertions.assertThat((String)file.getPath()).isEmpty();
    }

    @Test
    public final void protocolOnlyShouldReturnEmptyPathFile() throws Exception {
        File file = this.fileSystem.getFile("file:");
        Assertions.assertThat((String)file.getPath()).isEmpty();
    }

    @Test
    public final void protocolWithDoubleSlashesOnlyShouldReturnDir() throws Exception {
        File file = this.fileSystem.getFile("file://");
        Assertions.assertThat((File)file).isDirectory();
    }

    @Test(expected=FileNotFoundException.class)
    @Parameters(source=UrlsAsFileThrowingFileNotFoundExceptionProvider.class)
    public final void urlAsFileThrowingFileNotFoundException(String url) throws Exception {
        url = this.replacePort(url);
        this.fileSystem.getFile(url);
    }

    @Test
    @Parameters(source=NonExistingFilesProvider.class)
    public void nonExistingFilesShouldNotExist(String url) throws Exception {
        File f = this.fileSystem.getFile(url);
        Assertions.assertThat((File)f).doesNotExist();
    }

    @Test(expected=FileNotFoundException.class)
    @Parameters(source=NonAvailableStreamsProvider.class)
    public final void getFakeHttpResourceAsInputStreamShouldThrow(String url) throws Exception {
        url = this.replacePort(url);
        this.fileSystem.getResource(url);
    }

    @Test
    @Parameters(source=AvailableStreamsProvider.class)
    public final void availableInputStreamShouldReturnANonEmptyStream(String url) throws Exception {
        url = this.replacePort(url);
        InputStream inputStream = this.fileSystem.getResource(url);
        try {
            Assertions.assertThat((int)IOUtils.toByteArray((InputStream)inputStream).length).isGreaterThan(0);
        }
        finally {
            IOUtils.closeQuietly((InputStream)inputStream);
        }
    }

    private String replacePort(String url) {
        return url.replace("$PORT$", String.valueOf(this.httpServer.port()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Parameters(source=FileToCreateProvider.class)
    public final void createdFilesShouldExist(String name, String extension) throws Exception {
        File temp = this.createTempFile(name, extension);
        try {
            File expected = this.fileSystem.getFile("file:" + temp.getAbsolutePath());
            Assertions.assertThat((File)expected).exists();
        }
        finally {
            temp.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Parameters(source=FileToCreateProvider.class)
    public final void createdFilesShouldExistWhenAccessedWithTwoSlashes(String name, String extension) throws Exception {
        File temp = this.createTempFile(name, extension);
        try {
            File expected = this.fileSystem.getFile("file://" + temp.getAbsolutePath());
            Assertions.assertThat((File)expected).exists();
        }
        finally {
            temp.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Parameters(source=FileToCreateProvider.class)
    public final void createdFilesAsInputStreamShouldBeAvailable(String name, String extension) throws Exception {
        File temp = this.createTempFile(name, extension);
        InputStream inputStream = null;
        try {
            inputStream = this.fileSystem.getResource("file:" + temp.getAbsolutePath());
            Assertions.assertThat((String)IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8)).isEqualTo((Object)"content");
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(inputStream);
            temp.delete();
            throw throwable;
        }
        IOUtils.closeQuietly((InputStream)inputStream);
        temp.delete();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Parameters(source=FileToCreateProvider.class)
    public final void createdFilesAsInputStreamShouldBeAvailableWhenAccessedWithTwoSlashes(String name, String extension) throws Exception {
        File temp = this.createTempFile(name, extension);
        InputStream inputStream = null;
        try {
            inputStream = this.fileSystem.getResource("file://" + temp.getAbsolutePath());
            Assertions.assertThat((String)IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8)).isEqualTo((Object)"content");
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(inputStream);
            temp.delete();
            throw throwable;
        }
        IOUtils.closeQuietly((InputStream)inputStream);
        temp.delete();
    }

    private File createTempFile(String name, String extension) throws IOException {
        File temp = File.createTempFile(name, extension);
        FileUtils.write((File)temp, (CharSequence)"content", (Charset)StandardCharsets.UTF_8);
        return temp;
    }

    @Test
    public void testConfProtocolSouldReturnConfFile() throws Exception {
        File file = this.fileSystem.getFile("file://conf/conf.txt");
        Assertions.assertThat((File)file).hasContent("confcontent");
    }

    @Test
    public void testVarProtocolSouldReturnVarFile() throws Exception {
        File file = this.fileSystem.getFile("file://var/var.txt");
        Assertions.assertThat((File)file).hasContent("varcontent");
    }

    public static class FileToCreateProvider {
        public static Object[] provide() {
            return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{"fileSystemTest", ".txt"}), JUnitParamsRunner.$((Object[])new Object[]{"file System Test", ".txt"})});
        }
    }

    public static class AvailableStreamsProvider {
        public static Object[] provide() {
            return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{"http://localhost:$PORT$/"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:java/lang/String.class"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:classpathTest.txt"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:class path Test.txt"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:/classpathTest.txt"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:/class path Test.txt"})});
        }
    }

    public static class NonAvailableStreamsProvider {
        public static Object[] provide() {
            return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{"http://localhost:$PORT$/d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:java/lang/d9091ae6-521f-11e5-b666-bb11fef67c2a.clas"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:d9091ae6-521f-11e5-b666-bb11fef67c2a"})});
        }
    }

    public static class NonExistingFilesProvider {
        public static Object[] provides() {
            return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{"file:///d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"file:///b7b73e3a-5234-11e5-87f2-9b171f273b49/d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"file://conf/d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"file://var/d9091ae6-521f-11e5-b666-bb11fef67c2a"})});
        }
    }

    public static class UrlsAsFileThrowingFileNotFoundExceptionProvider {
        public static Object[] provides() {
            return JUnitParamsRunner.$((Object[])new Object[]{JUnitParamsRunner.$((Object[])new Object[]{"bad://file"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:/d9091ae6-521f-11e5-b666-bb11fef67c2a"}), JUnitParamsRunner.$((Object[])new Object[]{"http://localhost:$PORT$/"}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:" + new File(ClassLoader.getSystemResource(AbstractFileSystemTest.EXISTING_CLASSPATH_FILE).getFile()).getAbsolutePath()}), JUnitParamsRunner.$((Object[])new Object[]{"classpath:java/lang/String.class"})});
        }
    }
}

