/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/
package org.apache.james.rrt.lib;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.fail;

import java.util.Map;

import org.apache.james.lifecycle.api.LifecycleUtil;
import org.apache.james.rrt.api.RecipientRewriteTable;
import org.apache.james.rrt.api.RecipientRewriteTable.ErrorMappingException;
import org.apache.james.rrt.api.RecipientRewriteTableException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * The abstract test for the virtual user table. Contains tests related to
 * simple, regexp, wildcard, error,... Extend this and instanciate the needed
 * virtualUserTable implementation.
 */
public abstract class AbstractRecipientRewriteTableTest {

    protected AbstractRecipientRewriteTable virtualUserTable;
    protected final static int REGEX_TYPE = 0;
    protected final static int ERROR_TYPE = 1;
    protected final static int ADDRESS_TYPE = 2;
    protected final static int ALIASDOMAIN_TYPE = 3;

    @Before
    public void setUp() throws Exception {
        virtualUserTable = getRecipientRewriteTable();
    }

    @After
    public void tearDown() throws Exception {

        Map<String, Mappings> mappings = virtualUserTable.getAllMappings();

        if (mappings != null) {

            for (String key : virtualUserTable.getAllMappings().keySet()) {
                String args[] = key.split("@");

                Mappings map = mappings.get(key);

                for (String aMap : map.asStrings()) {
                    try {
                        removeMapping(args[0], args[1], aMap);
                    } catch (IllegalArgumentException e) {
                        e.printStackTrace();
                    }
                }
            }
        }

        LifecycleUtil.dispose(virtualUserTable);

    }

    @Test
    public void testStoreAndRetrieveRegexMapping() throws
            org.apache.james.rrt.api.RecipientRewriteTable.ErrorMappingException, RecipientRewriteTableException {

        String user = "test";
        String domain = "localhost";
        // String regex = "(.*):{$1}@localhost";
        // String regex2 = "(.+):{$1}@test";
        String regex = "(.*)@localhost";
        String regex2 = "(.+)@test";
        String invalidRegex = ".*):";
        boolean catched = false;

        try {

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();

            assertThat(addMapping(user, domain, regex, REGEX_TYPE)).describedAs("Added virtual mapping").isTrue();
            assertThat(addMapping(user, domain, regex2, REGEX_TYPE)).describedAs("Added virtual mapping").isTrue();
            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("Two mappings").hasSize(2);
            assertThat(virtualUserTable.getAllMappings()).describedAs("One mappingline").hasSize(1);
            assertThat(removeMapping(user, domain, regex, REGEX_TYPE)).describedAs("remove virtual mapping").isTrue();
            
            try {
                virtualUserTable.addRegexMapping(user, domain, invalidRegex);
            } catch (RecipientRewriteTableException e) {
                catched = true;
            }
            
            assertThat(catched).describedAs("Invalid Mapping throw exception").isTrue();
            assertThat(removeMapping(user, domain, regex2, REGEX_TYPE)).describedAs("remove virtual mapping").isTrue();

            
            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();
            assertThat(virtualUserTable.getAllMappings()).describedAs("No mapping").isNull();

        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            fail("Storing failed");
        }

    }

    @Test
    public void testStoreAndRetrieveAddressMapping() throws ErrorMappingException, RecipientRewriteTableException {

        String user = "test";
        String domain = "localhost";
        String address = "test@localhost2";
        String address2 = "test@james";

        try {

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();

            assertThat(addMapping(user, domain, address, ADDRESS_TYPE)).describedAs("Added virtual mapping").isTrue();
            assertThat(addMapping(user, domain, address2, ADDRESS_TYPE)).describedAs("Added virtual mapping").isTrue();

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("Two mappings").hasSize(2);
            assertThat(virtualUserTable.getAllMappings()).describedAs("One mappingline").hasSize(1);

            assertThat(removeMapping(user, domain, address, ADDRESS_TYPE)).describedAs("remove virtual mapping").isTrue();

            /*
             * TEMPORARILY REMOVE JDBC specific test String invalidAddress=
             * ".*@localhost2:"; boolean catched = false; if (virtualUserTable
             * instanceof JDBCRecipientRewriteTable) { try {
             * assertTrue("Added virtual mapping", addMapping(user, domain,
             * invalidAddress, ADDRESS_TYPE)); } catch (InvalidMappingException
             * e) { catched = true; }
             * assertTrue("Invalid Mapping throw exception" , catched); }
             */

            
            assertThat(removeMapping(user, domain, address2, ADDRESS_TYPE)).describedAs("remove virtual mapping").isTrue();

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();
            assertThat(virtualUserTable.getAllMappings()).describedAs("No mapping").isNull();

        } catch (IllegalArgumentException e) {
            fail("Storing failed");
        }

    }

    @Test
    public void testStoreAndRetrieveErrorMapping() throws ErrorMappingException, RecipientRewriteTableException {

        String user = "test";
        String domain = "localhost";
        String error = "bounce!";
        boolean catched = false;

        try {
            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();

            assertThat(addMapping(user, domain, error, ERROR_TYPE)).describedAs("Added virtual mapping").isTrue();
            assertThat(virtualUserTable.getAllMappings()).describedAs("One mappingline").hasSize(1);

            try {
                virtualUserTable.getMappings(user, domain);
            } catch (ErrorMappingException e) {
                catched = true;
            }
            assertThat(catched).describedAs("Error Mapping throw exception").isTrue();

            assertThat(removeMapping(user, domain, error, ERROR_TYPE)).describedAs("remove virtual mapping").isTrue();

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();
            assertThat(virtualUserTable.getAllMappings()).describedAs("No mapping").isNull();
        } catch (IllegalArgumentException e) {
            fail("Storing failed");
        }

    }

    @Test
    public void testStoreAndRetrieveWildCardAddressMapping() throws ErrorMappingException,
            RecipientRewriteTableException {

        String user = "test";
        String user2 = "test2";
        String domain = "localhost";
        String address = "test@localhost2";
        String address2 = "test@james";

        try {

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();

            assertThat(addMapping(RecipientRewriteTable.WILDCARD, domain, address, ADDRESS_TYPE)).describedAs("Added virtual mapping").isTrue();
            assertThat(addMapping(user, domain, address2, ADDRESS_TYPE)).describedAs("Added virtual mapping").isTrue();

            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("One mappings").hasSize(1);
            assertThat(virtualUserTable.getMappings(user2, domain)).describedAs("One mappings").hasSize(1);

            assertThat(removeMapping(user, domain, address2, ADDRESS_TYPE)).describedAs("remove virtual mapping").isTrue();
            assertThat(removeMapping(RecipientRewriteTable.WILDCARD, domain, address, ADDRESS_TYPE)).describedAs("remove virtual mapping").isTrue();
            
            assertThat(virtualUserTable.getMappings(user, domain)).describedAs("No mapping").isNull();
            assertThat(virtualUserTable.getMappings(user2, domain)).describedAs("No mapping").isNull();

        } catch (IllegalArgumentException e) {
            fail("Storing failed");
        }

    }

    @Test
    public void testRecursiveMapping() throws ErrorMappingException, RecipientRewriteTableException {

        String user1 = "user1";
        String user2 = "user2";
        String user3 = "user3";
        String domain1 = "domain1";
        String domain2 = "domain2";
        String domain3 = "domain3";
        boolean exception1 = false;

        virtualUserTable.setRecursiveMapping(true);

        try {
            assertThat(virtualUserTable.getAllMappings()).describedAs("No mapping").isNull();

            assertThat(addMapping(user1, domain1, user2 + "@" + domain2, ADDRESS_TYPE)).describedAs("Added mapping").isTrue();
            assertThat(addMapping(user2, domain2, user3 + "@" + domain3, ADDRESS_TYPE)).describedAs("Added mapping").isTrue();
            assertThat(virtualUserTable.getMappings(user1, domain1)).containsOnly(MappingImpl.address(user3 + "@" + domain3));
            assertThat(addMapping(user3, domain3, user1 + "@" + domain1, ADDRESS_TYPE)).describedAs("Added mapping").isTrue();
            
            try {
                virtualUserTable.getMappings(user1, domain1);
            } catch (ErrorMappingException e) {
                exception1 = true;
            }
            assertThat(exception1).describedAs("Exception thrown on to many mappings").isTrue();

            // disable recursive mapping
            virtualUserTable.setRecursiveMapping(false);
            assertThat(virtualUserTable.getMappings(user1, domain1)).describedAs("Not recursive mapped").containsExactly(MappingImpl.address(user2 + "@" + domain2));

        } catch (IllegalArgumentException e) {
            fail("Storing failed");
        }
    }

    @Test
    public void testAliasDomainMapping() throws ErrorMappingException, RecipientRewriteTableException {

        String domain = "realdomain";
        String aliasDomain = "aliasdomain";
        String user = "user";
        String user2 = "user2";

        assertThat(virtualUserTable.getAllMappings()).describedAs("No mappings").isNull();

        try {

            assertThat(addMapping(RecipientRewriteTable.WILDCARD, aliasDomain, user2 + "@" + domain,
                    ADDRESS_TYPE)).describedAs("Add mapping").isTrue();
            assertThat(addMapping(RecipientRewriteTable.WILDCARD, aliasDomain, domain,
                    ALIASDOMAIN_TYPE)).describedAs("Add aliasDomain mapping").isTrue();

            assertThat(virtualUserTable.getMappings(user, aliasDomain))
                .describedAs("Domain mapped as first, Address mapped as second")
                .containsExactly(MappingImpl.address(user + "@" + domain), MappingImpl.address(user2 + "@" + domain));

            assertThat(removeMapping(RecipientRewriteTable.WILDCARD, aliasDomain, user2 + "@" + domain,
                    ADDRESS_TYPE)).describedAs("Remove mapping").isTrue();

            assertThat(removeMapping(RecipientRewriteTable.WILDCARD, aliasDomain, domain,
                    ALIASDOMAIN_TYPE)).describedAs("Remove aliasDomain mapping").isTrue();

        } catch (IllegalArgumentException e) {
            fail("Storing failed");
        }

    }

    @Test
    public void sortMappingsShouldReturnEmptyWhenEmpty() {
        assertThat(AbstractRecipientRewriteTable.sortMappings(MappingsImpl.empty())).isEmpty();
    }

    @Test
    public void sortMappingsShouldReturnSameStringWhenSingleDomainAlias() {
        String singleDomainAlias = RecipientRewriteTable.ALIASDOMAIN_PREFIX + "first";
        assertThat(AbstractRecipientRewriteTable.sortMappings(MappingsImpl.fromRawString(singleDomainAlias))).containsExactly(MappingImpl.domain("first"));
    }
     
    @Test
    public void sortMappingsShouldReturnSameStringWhenTwoDomainAliases() {
        MappingsImpl mappings = MappingsImpl.builder()
                .add(RecipientRewriteTable.ALIASDOMAIN_PREFIX + "first")
                .add(RecipientRewriteTable.ALIASDOMAIN_PREFIX + "second")
                .build();
        assertThat(AbstractRecipientRewriteTable.sortMappings(mappings)).isEqualTo(mappings);
    }
    
    @Test
    public void sortMappingsShouldPutDomainAliasFirstWhenVariousMappings() {
        String regexMapping = RecipientRewriteTable.REGEX_PREFIX + "first";
        String domainMapping = RecipientRewriteTable.ALIASDOMAIN_PREFIX + "second";
        MappingsImpl mappings = MappingsImpl.builder()
                .add(regexMapping)
                .add(domainMapping)
                .build();
        assertThat(AbstractRecipientRewriteTable.sortMappings(mappings))
                .isEqualTo(MappingsImpl.builder()
                        .add(domainMapping)
                        .add(regexMapping)
                        .build());
    }


    protected abstract AbstractRecipientRewriteTable getRecipientRewriteTable() throws Exception;

    protected abstract boolean addMapping(String user, String domain, String mapping, int type) throws
            RecipientRewriteTableException;

    protected abstract boolean removeMapping(String user, String domain, String mapping, int type) throws
            RecipientRewriteTableException;

    private void removeMapping(String user, String domain, String rawMapping) throws RecipientRewriteTableException {
        if (rawMapping.startsWith(RecipientRewriteTable.ERROR_PREFIX)) {
            removeMapping(user, domain, rawMapping.substring(RecipientRewriteTable.ERROR_PREFIX.length()), ERROR_TYPE);
        } else if (rawMapping.startsWith(RecipientRewriteTable.REGEX_PREFIX)) {
            removeMapping(user, domain, rawMapping.substring(RecipientRewriteTable.REGEX_PREFIX.length()), REGEX_TYPE);
        } else if (rawMapping.startsWith(RecipientRewriteTable.ALIASDOMAIN_PREFIX)) {
            removeMapping(user, domain, rawMapping.substring(RecipientRewriteTable.ALIASDOMAIN_PREFIX.length()),
                    ALIASDOMAIN_TYPE);
        } else {
            removeMapping(user, domain, rawMapping, ADDRESS_TYPE);
        }
    }
}
