/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/
package org.apache.james.rrt.file;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.james.rrt.api.RecipientRewriteTableException;
import org.apache.james.rrt.lib.AbstractRecipientRewriteTable;
import org.apache.james.rrt.lib.RecipientRewriteTableUtil;

/**
 * Class responsible to implement the Virtual User Table in XML disk file.
 */
public class XMLRecipientRewriteTable extends AbstractRecipientRewriteTable {

    /**
     * Holds the configured mappings
     */
    private Map<String, String> mappings;

    /**
     * @see org.apache.james.rrt.lib.AbstractRecipientRewriteTable#doConfigure(org.apache.commons.configuration.HierarchicalConfiguration)
     */
    @SuppressWarnings("unchecked")
    protected void doConfigure(HierarchicalConfiguration arg0) throws ConfigurationException {

        List<String> mapConf = arg0.getList("mapping");

        mappings = new HashMap<String, String>();

        if (mapConf != null && mapConf.size() > 0) {
            for (int i = 0; i < mapConf.size(); i++) {
                mappings.putAll(RecipientRewriteTableUtil.getXMLMappings(mapConf.get(i)));
            }
        } else {
            throw new ConfigurationException("No mapping configured");
        }

    }

    /**
     * @see org.apache.james.rrt.lib.AbstractRecipientRewriteTable#mapAddressInternal(java.lang.String,
     *      java.lang.String)
     */
    protected String mapAddressInternal(String user, String domain) throws RecipientRewriteTableException {
        if (mappings == null) {
            return null;
        } else {
            return RecipientRewriteTableUtil.getTargetString(user, domain, mappings);
        }
    }

    /**
     * @see org.apache.james.rrt.lib.AbstractRecipientRewriteTable#getUserDomainMappingsInternal(java.lang.String,
     *      java.lang.String)
     */
    protected Collection<String> getUserDomainMappingsInternal(String user, String domain) throws RecipientRewriteTableException {
        if (mappings == null) {
            return null;
        } else {
            String maps = mappings.get(user + "@" + domain);
            if (maps != null) {
                return RecipientRewriteTableUtil.mappingToCollection(maps);
            } else {
                return null;
            }
        }
    }

    /**
     * @see org.apache.james.rrt.lib.AbstractRecipientRewriteTable#getAllMappingsInternal()
     */
    protected Map<String, Collection<String>> getAllMappingsInternal() throws RecipientRewriteTableException {
        if (mappings != null && mappings.size() > 0) {
            Map<String, Collection<String>> mappingsNew = new HashMap<String, Collection<String>>();
            Iterator<String> maps = mappings.keySet().iterator();
            while (maps.hasNext()) {
                String key = maps.next();
                mappingsNew.put(key, RecipientRewriteTableUtil.mappingToCollection(mappings.get(key).toString()));
            }
            return mappingsNew;
        } else {
            return null;
        }
    }

    @Override
    protected void addMappingInternal(String user, String domain, String mapping) throws RecipientRewriteTableException {
        throw new RecipientRewriteTableException("Read-Only implementation");
    }

    @Override
    protected void removeMappingInternal(String user, String domain, String mapping) throws RecipientRewriteTableException {
        throw new RecipientRewriteTableException("Read-Only implementation");
    }

}
