
package org.apache.iotdb.db.queryengine.execution.aggregation;

import com.google.common.collect.ImmutableList;
import org.apache.commons.collections4.comparators.ComparatorChain;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.file.metadata.statistics.Statistics;
import org.apache.iotdb.tsfile.read.common.block.column.Column;
import org.apache.iotdb.tsfile.read.common.block.column.ColumnBuilder;
import org.apache.iotdb.tsfile.utils.Binary;
import org.apache.iotdb.tsfile.utils.BitMap;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.utils.ReadWriteIOUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;

/*
* This class is generated using freemarker and the ModeAccumulator.ftl template.
*/
@SuppressWarnings("unused")
public class BooleanModeAccumulator implements Accumulator {
  // pair left records count of element, pair right records min time of element
  private final Map<Boolean, Pair<Long, Long>> countMap = new HashMap<>();

  @Override
  public void addInput(Column[] column, BitMap bitMap, int lastIndex) {
    for (int i = 0; i <= lastIndex; i++) {
      if (bitMap != null && !bitMap.isMarked(i)) {
        continue;
      }
      if (!column[1].isNull(i)) {
          final long time = column[0].getLong(i);
        countMap.compute(
            column[1].getBoolean(i),
            (k, v) ->
                v == null ? new Pair<>(1L, time) : new Pair<>(v.left + 1, Math.min(v.right, time)));
      }
    }
  }

  @Override
  public void addIntermediate(Column[] partialResult) {
    checkArgument(partialResult.length == 1, "partialResult of Mode should be 1");
    checkArgument(!partialResult[0].isNull(0), "partialResult of Mode should not be null");
    deserializeAndMergeCountMap(partialResult[0].getBinary(0));
  }

  @Override
  public void addStatistics(Statistics statistics) {
    throw new UnsupportedOperationException(getClass().getName());
  }

  @Override
  public void setFinal(Column finalResult) {
    if (finalResult.isNull(0)) {
      return;
    }

    // Step of ModeAccumulator is STATIC,
    // countMap only need to record one entry which key is finalResult
    countMap.put(finalResult.getBoolean(0), new Pair<>(0L, Long.MIN_VALUE));
  }

  @Override
  public void outputIntermediate(ColumnBuilder[] tsBlockBuilder) {
    tsBlockBuilder[0].writeBinary(serializeCountMap());
  }

  @Override
  public void outputFinal(ColumnBuilder tsBlockBuilder) {
    if (countMap.isEmpty()) {
      tsBlockBuilder.appendNull();
    } else {
      tsBlockBuilder.writeBoolean(
          Collections.max(
                  countMap.entrySet(),
                  Map.Entry.comparingByValue(
                      new ComparatorChain<>(
                          ImmutableList.of(
                              (v1, v2) -> v1.left.compareTo(v2.left),
                              (v1, v2) -> v2.right.compareTo(v1.right)))))
              .getKey());
    }
  }

  @Override
  public void reset() {
    countMap.clear();
  }

  @Override
  public boolean hasFinalResult() {
    return false;
  }

  @Override
  public TSDataType[] getIntermediateType() {
    return new TSDataType[] {TSDataType.TEXT};
  }

  @Override
  public TSDataType getFinalType() {
    return TSDataType.BOOLEAN;
  }

  private Binary serializeCountMap() {
    ByteArrayOutputStream stream = new ByteArrayOutputStream();
    try {
      ReadWriteIOUtils.write(countMap.size(), stream);
      for (Map.Entry<Boolean, Pair<Long, Long>> entry : countMap.entrySet()) {
        ReadWriteIOUtils.write(entry.getKey(), stream);
        ReadWriteIOUtils.write(entry.getValue().left, stream);
        ReadWriteIOUtils.write(entry.getValue().right, stream);
      }
    } catch (IOException e) {
      // Totally memory operation. This case won't happen.
    }
    return new Binary(stream.toByteArray());
  }

  private void deserializeAndMergeCountMap(Binary partialResult) {
    InputStream stream = new ByteArrayInputStream(partialResult.getValues());
    try {
      int size = ReadWriteIOUtils.readInt(stream);
      for (int i = 0; i < size; i++) {
        countMap.compute(
            ReadWriteIOUtils.readBoolean(stream),
            (k, v) -> {
              try {
                return v == null
                    ? new Pair<>(
                        ReadWriteIOUtils.readLong(stream), ReadWriteIOUtils.readLong(stream))
                    : new Pair<>(
                        v.left + ReadWriteIOUtils.readLong(stream),
                        Math.min(v.right, ReadWriteIOUtils.readLong(stream)));
              } catch (IOException e) {
                throw new RuntimeException(e);
              }
            });
      }
    } catch (IOException e) {
      // Totally memory operation. This case won't happen.
    }
  }
}


