/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.queryengine.execution.memory;

import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.lang3.Validate;
import org.apache.iotdb.db.exception.runtime.MemoryLeakException;
import org.apache.iotdb.tsfile.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryPool {
    private static final Logger LOGGER = LoggerFactory.getLogger(MemoryPool.class);
    private final String id;
    private final long maxBytes;
    private final long maxBytesPerFragmentInstance;
    private final AtomicLong remainingBytes;
    private final Map<String, Map<String, Map<String, Long>>> queryMemoryReservations = new ConcurrentHashMap<String, Map<String, Map<String, Long>>>();
    private final Queue<MemoryReservationFuture<Void>> memoryReservationFutures = new ConcurrentLinkedQueue<MemoryReservationFuture<Void>>();

    public MemoryPool(String id, long maxBytes, long maxBytesPerFragmentInstance) {
        this.id = (String)Validate.notNull((Object)id);
        Validate.isTrue((maxBytes > 0L ? 1 : 0) != 0, (String)"max bytes should be greater than zero: %d", (long)maxBytes);
        this.maxBytes = maxBytes;
        Validate.isTrue((maxBytesPerFragmentInstance > 0L && maxBytesPerFragmentInstance <= maxBytes ? 1 : 0) != 0, (String)"max bytes per FI should be in (0,maxBytes]. maxBytesPerFI: %d, maxBytes: %d", (Object[])new Object[]{maxBytesPerFragmentInstance, maxBytes});
        this.maxBytesPerFragmentInstance = maxBytesPerFragmentInstance;
        this.remainingBytes = new AtomicLong(maxBytes);
    }

    public String getId() {
        return this.id;
    }

    public long getMaxBytes() {
        return this.maxBytes;
    }

    public long getRemainingBytes() {
        return this.remainingBytes.get();
    }

    public int getQueryMemoryReservationSize() {
        return this.queryMemoryReservations.size();
    }

    public int getMemoryReservationSize() {
        return this.memoryReservationFutures.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerPlanNodeIdToQueryMemoryMap(String queryId, String fragmentInstanceId, String planNodeId) {
        Map<String, Map<String, Map<String, Long>>> map = this.queryMemoryReservations;
        synchronized (map) {
            this.queryMemoryReservations.computeIfAbsent(queryId, x -> new ConcurrentHashMap()).computeIfAbsent(fragmentInstanceId, x -> new ConcurrentHashMap()).putIfAbsent(planNodeId, 0L);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deRegisterFragmentInstanceFromQueryMemoryMap(String queryId, String fragmentInstanceId) {
        Map<String, Map<String, Long>> queryRelatedMemory = this.queryMemoryReservations.get(queryId);
        if (queryRelatedMemory != null) {
            Map<String, Long> fragmentRelatedMemory = queryRelatedMemory.get(fragmentInstanceId);
            boolean hasPotentialMemoryLeak = false;
            if (fragmentRelatedMemory != null) {
                hasPotentialMemoryLeak = fragmentRelatedMemory.values().stream().anyMatch(value -> value != 0L);
            }
            Map<String, Map<String, Map<String, Long>>> map = this.queryMemoryReservations;
            synchronized (map) {
                queryRelatedMemory.remove(fragmentInstanceId);
                if (queryRelatedMemory.isEmpty()) {
                    this.queryMemoryReservations.remove(queryId);
                }
            }
            if (hasPotentialMemoryLeak) {
                List invalidEntryList = fragmentRelatedMemory.entrySet().stream().filter(entry -> (Long)entry.getValue() != 0L).collect(Collectors.toList());
                throw new MemoryLeakException(String.format("PlanNode related memory is not zero when trying to deregister FI from query memory pool. QueryId is : %s, FragmentInstanceId is : %s, Non-zero PlanNode related memory is : %s.", queryId, fragmentInstanceId, invalidEntryList));
            }
        }
    }

    public Pair<ListenableFuture<Void>, Boolean> reserve(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve, long maxBytesCanReserve) {
        Validate.notNull((Object)queryId);
        Validate.notNull((Object)fragmentInstanceId);
        Validate.notNull((Object)planNodeId);
        Validate.isTrue((bytesToReserve > 0L && bytesToReserve <= this.maxBytesPerFragmentInstance ? 1 : 0) != 0, (String)"bytesToReserve should be in (0,maxBytesPerFI]. maxBytesPerFI: %d", (long)bytesToReserve);
        if (bytesToReserve > maxBytesCanReserve) {
            LOGGER.warn("Cannot reserve {}(Max: {}) bytes memory from MemoryPool for planNodeId{}", new Object[]{bytesToReserve, maxBytesCanReserve, planNodeId});
            throw new IllegalArgumentException("Query is aborted since it requests more memory than can be allocated.");
        }
        if (this.tryReserve(queryId, fragmentInstanceId, planNodeId, bytesToReserve, maxBytesCanReserve)) {
            ListenableFuture result = Futures.immediateFuture(null);
            return new Pair((Object)result, (Object)Boolean.TRUE);
        }
        LOGGER.debug("Blocked reserve request: {} bytes memory for planNodeId{}", (Object)bytesToReserve, (Object)planNodeId);
        this.rollbackReserve(queryId, fragmentInstanceId, planNodeId, bytesToReserve);
        MemoryReservationFuture result = MemoryReservationFuture.create(queryId, fragmentInstanceId, planNodeId, bytesToReserve, maxBytesCanReserve);
        this.memoryReservationFutures.add(result);
        return new Pair(result, (Object)Boolean.FALSE);
    }

    public boolean tryReserveForTest(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve, long maxBytesCanReserve) {
        Validate.notNull((Object)queryId);
        Validate.notNull((Object)fragmentInstanceId);
        Validate.notNull((Object)planNodeId);
        Validate.isTrue((bytesToReserve > 0L && bytesToReserve <= this.maxBytesPerFragmentInstance ? 1 : 0) != 0, (String)"bytesToReserve should be in (0,maxBytesPerFI]. maxBytesPerFI: %d", (long)bytesToReserve);
        if (this.tryReserve(queryId, fragmentInstanceId, planNodeId, bytesToReserve, maxBytesCanReserve)) {
            return true;
        }
        this.rollbackReserve(queryId, fragmentInstanceId, planNodeId, bytesToReserve);
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized long tryCancel(ListenableFuture<Void> future) {
        ListenableFuture<Void> listenableFuture = future;
        synchronized (listenableFuture) {
            Validate.notNull(future);
            if (future.isDone()) {
                return 0L;
            }
            Validate.isTrue((boolean)(future instanceof MemoryReservationFuture), (String)("invalid future type " + future.getClass().getSimpleName()), (Object[])new Object[0]);
            future.cancel(true);
        }
        return ((MemoryReservationFuture)future).getBytesToReserve();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void free(String queryId, String fragmentInstanceId, String planNodeId, long bytes) {
        Validate.notNull((Object)queryId);
        Validate.isTrue((bytes > 0L ? 1 : 0) != 0);
        try {
            this.queryMemoryReservations.get(queryId).get(fragmentInstanceId).computeIfPresent(planNodeId, (k, reservedMemory) -> {
                if (reservedMemory < bytes) {
                    throw new IllegalArgumentException("Free more memory than has been reserved.");
                }
                return reservedMemory - bytes;
            });
        }
        catch (NullPointerException e) {
            throw new IllegalArgumentException("RelatedMemoryReserved can't be null when freeing memory");
        }
        this.remainingBytes.addAndGet(bytes);
        if (this.memoryReservationFutures.isEmpty()) {
            return;
        }
        Iterator iterator = this.memoryReservationFutures.iterator();
        while (iterator.hasNext()) {
            MemoryReservationFuture future;
            MemoryReservationFuture memoryReservationFuture = future = (MemoryReservationFuture)((Object)iterator.next());
            synchronized (memoryReservationFuture) {
                long maxBytesCanReserve;
                String curPlanNodeId;
                String curFragmentInstanceId;
                if (future.isCancelled() || future.isDone()) {
                    iterator.remove();
                    continue;
                }
                long bytesToReserve = future.getBytesToReserve();
                String curQueryId = future.getQueryId();
                if (this.tryReserve(curQueryId, curFragmentInstanceId = future.getFragmentInstanceId(), curPlanNodeId = future.getPlanNodeId(), bytesToReserve, maxBytesCanReserve = future.getMaxBytesCanReserve())) {
                    future.set(null);
                    iterator.remove();
                } else {
                    this.rollbackReserve(curQueryId, curFragmentInstanceId, curPlanNodeId, bytesToReserve);
                }
            }
        }
    }

    public long getQueryMemoryReservedBytes(String queryId) {
        if (!this.queryMemoryReservations.containsKey(queryId)) {
            return 0L;
        }
        long sum = 0L;
        for (Map<String, Long> map : this.queryMemoryReservations.get(queryId).values()) {
            sum += map.values().stream().reduce(0L, Long::sum).longValue();
        }
        return sum;
    }

    public long getReservedBytes() {
        return this.maxBytes - this.remainingBytes.get();
    }

    public boolean tryReserve(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve, long maxBytesCanReserve) {
        long tryRemainingBytes = this.remainingBytes.addAndGet(-bytesToReserve);
        long queryRemainingBytes = maxBytesCanReserve - this.queryMemoryReservations.get(queryId).get(fragmentInstanceId).merge(planNodeId, bytesToReserve, Long::sum);
        return tryRemainingBytes >= 0L && queryRemainingBytes >= 0L;
    }

    private void rollbackReserve(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve) {
        this.queryMemoryReservations.get(queryId).get(fragmentInstanceId).merge(planNodeId, -bytesToReserve, Long::sum);
        this.remainingBytes.addAndGet(bytesToReserve);
    }

    public static class MemoryReservationFuture<V>
    extends AbstractFuture<V> {
        private final String queryId;
        private final String fragmentInstanceId;
        private final String planNodeId;
        private final long bytesToReserve;
        private final long maxBytesCanReserve;

        private MemoryReservationFuture(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve, long maxBytesCanReserve) {
            this.queryId = (String)Validate.notNull((Object)queryId, (String)"queryId cannot be null", (Object[])new Object[0]);
            this.fragmentInstanceId = (String)Validate.notNull((Object)fragmentInstanceId, (String)"fragmentInstanceId cannot be null", (Object[])new Object[0]);
            this.planNodeId = (String)Validate.notNull((Object)planNodeId, (String)"planNodeId cannot be null", (Object[])new Object[0]);
            Validate.isTrue((bytesToReserve > 0L ? 1 : 0) != 0, (String)"bytesToReserve should be greater than zero.", (Object[])new Object[0]);
            Validate.isTrue((maxBytesCanReserve > 0L ? 1 : 0) != 0, (String)"maxBytesCanReserve should be greater than zero.", (Object[])new Object[0]);
            this.bytesToReserve = bytesToReserve;
            this.maxBytesCanReserve = maxBytesCanReserve;
        }

        public String getQueryId() {
            return this.queryId;
        }

        public String getFragmentInstanceId() {
            return this.fragmentInstanceId;
        }

        public String getPlanNodeId() {
            return this.planNodeId;
        }

        public long getBytesToReserve() {
            return this.bytesToReserve;
        }

        public long getMaxBytesCanReserve() {
            return this.maxBytesCanReserve;
        }

        public static <V> MemoryReservationFuture<V> create(String queryId, String fragmentInstanceId, String planNodeId, long bytesToReserve, long maxBytesCanReserve) {
            return new MemoryReservationFuture<V>(queryId, fragmentInstanceId, planNodeId, bytesToReserve, maxBytesCanReserve);
        }

        public boolean set(@Nullable V value) {
            return super.set(value);
        }
    }
}

