/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.consensus.ratis;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.concurrent.threadpool.ScheduledExecutorUtil;
import org.apache.iotdb.consensus.config.RatisConfig;
import org.apache.iotdb.consensus.exception.ConsensusException;
import org.apache.iotdb.consensus.ratis.RatisConsensus;
import org.apache.iotdb.consensus.ratis.utils.Utils;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.TimeDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DiskGuardian {
    private static final Logger logger = LoggerFactory.getLogger(DiskGuardian.class);
    private final MemoizedSupplier<RatisConsensus> serverRef;
    private final Map<RaftGroupId, AtomicBoolean> snapshotFlag = new ConcurrentHashMap<RaftGroupId, AtomicBoolean>();
    private final Map<RaftGroupId, RaftLogSummary> bookkeeper = new ConcurrentHashMap<RaftGroupId, RaftLogSummary>();
    private final Map<TimeDuration, List<Predicate<RaftLogSummary>>> snapshotArbitrators = new HashMap<TimeDuration, List<Predicate<RaftLogSummary>>>();
    private final ScheduledExecutorService workerThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)ThreadName.RATIS_BG_DISK_GUARDIAN.getName());
    private final AtomicBoolean isStopped = new AtomicBoolean(false);
    private final long daemonIntervalMs;

    DiskGuardian(Supplier<RatisConsensus> server, RatisConfig config) {
        this.serverRef = MemoizedSupplier.valueOf(server);
        this.daemonIntervalMs = config.getImpl().getCheckAndTakeSnapshotInterval();
    }

    void start() {
        ScheduledExecutorUtil.safelyScheduleWithFixedDelay((ScheduledExecutorService)this.workerThread, this::snapshotDaemon, (long)0L, (long)this.daemonIntervalMs, (TimeUnit)TimeUnit.SECONDS);
        this.snapshotArbitrators.forEach((interval, checkers) -> ScheduledExecutorUtil.safelyScheduleWithFixedDelay((ScheduledExecutorService)this.workerThread, () -> this.checkerDaemon((List<Predicate<RaftLogSummary>>)checkers), (long)0L, (long)interval.toLong(TimeUnit.SECONDS), (TimeUnit)TimeUnit.SECONDS));
    }

    void stop() throws InterruptedException {
        if (this.isStopped.compareAndSet(true, false)) {
            this.workerThread.shutdown();
            this.workerThread.awaitTermination(5L, TimeUnit.SECONDS);
        }
    }

    void registerChecker(Predicate<RaftLogSummary> checker, TimeDuration interval) {
        List checkers = this.snapshotArbitrators.computeIfAbsent(interval, i -> new CopyOnWriteArrayList());
        checkers.add(checker);
    }

    private void snapshotDaemon() {
        if (this.isStopped.get()) {
            return;
        }
        for (RaftGroupId groupId : ((RatisConsensus)this.serverRef.get()).getServer().getGroupIds()) {
            if (!this.getSnapshotFlag(groupId).get()) continue;
            try {
                ((RatisConsensus)this.serverRef.get()).triggerSnapshot(Utils.fromRaftGroupIdToConsensusGroupId(groupId));
                boolean flagCleared = this.snapshotFlag.get(groupId).compareAndSet(true, false);
                if (flagCleared) continue;
                logger.warn("{}: clear snapshot flag failed for group {}, please check the related implementation", (Object)this, (Object)groupId);
            }
            catch (ConsensusException e) {
                logger.warn("{} take snapshot failed for group {} due to {}. Disk file status {}", new Object[]{this, groupId, e, this.getLatestSummary(groupId).orElse(null)});
            }
        }
    }

    private void checkerDaemon(List<Predicate<RaftLogSummary>> checkerList) {
        if (this.isStopped.get()) {
            return;
        }
        for (RaftGroupId groupId : ((RatisConsensus)this.serverRef.get()).getServer().getGroupIds()) {
            Optional<Boolean> anyCheckerPositive;
            Optional<RaftLogSummary> summary = this.getLatestSummary(groupId);
            if (!summary.isPresent() || !(anyCheckerPositive = checkerList.stream().map(checker -> checker.test((RaftLogSummary)summary.get())).filter(Boolean::booleanValue).findAny()).isPresent()) continue;
            this.getSnapshotFlag(groupId).set(true);
        }
    }

    private AtomicBoolean getSnapshotFlag(RaftGroupId groupId) {
        return this.snapshotFlag.computeIfAbsent(groupId, id -> new AtomicBoolean(false));
    }

    private Optional<RaftLogSummary> getLatestSummary(RaftGroupId groupId) {
        RaftLogSummary summary = this.bookkeeper.computeIfAbsent(groupId, gid -> {
            try {
                File root = ((RatisConsensus)this.serverRef.get()).getServer().getDivision(groupId).getRaftStorage().getStorageDir().getCurrentDir();
                return new RaftLogSummary((RaftGroupId)gid, root);
            }
            catch (IOException e) {
                logger.warn("{}: group not exists for {} and caught exception ", new Object[]{this, groupId, e});
                return null;
            }
        });
        if (summary != null) {
            summary.updateNow();
        }
        return Optional.ofNullable(summary);
    }

    static final class RaftLogSummary {
        private static final Predicate<Path> isOpenSegment = p -> p.toFile().getName().startsWith("log_inprogress");
        private final RaftGroupId gid;
        private final File raftLogStorageRoot;
        private long totalSize;
        private Set<Path> logFiles;

        private RaftLogSummary(RaftGroupId gid, File raftLogStorageRoot) {
            this.gid = gid;
            this.raftLogStorageRoot = raftLogStorageRoot;
            this.totalSize = 0L;
            this.logFiles = Collections.emptySet();
        }

        public long getTotalSize() {
            return this.totalSize;
        }

        void updateNow() {
            try (Stream<Path> files = Files.list(this.raftLogStorageRoot.toPath());){
                Set<Path> latest = files.filter(isOpenSegment).collect(Collectors.toSet());
                this.totalSize += RaftLogSummary.diff(this.logFiles, latest);
                this.logFiles = latest;
            }
            catch (IOException e) {
                logger.warn("{}: Error caught when listing files for {} at {}:", new Object[]{this, this.gid, e});
            }
        }

        private static long diff(Set<Path> old, Set<Path> latest) {
            long incremental = RaftLogSummary.totalSize(latest.stream().filter(p -> !old.contains(p)));
            long decremental = RaftLogSummary.totalSize(old.stream().filter(p -> !latest.contains(p)));
            return incremental - decremental;
        }

        private static long totalSize(Stream<Path> files) {
            return files.mapToLong(p -> p.toFile().length()).sum();
        }

        public String toString() {
            return String.format("[Raft Log Summary]: group=%s, total size=%d, files=%s", this.gid, this.totalSize, this.logFiles);
        }
    }
}

