/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.compaction;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.iotdb.db.engine.compaction.inner.utils.MultiTsFileDeviceIterator;
import org.apache.iotdb.db.engine.compaction.writer.AbstractCompactionWriter;
import org.apache.iotdb.db.engine.compaction.writer.CrossSpaceCompactionWriter;
import org.apache.iotdb.db.engine.compaction.writer.InnerSpaceCompactionWriter;
import org.apache.iotdb.db.engine.modification.Modification;
import org.apache.iotdb.db.engine.modification.ModificationFile;
import org.apache.iotdb.db.engine.querycontext.QueryDataSource;
import org.apache.iotdb.db.engine.storagegroup.TsFileNameGenerator;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.metadata.IllegalPathException;
import org.apache.iotdb.db.exception.metadata.MetadataException;
import org.apache.iotdb.db.exception.metadata.PathNotExistException;
import org.apache.iotdb.db.metadata.path.AlignedPath;
import org.apache.iotdb.db.metadata.path.MeasurementPath;
import org.apache.iotdb.db.metadata.path.PartialPath;
import org.apache.iotdb.db.query.context.QueryContext;
import org.apache.iotdb.db.query.control.QueryResourceManager;
import org.apache.iotdb.db.query.reader.series.SeriesRawDataBatchReader;
import org.apache.iotdb.db.service.IoTDB;
import org.apache.iotdb.db.utils.QueryUtils;
import org.apache.iotdb.tsfile.exception.write.WriteProcessException;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.fileSystem.FSFactoryProducer;
import org.apache.iotdb.tsfile.read.common.BatchData;
import org.apache.iotdb.tsfile.read.reader.IBatchReader;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.write.schema.IMeasurementSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionUtils {
    private static final Logger logger = LoggerFactory.getLogger((String)"COMPACTION");

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void compact(List<TsFileResource> seqFileResources, List<TsFileResource> unseqFileResources, List<TsFileResource> targetFileResources) throws IOException, MetadataException, StorageEngineException, InterruptedException {
        long queryId = QueryResourceManager.getInstance().assignCompactionQueryId();
        QueryContext queryContext = new QueryContext(queryId);
        QueryDataSource queryDataSource = new QueryDataSource(seqFileResources, unseqFileResources);
        QueryResourceManager.getInstance().getQueryFileManager().addUsedFilesForQuery(queryId, queryDataSource);
        try (AbstractCompactionWriter compactionWriter = CompactionUtils.getCompactionWriter(seqFileResources, unseqFileResources, targetFileResources);){
            MultiTsFileDeviceIterator deviceIterator = new MultiTsFileDeviceIterator(seqFileResources, unseqFileResources);
            while (deviceIterator.hasNextDevice()) {
                CompactionUtils.checkThreadInterrupted(targetFileResources);
                Pair<String, Boolean> deviceInfo = deviceIterator.nextDevice();
                String device = (String)deviceInfo.left;
                boolean isAligned = (Boolean)deviceInfo.right;
                QueryUtils.fillOrderIndexes(queryDataSource, device, true);
                if (isAligned) {
                    CompactionUtils.compactAlignedSeries(device, deviceIterator, compactionWriter, queryContext, queryDataSource);
                    continue;
                }
                CompactionUtils.compactNonAlignedSeries(device, deviceIterator, compactionWriter, queryContext, queryDataSource);
            }
            compactionWriter.endFile();
            CompactionUtils.updatePlanIndexes(targetFileResources, seqFileResources, unseqFileResources);
        }
        finally {
            QueryResourceManager.getInstance().endQuery(queryId);
        }
    }

    private static void compactAlignedSeries(String device, MultiTsFileDeviceIterator deviceIterator, AbstractCompactionWriter compactionWriter, QueryContext queryContext, QueryDataSource queryDataSource) throws IOException, MetadataException {
        MultiTsFileDeviceIterator.AlignedMeasurementIterator alignedMeasurementIterator = deviceIterator.iterateAlignedSeries(device);
        Set<String> allMeasurements = alignedMeasurementIterator.getAllMeasurements();
        ArrayList<IMeasurementSchema> measurementSchemas = new ArrayList<IMeasurementSchema>();
        for (String measurement : allMeasurements) {
            try {
                measurementSchemas.add(IoTDB.metaManager.getSeriesSchema(new PartialPath(device, measurement)));
            }
            catch (PathNotExistException e) {
                logger.info("A deleted path is skipped: {}", (Object)e.getMessage());
            }
        }
        if (measurementSchemas.isEmpty()) {
            return;
        }
        List<String> existedMeasurements = measurementSchemas.stream().map(IMeasurementSchema::getMeasurementId).collect(Collectors.toList());
        IBatchReader dataBatchReader = CompactionUtils.constructReader(device, existedMeasurements, measurementSchemas, allMeasurements, queryContext, queryDataSource, true);
        if (dataBatchReader.hasNextBatch()) {
            compactionWriter.startChunkGroup(device, true);
            compactionWriter.startMeasurement(measurementSchemas);
            CompactionUtils.writeWithReader(compactionWriter, dataBatchReader);
            compactionWriter.endMeasurement();
            compactionWriter.endChunkGroup();
        }
    }

    private static void compactNonAlignedSeries(String device, MultiTsFileDeviceIterator deviceIterator, AbstractCompactionWriter compactionWriter, QueryContext queryContext, QueryDataSource queryDataSource) throws MetadataException, IOException {
        boolean hasStartChunkGroup = false;
        MultiTsFileDeviceIterator.MeasurementIterator measurementIterator = deviceIterator.iterateNotAlignedSeries(device, false);
        Set<String> allMeasurements = measurementIterator.getAllMeasurements();
        for (String measurement : allMeasurements) {
            ArrayList<IMeasurementSchema> measurementSchemas = new ArrayList<IMeasurementSchema>();
            try {
                measurementSchemas.add(IoTDB.metaManager.getSeriesSchema(new PartialPath(device, measurement)));
            }
            catch (PathNotExistException e) {
                logger.info("A deleted path is skipped: {}", (Object)e.getMessage());
                continue;
            }
            IBatchReader dataBatchReader = CompactionUtils.constructReader(device, Collections.singletonList(measurement), measurementSchemas, allMeasurements, queryContext, queryDataSource, false);
            if (!dataBatchReader.hasNextBatch()) continue;
            if (!hasStartChunkGroup) {
                compactionWriter.startChunkGroup(device, false);
                hasStartChunkGroup = true;
            }
            compactionWriter.startMeasurement(measurementSchemas);
            CompactionUtils.writeWithReader(compactionWriter, dataBatchReader);
            compactionWriter.endMeasurement();
        }
        if (hasStartChunkGroup) {
            compactionWriter.endChunkGroup();
        }
    }

    private static void writeWithReader(AbstractCompactionWriter writer, IBatchReader reader) throws IOException {
        while (reader.hasNextBatch()) {
            BatchData batchData = reader.nextBatch();
            while (batchData.hasCurrent()) {
                writer.write(batchData.currentTime(), batchData.currentValue());
                batchData.next();
            }
        }
    }

    private static IBatchReader constructReader(String deviceId, List<String> measurementIds, List<IMeasurementSchema> measurementSchemas, Set<String> allSensors, QueryContext queryContext, QueryDataSource queryDataSource, boolean isAlign) throws IllegalPathException {
        TSDataType tsDataType;
        PartialPath seriesPath;
        if (isAlign) {
            seriesPath = new AlignedPath(deviceId, measurementIds, measurementSchemas);
            tsDataType = TSDataType.VECTOR;
        } else {
            seriesPath = new MeasurementPath(deviceId, measurementIds.get(0), measurementSchemas.get(0));
            tsDataType = measurementSchemas.get(0).getType();
        }
        return new SeriesRawDataBatchReader(seriesPath, allSensors, tsDataType, queryContext, queryDataSource, null, null, null, true);
    }

    private static AbstractCompactionWriter getCompactionWriter(List<TsFileResource> seqFileResources, List<TsFileResource> unseqFileResources, List<TsFileResource> targetFileResources) throws IOException {
        if (!seqFileResources.isEmpty() && !unseqFileResources.isEmpty()) {
            return new CrossSpaceCompactionWriter(targetFileResources, seqFileResources);
        }
        return new InnerSpaceCompactionWriter(targetFileResources.get(0));
    }

    private static void updatePlanIndexes(List<TsFileResource> targetResources, List<TsFileResource> seqResources, List<TsFileResource> unseqResources) {
        for (int i = 0; i < targetResources.size(); ++i) {
            TsFileResource targetResource = targetResources.get(i);
            if (!targetResource.getTsFile().exists()) {
                targetResources.remove(i--);
                continue;
            }
            for (TsFileResource unseqResource : unseqResources) {
                targetResource.updatePlanIndexes(unseqResource);
            }
            for (TsFileResource seqResource : seqResources) {
                targetResource.updatePlanIndexes(seqResource);
            }
        }
    }

    public static void moveTargetFile(List<TsFileResource> targetResources, boolean isInnerSpace, String fullStorageGroupName) throws IOException, WriteProcessException {
        String fileSuffix = isInnerSpace ? ".inner" : ".cross";
        for (TsFileResource targetResource : targetResources) {
            CompactionUtils.moveOneTargetFile(targetResource, fileSuffix, fullStorageGroupName);
        }
    }

    private static void moveOneTargetFile(TsFileResource targetResource, String tmpFileSuffix, String fullStorageGroupName) throws IOException {
        if (!targetResource.getTsFile().exists()) {
            logger.info("{} [Compaction] Tmp target tsfile {} may be deleted after compaction.", (Object)fullStorageGroupName, (Object)targetResource.getTsFilePath());
            return;
        }
        File newFile = new File(targetResource.getTsFilePath().replace(tmpFileSuffix, ".tsfile"));
        if (!newFile.exists()) {
            FSFactoryProducer.getFSFactory().moveFile(targetResource.getTsFile(), newFile);
        }
        targetResource.setFile(newFile);
        targetResource.serialize();
        targetResource.close();
    }

    public static void combineModsInCompaction(List<TsFileResource> seqResources, List<TsFileResource> unseqResources, List<TsFileResource> targetResources) throws IOException {
        HashMap<String, TsFileResource> seqFileInfoMap = new HashMap<String, TsFileResource>();
        for (TsFileResource tsFileResource : seqResources) {
            seqFileInfoMap.put(TsFileNameGenerator.increaseCrossCompactionCnt(tsFileResource.getTsFile()).getName(), tsFileResource);
        }
        for (TsFileResource tsFileResource : targetResources) {
            CompactionUtils.updateOneTargetMods(tsFileResource, (TsFileResource)seqFileInfoMap.get(tsFileResource.getTsFile().getName()), unseqResources);
        }
    }

    private static void updateOneTargetMods(TsFileResource targetFile, TsFileResource seqFile, List<TsFileResource> unseqFiles) throws IOException {
        if (seqFile != null) {
            ModificationFile seqCompactionModificationFile = ModificationFile.getCompactionMods(seqFile);
            for (Modification modification : seqCompactionModificationFile.getModifications()) {
                targetFile.getModFile().write(modification);
            }
        }
        for (TsFileResource unseqFile : unseqFiles) {
            ModificationFile compactionUnseqModificationFile = ModificationFile.getCompactionMods(unseqFile);
            for (Modification modification : compactionUnseqModificationFile.getModifications()) {
                targetFile.getModFile().write(modification);
            }
        }
        targetFile.getModFile().close();
    }

    public static void deleteCompactionModsFile(List<TsFileResource> selectedSeqTsFileResourceList, List<TsFileResource> selectedUnSeqTsFileResourceList) throws IOException {
        ModificationFile modificationFile;
        for (TsFileResource seqFile : selectedSeqTsFileResourceList) {
            modificationFile = seqFile.getCompactionModFile();
            if (!modificationFile.exists()) continue;
            modificationFile.remove();
        }
        for (TsFileResource unseqFile : selectedUnSeqTsFileResourceList) {
            modificationFile = unseqFile.getCompactionModFile();
            if (!modificationFile.exists()) continue;
            modificationFile.remove();
        }
    }

    private static void checkThreadInterrupted(List<TsFileResource> tsFileResource) throws InterruptedException {
        if (Thread.currentThread().isInterrupted()) {
            throw new InterruptedException(String.format("[Compaction] compaction for target file %s abort", tsFileResource.toString()));
        }
    }
}

