/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.query.udf.service;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.exception.StartupException;
import org.apache.iotdb.db.exception.UDFRegistrationException;
import org.apache.iotdb.db.exception.query.QueryProcessException;
import org.apache.iotdb.db.qp.constant.SQLConstant;
import org.apache.iotdb.db.query.udf.api.UDF;
import org.apache.iotdb.db.query.udf.builtin.BuiltinFunction;
import org.apache.iotdb.db.query.udf.core.context.UDFContext;
import org.apache.iotdb.db.query.udf.service.UDFClassLoader;
import org.apache.iotdb.db.query.udf.service.UDFClassLoaderManager;
import org.apache.iotdb.db.query.udf.service.UDFLogWriter;
import org.apache.iotdb.db.query.udf.service.UDFRegistrationInformation;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.ServiceType;
import org.apache.iotdb.tsfile.fileSystem.FSFactoryProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UDFRegistrationService
implements IService {
    private static final Logger logger = LoggerFactory.getLogger(UDFRegistrationService.class);
    private static final String ULOG_FILE_DIR = IoTDBDescriptor.getInstance().getConfig().getSystemDir() + File.separator + "udf" + File.separator;
    private static final String LOG_FILE_NAME = ULOG_FILE_DIR + "ulog.txt";
    private static final String TEMPORARY_LOG_FILE_NAME = LOG_FILE_NAME + ".tmp";
    private final ReentrantLock registrationLock = new ReentrantLock();
    private final ConcurrentHashMap<String, UDFRegistrationInformation> registrationInformation = new ConcurrentHashMap();
    private final ReentrantReadWriteLock logWriterLock = new ReentrantReadWriteLock();
    private UDFLogWriter logWriter;

    private UDFRegistrationService() {
    }

    public void acquireRegistrationLock() {
        this.registrationLock.lock();
    }

    public void releaseRegistrationLock() {
        this.registrationLock.unlock();
    }

    public void register(String functionName, String className, boolean isTemporary, boolean writeToTemporaryLogFile) throws UDFRegistrationException {
        functionName = functionName.toUpperCase();
        UDFRegistrationService.validateFunctionName(functionName, className);
        this.checkIfRegistered(functionName, className, isTemporary);
        this.doRegister(functionName, className, isTemporary);
        this.tryAppendRegistrationLog(functionName, className, isTemporary, writeToTemporaryLogFile);
    }

    private static void validateFunctionName(String functionName, String className) throws UDFRegistrationException {
        if (!SQLConstant.getNativeFunctionNames().contains(functionName.toLowerCase())) {
            return;
        }
        String errorMessage = String.format("Failed to register UDF %s(%s), because the given function name conflicts with the built-in function name", functionName, className);
        logger.warn(errorMessage);
        throw new UDFRegistrationException(errorMessage);
    }

    private void checkIfRegistered(String functionName, String className, boolean isTemporary) throws UDFRegistrationException {
        UDFRegistrationInformation information = this.registrationInformation.get(functionName);
        if (information == null) {
            return;
        }
        String errorMessage = information.isBuiltin() ? String.format("Failed to register UDF %s(%s), because the given function name is the same as a built-in UDF function name.", functionName, className) : (information.getClassName().equals(className) ? String.format("Failed to register %sTEMPORARY UDF %s(%s), because a %sTEMPORARY UDF %s(%s) with the same function name and the class name has already been registered.", isTemporary ? "" : "non-", functionName, className, information.isTemporary() ? "" : "non-", information.getFunctionName(), information.getClassName()) : String.format("Failed to register UDF %s(%s), because a UDF %s(%s) with the same function name but a different class name has already been registered.", functionName, className, information.getFunctionName(), information.getClassName()));
        logger.warn(errorMessage);
        throw new UDFRegistrationException(errorMessage);
    }

    private void doRegister(String functionName, String className, boolean isTemporary) throws UDFRegistrationException {
        this.acquireRegistrationLock();
        try {
            UDFClassLoader currentActiveClassLoader = UDFClassLoaderManager.getInstance().updateAndGetActiveClassLoader();
            this.updateAllRegisteredClasses(currentActiveClassLoader);
            Class<?> functionClass = Class.forName(className, true, currentActiveClassLoader);
            functionClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            this.registrationInformation.put(functionName, new UDFRegistrationInformation(functionName, className, isTemporary, false, functionClass));
        }
        catch (IOException | ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            String errorMessage = String.format("Failed to register UDF %s(%s), because its instance can not be constructed successfully. Exception: %s", functionName, className, e);
            logger.warn(errorMessage);
            throw new UDFRegistrationException(errorMessage);
        }
        finally {
            this.releaseRegistrationLock();
        }
    }

    private void tryAppendRegistrationLog(String functionName, String className, boolean isTemporary, boolean writeToTemporaryLogFile) throws UDFRegistrationException {
        if (!writeToTemporaryLogFile || isTemporary) {
            return;
        }
        try {
            this.appendRegistrationLog(functionName, className);
        }
        catch (IOException e) {
            this.registrationInformation.remove(functionName);
            String errorMessage = String.format("Failed to append UDF log when registering UDF %s(%s), because %s", functionName, className, e);
            logger.error(errorMessage);
            throw new UDFRegistrationException(errorMessage, e);
        }
    }

    private void updateAllRegisteredClasses(UDFClassLoader activeClassLoader) throws ClassNotFoundException {
        for (UDFRegistrationInformation information : this.getRegistrationInformation()) {
            if (information.isBuiltin()) continue;
            information.updateFunctionClass(activeClassLoader);
        }
    }

    public void deregister(String functionName) throws UDFRegistrationException {
        UDFRegistrationInformation information = this.registrationInformation.get(functionName = functionName.toUpperCase());
        if (information == null) {
            String errorMessage = String.format("UDF %s does not exist.", functionName);
            logger.warn(errorMessage);
            throw new UDFRegistrationException(errorMessage);
        }
        if (information.isBuiltin()) {
            String errorMessage = String.format("Built-in function %s can not be deregistered.", functionName);
            logger.error(errorMessage);
            throw new UDFRegistrationException(errorMessage);
        }
        if (!information.isTemporary()) {
            try {
                this.appendDeregistrationLog(functionName);
            }
            catch (IOException e) {
                String errorMessage = String.format("Failed to append UDF log when deregistering UDF %s, because %s", functionName, e);
                logger.error(errorMessage);
                throw new UDFRegistrationException(errorMessage, e);
            }
        }
        this.registrationInformation.remove(functionName);
    }

    private void appendRegistrationLog(String functionName, String className) throws IOException {
        this.logWriterLock.writeLock().lock();
        try {
            this.logWriter.register(functionName, className);
        }
        finally {
            this.logWriterLock.writeLock().unlock();
        }
    }

    private void appendDeregistrationLog(String functionName) throws IOException {
        this.logWriterLock.writeLock().lock();
        try {
            this.logWriter.deregister(functionName);
        }
        finally {
            this.logWriterLock.writeLock().unlock();
        }
    }

    public UDF reflect(UDFContext context) throws QueryProcessException {
        String functionName = context.getName().toUpperCase();
        UDFRegistrationInformation information = this.registrationInformation.get(functionName);
        if (information == null) {
            String errorMessage = String.format("Failed to reflect UDF instance, because UDF %s has not been registered.", functionName);
            logger.warn(errorMessage);
            throw new QueryProcessException(errorMessage);
        }
        if (!information.isBuiltin()) {
            Thread.currentThread().setContextClassLoader(UDFClassLoaderManager.getInstance().getActiveClassLoader());
        }
        try {
            return (UDF)information.getFunctionClass().getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            String errorMessage = String.format("Failed to reflect UDF %s(%s) instance, because %s", functionName, information.getClassName(), e);
            logger.warn(errorMessage);
            throw new QueryProcessException(errorMessage);
        }
    }

    public UDFRegistrationInformation[] getRegistrationInformation() {
        return this.registrationInformation.values().toArray(new UDFRegistrationInformation[0]);
    }

    @Override
    public void start() throws StartupException {
        try {
            this.registerBuiltinFunctions();
            this.makeDirIfNecessary();
            this.doRecovery();
            this.logWriter = new UDFLogWriter(LOG_FILE_NAME);
        }
        catch (Exception e) {
            throw new StartupException(e);
        }
    }

    private void registerBuiltinFunctions() {
        for (BuiltinFunction builtinFunction : BuiltinFunction.values()) {
            String functionName = builtinFunction.getFunctionName();
            this.registrationInformation.put(functionName, new UDFRegistrationInformation(functionName, builtinFunction.getClassName(), false, true, builtinFunction.getFunctionClass()));
        }
    }

    private void makeDirIfNecessary() throws IOException {
        File file = SystemFileFactory.INSTANCE.getFile(ULOG_FILE_DIR);
        if (file.exists() && file.isDirectory()) {
            return;
        }
        FileUtils.forceMkdir((File)file);
    }

    private void doRecovery() throws IOException {
        File temporaryLogFile = SystemFileFactory.INSTANCE.getFile(TEMPORARY_LOG_FILE_NAME);
        File logFile = SystemFileFactory.INSTANCE.getFile(LOG_FILE_NAME);
        if (temporaryLogFile.exists()) {
            if (logFile.exists()) {
                this.recoveryFromLogFile(logFile);
                FileUtils.deleteQuietly((File)temporaryLogFile);
            } else {
                this.recoveryFromLogFile(temporaryLogFile);
                FSFactoryProducer.getFSFactory().moveFile(temporaryLogFile, logFile);
            }
        } else if (logFile.exists()) {
            this.recoveryFromLogFile(logFile);
        }
    }

    private void recoveryFromLogFile(File logFile) throws IOException {
        HashMap<String, String> recoveredUDFs = new HashMap<String, String>();
        try (BufferedReader reader = new BufferedReader(new FileReader(logFile));){
            String line;
            while ((line = reader.readLine()) != null) {
                String[] data = line.split(",");
                byte type = Byte.parseByte(data[0]);
                if (type == UDFLogWriter.REGISTER_TYPE) {
                    recoveredUDFs.put(data[1], data[2]);
                    continue;
                }
                if (type != UDFLogWriter.DEREGISTER_TYPE) continue;
                recoveredUDFs.remove(data[1]);
            }
        }
        for (Map.Entry udf : recoveredUDFs.entrySet()) {
            try {
                this.register((String)udf.getKey(), (String)udf.getValue(), false, false);
            }
            catch (UDFRegistrationException uDFRegistrationException) {}
        }
    }

    @Override
    public void stop() {
        try {
            this.writeTemporaryLogFile();
            this.logWriter.close();
            this.logWriter.deleteLogFile();
            File temporaryLogFile = SystemFileFactory.INSTANCE.getFile(TEMPORARY_LOG_FILE_NAME);
            File logFile = SystemFileFactory.INSTANCE.getFile(LOG_FILE_NAME);
            FSFactoryProducer.getFSFactory().moveFile(temporaryLogFile, logFile);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void writeTemporaryLogFile() throws IOException {
        UDFLogWriter temporaryLogFile = new UDFLogWriter(TEMPORARY_LOG_FILE_NAME);
        for (UDFRegistrationInformation information : this.registrationInformation.values()) {
            if (information.isBuiltin() || information.isTemporary()) continue;
            temporaryLogFile.register(information.getFunctionName(), information.getClassName());
        }
        temporaryLogFile.close();
    }

    public void deregisterAll() throws UDFRegistrationException {
        for (UDFRegistrationInformation information : this.getRegistrationInformation()) {
            if (information.isBuiltin()) continue;
            this.deregister(information.getFunctionName());
        }
    }

    public void registerBuiltinFunction(String functionName, String className) throws ClassNotFoundException {
        ClassLoader classLoader = this.getClass().getClassLoader();
        Class<?> functionClass = Class.forName(className, true, classLoader);
        functionName = functionName.toUpperCase();
        this.registrationInformation.put(functionName, new UDFRegistrationInformation(functionName, className, false, true, functionClass));
    }

    public void deregisterBuiltinFunction(String functionName) {
        this.registrationInformation.remove(functionName.toUpperCase());
    }

    @Override
    public ServiceType getID() {
        return ServiceType.UDF_REGISTRATION_SERVICE;
    }

    public static UDFRegistrationService getInstance() {
        return UDFRegistrationServiceHelper.INSTANCE;
    }

    private static class UDFRegistrationServiceHelper {
        private static final UDFRegistrationService INSTANCE = new UDFRegistrationService();

        private UDFRegistrationServiceHelper() {
        }
    }
}

