/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.ml.util.genetic;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.math3.util.Pair;
import org.apache.ignite.ml.environment.LearningEnvironment;
import org.apache.ignite.ml.environment.parallelism.Promise;
import org.apache.ignite.ml.math.functions.IgniteSupplier;
import org.apache.ignite.ml.util.genetic.Chromosome;
import org.apache.ignite.ml.util.genetic.CrossoverStrategy;
import org.apache.ignite.ml.util.genetic.Population;
import org.apache.ignite.ml.util.genetic.SelectionStrategy;

public class GeneticAlgorithm {
    private static final double UNIFORM_RATE = 0.5;
    private int populationSize;
    private int amountOfEliteChromosomes = 2;
    private int amountOfGenerations = 10;
    private long seed = 123L;
    private double crossingoverProbability = 0.9;
    private double mutationProbability = 0.1;
    private Random rnd = new Random(this.seed);
    private Population population;
    private Function<Chromosome, Double> fitnessFunction;
    private BiFunction<Integer, Double, Double> mutationOperator;
    private CrossoverStrategy crossoverStgy = CrossoverStrategy.UNIFORM;
    private SelectionStrategy selectionStgy = SelectionStrategy.ROULETTE_WHEEL;

    public GeneticAlgorithm(List<Double[]> paramSet) {
        this.initializePopulation(paramSet);
    }

    private Population initializePopulation(List<Double[]> rawDataForPopulationFormation) {
        this.populationSize = rawDataForPopulationFormation.size();
        this.population = new Population(this.populationSize);
        for (int i = 0; i < this.populationSize; ++i) {
            this.population.setChromosome(i, new Chromosome(rawDataForPopulationFormation.get(i)));
        }
        return this.population;
    }

    public void run() {
        if (this.population != null) {
            this.population.calculateFitnessForAll(this.fitnessFunction);
            int i = 0;
            while (this.stopCriteriaIsReached(i)) {
                Population newPopulation = new Population(this.populationSize);
                newPopulation = this.selectEliteChromosomes(newPopulation);
                Population parents = this.selectionParents();
                newPopulation = this.crossingover(parents, newPopulation);
                newPopulation = this.mutate(newPopulation);
                for (int j = this.amountOfEliteChromosomes; j < this.populationSize; ++j) {
                    newPopulation.calculateFitnessForChromosome(j, this.fitnessFunction);
                }
                this.population = newPopulation;
                ++i;
            }
        }
    }

    public void runParallel(LearningEnvironment environment) {
        if (this.population != null) {
            this.population.calculateFitnessForAll(this.fitnessFunction);
            int i = 0;
            while (this.stopCriteriaIsReached(i)) {
                Population newPopulation = new Population(this.populationSize);
                newPopulation = this.selectEliteChromosomes(newPopulation);
                Population parents = this.selectionParents();
                newPopulation = this.crossingover(parents, newPopulation);
                newPopulation = this.mutate(newPopulation);
                ArrayList tasks = new ArrayList();
                int j = this.amountOfEliteChromosomes;
                while (j < this.populationSize) {
                    int finalJ = j++;
                    Population finalNewPopulation1 = newPopulation;
                    IgniteSupplier task = () -> new Pair((Object)finalJ, (Object)this.fitnessFunction.apply(finalNewPopulation1.getChromosome(finalJ)));
                    tasks.add(task);
                }
                List<Pair> taskResults = environment.parallelismStrategy().submit(tasks).stream().map(Promise::unsafeGet).collect(Collectors.toList());
                Population finalNewPopulation = newPopulation;
                taskResults.forEach(p -> finalNewPopulation.setFitness((Integer)p.getKey(), (Double)p.getValue()));
                this.population = newPopulation;
                ++i;
            }
        }
    }

    private Population selectionParents() {
        switch (this.selectionStgy) {
            case ROULETTE_WHEEL: {
                return this.selectParentsByRouletteWheel();
            }
        }
        throw new UnsupportedOperationException("This strategy " + this.selectionStgy.name() + " is not supported yet.");
    }

    private Population selectParentsByRouletteWheel() {
        double totalFitness = this.population.getTotalFitness();
        double[] sectors = new double[this.population.size()];
        for (int i = 0; i < this.population.size(); ++i) {
            sectors[i] = this.population.getChromosome(i).getFitness() / totalFitness;
        }
        Population parentPopulation = new Population(this.population.size());
        for (int i = 0; i < parentPopulation.size(); ++i) {
            double rouletteVal = this.rnd.nextDouble();
            double accumulatedSectorLen = 0.0;
            int selectedChromosomeIdx = Integer.MIN_VALUE;
            for (int sectorIdx = 0; selectedChromosomeIdx == Integer.MIN_VALUE && sectorIdx < sectors.length; ++sectorIdx) {
                if (!(rouletteVal < (accumulatedSectorLen += sectors[sectorIdx]))) continue;
                selectedChromosomeIdx = sectorIdx;
            }
            if (selectedChromosomeIdx == Integer.MIN_VALUE) {
                selectedChromosomeIdx = this.rnd.nextInt(this.population.size());
            }
            parentPopulation.setChromosome(i, this.population.getChromosome(selectedChromosomeIdx));
        }
        return parentPopulation;
    }

    private boolean stopCriteriaIsReached(int iterationNum) {
        return iterationNum < this.amountOfGenerations;
    }

    private Population selectEliteChromosomes(Population newPopulation) {
        boolean elitism;
        boolean bl = elitism = this.amountOfEliteChromosomes > 0;
        if (elitism) {
            Chromosome[] elite = this.population.selectBestKChromosome(this.amountOfEliteChromosomes);
            for (int i = 0; i < elite.length; ++i) {
                newPopulation.setChromosome(i, elite[i]);
            }
        }
        return newPopulation;
    }

    private Population crossingover(Population parents, Population newPopulation) {
        for (int j = 0; j < this.populationSize - this.amountOfEliteChromosomes; j += 2) {
            Chromosome ch1 = parents.getChromosome(j);
            Chromosome ch2 = parents.getChromosome(j + 1);
            if (this.rnd.nextDouble() < this.crossingoverProbability) {
                List<Chromosome> twoChildren = this.crossover(ch1, ch2);
                newPopulation.setChromosome(this.amountOfEliteChromosomes + j, twoChildren.get(0));
                newPopulation.setChromosome(this.amountOfEliteChromosomes + j + 1, twoChildren.get(1));
                continue;
            }
            newPopulation.setChromosome(this.amountOfEliteChromosomes + j, ch1);
            newPopulation.setChromosome(this.amountOfEliteChromosomes + j + 1, ch2);
        }
        return newPopulation;
    }

    private Population mutate(Population newPopulation) {
        for (int j = this.amountOfEliteChromosomes; j < this.populationSize; ++j) {
            Chromosome possibleMutant = newPopulation.getChromosome(j);
            for (int geneIdx = 0; geneIdx < possibleMutant.size(); ++geneIdx) {
                if (!(this.rnd.nextDouble() < this.mutationProbability)) continue;
                Double gene = possibleMutant.getGene(geneIdx);
                Double newGeneVal = this.mutationOperator.apply(geneIdx, gene);
                possibleMutant.setGene(geneIdx, newGeneVal);
            }
        }
        return newPopulation;
    }

    private List<Chromosome> crossover(Chromosome firstParent, Chromosome secondParent) {
        if (firstParent.size() != secondParent.size()) {
            throw new RuntimeException("Different length of hyper-parameter vectors!");
        }
        switch (this.crossoverStgy) {
            case UNIFORM: {
                return this.uniformStrategy(firstParent, secondParent);
            }
            case ONE_POINT: {
                return this.onePointStrategy(firstParent, secondParent);
            }
        }
        throw new UnsupportedOperationException("This strategy " + this.crossoverStgy.name() + " is not supported yet.");
    }

    private List<Chromosome> onePointStrategy(Chromosome firstParent, Chromosome secondParent) {
        int i;
        int size = firstParent.size();
        Chromosome child1 = new Chromosome(size);
        Chromosome child2 = new Chromosome(size);
        int locusPnt = this.rnd.nextInt(size);
        for (i = 0; i < locusPnt; ++i) {
            child1.setGene(i, firstParent.getGene(i));
            child2.setGene(i, secondParent.getGene(i));
        }
        for (i = locusPnt; i < size; ++i) {
            child1.setGene(i, secondParent.getGene(i));
            child2.setGene(i, firstParent.getGene(i));
        }
        ArrayList<Chromosome> res = new ArrayList<Chromosome>();
        res.add(child1);
        res.add(child2);
        return res;
    }

    private List<Chromosome> uniformStrategy(Chromosome firstParent, Chromosome secondParent) {
        int size = firstParent.size();
        Chromosome child1 = new Chromosome(size);
        Chromosome child2 = new Chromosome(size);
        for (int i = 0; i < firstParent.size(); ++i) {
            if (this.rnd.nextDouble() < 0.5) {
                child1.setGene(i, firstParent.getGene(i));
                child2.setGene(i, secondParent.getGene(i));
                continue;
            }
            child1.setGene(i, secondParent.getGene(i));
            child2.setGene(i, firstParent.getGene(i));
        }
        ArrayList<Chromosome> res = new ArrayList<Chromosome>();
        res.add(child1);
        res.add(child2);
        return res;
    }

    public GeneticAlgorithm withFitnessFunction(Function<Chromosome, Double> fitnessFunction) {
        this.fitnessFunction = fitnessFunction;
        return this;
    }

    public double[] getTheBestSolution() {
        Double[] boxed = this.population.selectBestKChromosome(1)[0].toDoubleArray();
        return Stream.of(boxed).mapToDouble(Double::doubleValue).toArray();
    }

    public GeneticAlgorithm withAmountOfEliteChromosomes(int amountOfEliteChromosomes) {
        this.amountOfEliteChromosomes = amountOfEliteChromosomes;
        return this;
    }

    public GeneticAlgorithm withAmountOfGenerations(int amountOfGenerations) {
        this.amountOfGenerations = amountOfGenerations;
        return this;
    }

    public GeneticAlgorithm withMutationOperator(BiFunction<Integer, Double, Double> mutationOperator) {
        this.mutationOperator = mutationOperator;
        return this;
    }

    public GeneticAlgorithm withPopulationSize(int populationSize) {
        this.populationSize = populationSize;
        return this;
    }

    public GeneticAlgorithm withCrossingoverProbability(double crossingoverProbability) {
        this.crossingoverProbability = crossingoverProbability;
        return this;
    }

    public GeneticAlgorithm withMutationProbability(double mutationProbability) {
        this.mutationProbability = mutationProbability;
        return this;
    }

    public GeneticAlgorithm withCrossoverStgy(CrossoverStrategy crossoverStgy) {
        this.crossoverStgy = crossoverStgy;
        return this;
    }

    public GeneticAlgorithm withSelectionStgy(SelectionStrategy selectionStgy) {
        this.selectionStgy = selectionStgy;
        return this;
    }
}

