/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.gcp.transaction.lock;

import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.Properties;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.hudi.client.transaction.lock.StorageLockClient;
import org.apache.hudi.client.transaction.lock.models.LockGetResult;
import org.apache.hudi.client.transaction.lock.models.LockUpsertResult;
import org.apache.hudi.client.transaction.lock.models.StorageLockData;
import org.apache.hudi.client.transaction.lock.models.StorageLockFile;
import org.apache.hudi.common.util.Functions;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.VisibleForTesting;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieIOException;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class GCSStorageLockClient
implements StorageLockClient {
    private static final Logger DEFAULT_LOGGER = LoggerFactory.getLogger(GCSStorageLockClient.class);
    private static final long PRECONDITION_FAILURE_ERROR_CODE = 412L;
    private static final long NOT_FOUND_ERROR_CODE = 404L;
    private static final long RATE_LIMIT_ERROR_CODE = 429L;
    private static final long INTERNAL_SERVER_ERROR_CODE_MIN = 500L;
    private final Logger logger;
    private final Storage gcsClient;
    private final String bucketName;
    private final String lockFilePath;
    private final String ownerId;

    public GCSStorageLockClient(String ownerId, String lockFileUri, Properties props) {
        this(ownerId, lockFileUri, props, GCSStorageLockClient.createDefaultGcsClient(), DEFAULT_LOGGER);
    }

    @VisibleForTesting
    GCSStorageLockClient(String ownerId, String lockFileUri, Properties properties, Functions.Function1<Properties, Storage> gcsClientSupplier, Logger logger) {
        Pair bucketAndPath = StorageLockClient.parseBucketAndPath((String)lockFileUri);
        this.bucketName = (String)bucketAndPath.getLeft();
        this.lockFilePath = (String)bucketAndPath.getRight();
        this.gcsClient = (Storage)gcsClientSupplier.apply((Object)properties);
        this.ownerId = ownerId;
        this.logger = logger;
    }

    private static Functions.Function1<Properties, Storage> createDefaultGcsClient() {
        return (Functions.Function1 & Serializable)props -> (Storage)StorageOptions.newBuilder().build().getService();
    }

    private StorageLockFile createOrUpdateLockFileInternal(StorageLockData lockData, long generationNumber) throws StorageException {
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)BlobId.of((String)this.bucketName, (String)this.lockFilePath)).build();
        Blob updatedBlob = this.gcsClient.create(blobInfo, StorageLockFile.toByteArray((StorageLockData)lockData), new Storage.BlobTargetOption[]{Storage.BlobTargetOption.generationMatch((long)generationNumber)});
        return new StorageLockFile(lockData, String.valueOf(updatedBlob.getGeneration()));
    }

    public Pair<LockUpsertResult, Option<StorageLockFile>> tryUpsertLockFile(StorageLockData newLockData, Option<StorageLockFile> previousLockFile) {
        long generationNumber = this.getGenerationNumber(previousLockFile);
        try {
            StorageLockFile updatedFile = this.createOrUpdateLockFileInternal(newLockData, generationNumber);
            return Pair.of((Object)LockUpsertResult.SUCCESS, (Object)Option.of((Object)updatedFile));
        }
        catch (StorageException e) {
            if ((long)e.getCode() == 412L) {
                this.logger.info("OwnerId: {}, Unable to write new lock file. Another process has modified this lockfile {} already.", (Object)this.ownerId, (Object)this.lockFilePath);
                return Pair.of((Object)LockUpsertResult.ACQUIRED_BY_OTHERS, (Object)Option.empty());
            }
            if ((long)e.getCode() == 429L) {
                this.logger.warn("OwnerId: {}, Rate limit exceeded for lock file: {}", (Object)this.ownerId, (Object)this.lockFilePath);
            } else if ((long)e.getCode() >= 500L) {
                this.logger.warn("OwnerId: {}, GCS returned internal server error code for lock file: {}", new Object[]{this.ownerId, this.lockFilePath, e});
            } else {
                throw e;
            }
            return Pair.of((Object)LockUpsertResult.UNKNOWN_ERROR, (Object)Option.empty());
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private LockGetResult handleGetStorageException(StorageException e, boolean ignore404) {
        if ((long)e.getCode() == 404L) {
            if (ignore404) {
                this.logger.info("OwnerId: {}, GCS stream read failure detected: {}", (Object)this.ownerId, (Object)this.lockFilePath);
                return LockGetResult.UNKNOWN_ERROR;
            }
            this.logger.info("OwnerId: {}, Object not found in the path: {}", (Object)this.ownerId, (Object)this.lockFilePath);
            return LockGetResult.NOT_EXISTS;
        }
        if ((long)e.getCode() == 429L) {
            this.logger.warn("OwnerId: {}, Rate limit exceeded for lock file: {}", (Object)this.ownerId, (Object)this.lockFilePath);
            return LockGetResult.UNKNOWN_ERROR;
        }
        if ((long)e.getCode() < 500L) throw e;
        this.logger.warn("OwnerId: {}, GCS returned internal server error code for lock file: {}", new Object[]{this.ownerId, this.lockFilePath, e});
        return LockGetResult.UNKNOWN_ERROR;
    }

    public Pair<LockGetResult, Option<StorageLockFile>> readCurrentLockFile() {
        try {
            Blob blob = this.gcsClient.get(BlobId.of((String)this.bucketName, (String)this.lockFilePath));
            if (blob == null) {
                return Pair.of((Object)LockGetResult.NOT_EXISTS, (Object)Option.empty());
            }
            return this.getLockFileFromBlob(blob);
        }
        catch (StorageException e) {
            return Pair.of((Object)this.handleGetStorageException(e, false), (Object)Option.empty());
        }
        catch (HoodieIOException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException && cause.getCause() instanceof StorageException) {
                return Pair.of((Object)this.handleGetStorageException((StorageException)cause.getCause(), true), (Object)Option.empty());
            }
            throw e;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    private Pair<LockGetResult, Option<StorageLockFile>> getLockFileFromBlob(Blob blob) {
        try (InputStream inputStream = Channels.newInputStream((ReadableByteChannel)blob.reader(new Blob.BlobSourceOption[0]));){
            Pair pair = Pair.of((Object)LockGetResult.SUCCESS, (Object)Option.of((Object)StorageLockFile.createFromStream((InputStream)inputStream, (String)String.valueOf(blob.getGeneration()))));
            return pair;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed reading blob: " + this.lockFilePath, e);
        }
    }

    public Option<String> readObject(String filePath, boolean checkExistsFirst) {
        try {
            Pair bucketAndPath = StorageLockClient.parseBucketAndPath((String)filePath);
            String bucket = (String)bucketAndPath.getLeft();
            String objectPath = (String)bucketAndPath.getRight();
            BlobId blobId = BlobId.of((String)bucket, (String)objectPath);
            if (checkExistsFirst) {
                Blob blob = this.gcsClient.get(blobId);
                if (blob == null || !blob.exists(new Blob.BlobSourceOption[0])) {
                    this.logger.debug("JSON config file not found: {}", (Object)filePath);
                    return Option.empty();
                }
                byte[] content = blob.getContent(new Blob.BlobSourceOption[0]);
                return Option.of((Object)new String(content, StandardCharsets.UTF_8));
            }
            byte[] content = this.gcsClient.readAllBytes(blobId, new Storage.BlobSourceOption[0]);
            return Option.of((Object)new String(content, StandardCharsets.UTF_8));
        }
        catch (StorageException e) {
            if ((long)e.getCode() == 404L) {
                this.logger.debug("JSON config file not found: {}", (Object)filePath);
            } else {
                this.logger.warn("Error reading JSON config file: {}", (Object)filePath, (Object)e);
            }
            return Option.empty();
        }
        catch (Exception e) {
            this.logger.warn("Error reading JSON config file: {}", (Object)filePath, (Object)e);
            return Option.empty();
        }
    }

    public boolean writeObject(String filePath, String content) {
        try {
            Pair bucketAndPath = StorageLockClient.parseBucketAndPath((String)filePath);
            String bucket = (String)bucketAndPath.getLeft();
            String objectPath = (String)bucketAndPath.getRight();
            BlobId blobId = BlobId.of((String)bucket, (String)objectPath);
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).build();
            this.gcsClient.create(blobInfo, content.getBytes(StandardCharsets.UTF_8), new Storage.BlobTargetOption[0]);
            this.logger.debug("Successfully wrote object to: {}", (Object)filePath);
            return true;
        }
        catch (Exception e) {
            this.logger.error("Error writing object to: {}", (Object)filePath, (Object)e);
            return false;
        }
    }

    public void close() throws Exception {
        this.gcsClient.close();
    }

    private long getGenerationNumber(Option<StorageLockFile> file) {
        return file.isPresent() ? Long.parseLong(((StorageLockFile)file.get()).getVersionId()) : 0L;
    }
}

