/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.gcp.transaction.lock;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.cloud.ReadChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import java.io.IOException;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Properties;
import org.apache.hudi.client.transaction.lock.models.LockGetResult;
import org.apache.hudi.client.transaction.lock.models.LockUpsertResult;
import org.apache.hudi.client.transaction.lock.models.StorageLockData;
import org.apache.hudi.client.transaction.lock.models.StorageLockFile;
import org.apache.hudi.common.util.Functions;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.gcp.transaction.lock.GCSStorageLockClient;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.mockito.ArgumentMatchers;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mockito.stubbing.Answer;
import org.slf4j.Logger;

@ExtendWith(value={MockitoExtension.class})
public class TestGCSStorageLockClient {
    private static final String OWNER_ID = "ownerId";
    private static final String LOCK_FILE_URI = "gs://bucket/lockFilePath";
    private static final String LOCK_FILE_URI_WITH_UNDERSCORES = "gs://bucket_with_underscores/lockFilePath";
    private static final String LOCK_FILE_PATH = "lockFilePath";
    private static final String BUCKET_NAME = "bucket";
    @Mock
    private Storage mockStorage;
    @Mock
    private Blob mockBlob;
    @Mock
    private Logger mockLogger;
    private GCSStorageLockClient lockService;
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    public static void mockBlobReaderWithLockData(Blob mockBlob, StorageLockData data) throws IOException {
        String json = OBJECT_MAPPER.writeValueAsString((Object)data);
        byte[] jsonBytes = json.getBytes();
        ReadChannel mockReadChannel = (ReadChannel)Mockito.mock(ReadChannel.class);
        Answer fillBufferWithJson = invocation -> {
            ByteBuffer buffer = (ByteBuffer)invocation.getArgument(0);
            buffer.put(jsonBytes);
            return jsonBytes.length;
        };
        ((ReadChannel)Mockito.doAnswer((Answer)fillBufferWithJson).doAnswer(invocation -> -1).when((Object)mockReadChannel)).read((ByteBuffer)ArgumentMatchers.any(ByteBuffer.class));
        Mockito.when((Object)mockBlob.reader(new Blob.BlobSourceOption[0])).thenReturn((Object)mockReadChannel);
    }

    @BeforeEach
    void setUp() {
        this.setUp(LOCK_FILE_URI);
    }

    private void setUp(String lockFileUri) {
        this.lockService = new GCSStorageLockClient(OWNER_ID, lockFileUri, new Properties(), (Functions.Function1 & Serializable)a -> this.mockStorage, this.mockLogger);
    }

    @ParameterizedTest
    @ValueSource(strings={"gs://bucket/lockFilePath", "gs://bucket_with_underscores/lockFilePath"})
    void testTryCreateOrUpdateLockFile_noPreviousLock_success(String lockFileUri) {
        this.setUp(lockFileUri);
        StorageLockData lockData = new StorageLockData(false, 123L, "test-owner");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenReturn((Object)this.mockBlob);
        Pair result = this.lockService.tryUpsertLockFile(lockData, Option.empty());
        Assertions.assertNotNull((Object)result.getRight(), (String)"Expected a valid StorageLockFile on success");
        ((Storage)Mockito.verify((Object)this.mockStorage)).create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.eq((Object)Storage.BlobTargetOption.generationMatch((long)0L))});
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.mockLogger});
    }

    @Test
    void testTryCreateOrUpdateLockFile_withPreviousLock_success() {
        StorageLockData lockData = new StorageLockData(false, 999L, "existing-owner");
        StorageLockFile previousLockFile = new StorageLockFile(lockData, "123");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenReturn((Object)this.mockBlob);
        Pair result = this.lockService.tryUpsertLockFile(lockData, Option.of((Object)previousLockFile));
        Assertions.assertNotNull((Object)result, (String)"Expected a valid StorageLockFile on success");
        ((Storage)Mockito.verify((Object)this.mockStorage)).create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.eq((Object)Storage.BlobTargetOption.generationMatch((long)123L))});
    }

    @Test
    void testTryCreateOrUpdateLockFile_preconditionFailed() {
        StorageLockData lockData = new StorageLockData(false, 999L, "owner");
        StorageLockFile previousLockFile = new StorageLockFile(lockData, "123");
        StorageException exception = new StorageException(412, "Precondition Failed");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenThrow(new Throwable[]{exception});
        Pair result = this.lockService.tryUpsertLockFile(lockData, Option.of((Object)previousLockFile));
        Assertions.assertEquals((Object)LockUpsertResult.ACQUIRED_BY_OTHERS, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty when a 412 occurs");
        ((Logger)Mockito.verify((Object)this.mockLogger)).info(ArgumentMatchers.contains((String)"Unable to write new lock file. Another process has modified this lockfile"), ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH));
    }

    @Test
    void testTryCreateOrUpdateLockFile_rateLimitExceeded() {
        StorageLockData lockData = new StorageLockData(false, 999L, "owner");
        StorageException exception = new StorageException(429, "Rate Limit Exceeded");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenThrow(new Throwable[]{exception});
        Pair result = this.lockService.tryUpsertLockFile(lockData, Option.empty());
        Assertions.assertEquals((Object)LockUpsertResult.UNKNOWN_ERROR, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty when a 429 occurs");
        ((Logger)Mockito.verify((Object)this.mockLogger)).warn(ArgumentMatchers.contains((String)"Rate limit exceeded"), ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH));
    }

    @Test
    void testTryCreateOrUpdateLockFile_serverError() {
        StorageLockData lockData = new StorageLockData(false, 999L, "owner");
        StorageException exception = new StorageException(503, "Service Unavailable");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenThrow(new Throwable[]{exception});
        Pair result = this.lockService.tryUpsertLockFile(lockData, Option.empty());
        Assertions.assertEquals((Object)LockUpsertResult.UNKNOWN_ERROR, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty when a 5xx error occurs");
        ((Logger)Mockito.verify((Object)this.mockLogger)).warn(ArgumentMatchers.contains((String)"GCS returned internal server error code"), new Object[]{ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH), ArgumentMatchers.eq((Object)exception)});
    }

    @Test
    void testTryCreateOrUpdateLockFile_unexpectedError() {
        StorageLockData lockData = new StorageLockData(false, 999L, "owner");
        StorageException exception = new StorageException(400, "Bad Request");
        Mockito.when((Object)this.mockStorage.create((BlobInfo)ArgumentMatchers.any(BlobInfo.class), (byte[])ArgumentMatchers.any(byte[].class), new Storage.BlobTargetOption[]{(Storage.BlobTargetOption)ArgumentMatchers.any(Storage.BlobTargetOption.class)})).thenThrow(new Throwable[]{exception});
        Assertions.assertThrows(StorageException.class, () -> this.lockService.tryUpsertLockFile(lockData, Option.empty()), (String)"Expected the method to rethrow the exception");
    }

    @Test
    void testGetCurrentLockFile_blobNotFound() {
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenReturn(null);
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.NOT_EXISTS, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Expected empty when no blob is found");
    }

    @Test
    void testGetCurrentLockFile_blobFound() throws IOException {
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenReturn((Object)this.mockBlob);
        TestGCSStorageLockClient.mockBlobReaderWithLockData(this.mockBlob, new StorageLockData(false, 123L, "owner"));
        Mockito.when((Object)this.mockBlob.getGeneration()).thenReturn((Object)123L);
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.SUCCESS, (Object)result.getLeft());
        Assertions.assertNotNull((Object)result.getRight(), (String)"Should return a StorageLockFile if blob is found");
        Assertions.assertEquals((Object)"123", (Object)((StorageLockFile)((Option)result.getRight()).get()).getVersionId(), (String)"Version ID should match blob generation");
    }

    @Test
    void testGetCurrentLockFile_404Error() {
        StorageException exception404 = new StorageException(404, "Not Found");
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenThrow(new Throwable[]{exception404});
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.NOT_EXISTS, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty on 404 error");
        ((Logger)Mockito.verify((Object)this.mockLogger)).info(ArgumentMatchers.contains((String)"Object not found"), ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH));
    }

    @Test
    void testGetCurrentLockFile_rateLimit() {
        StorageException exception429 = new StorageException(429, "Rate Limit Exceeded");
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenThrow(new Throwable[]{exception429});
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.UNKNOWN_ERROR, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty on 429 error");
        ((Logger)Mockito.verify((Object)this.mockLogger)).warn(ArgumentMatchers.contains((String)"Rate limit exceeded"), ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH));
    }

    @Test
    void testGetCurrentLockFile_serverError() {
        StorageException exception500 = new StorageException(503, "Service Unavailable");
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenThrow(new Throwable[]{exception500});
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.UNKNOWN_ERROR, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty on 5xx error");
        ((Logger)Mockito.verify((Object)this.mockLogger)).warn(ArgumentMatchers.contains((String)"GCS returned internal server error code"), new Object[]{ArgumentMatchers.eq((Object)OWNER_ID), ArgumentMatchers.eq((Object)LOCK_FILE_PATH), ArgumentMatchers.eq((Object)exception500)});
    }

    @Test
    void testGetCurrentLockFile_unexpectedError() {
        StorageException exception400 = new StorageException(400, "Bad Request");
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenThrow(new Throwable[]{exception400});
        Assertions.assertThrows(StorageException.class, () -> this.lockService.readCurrentLockFile(), (String)"Should rethrow unexpected errors");
    }

    @Test
    void testGetCurrentLockFile_IOException() {
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenReturn((Object)this.mockBlob);
        Mockito.when((Object)this.mockBlob.reader(new Blob.BlobSourceOption[0])).thenThrow(new Throwable[]{new HoodieIOException("IO Error")});
        Assertions.assertThrows(HoodieIOException.class, () -> this.lockService.readCurrentLockFile());
    }

    @Test
    void testGetCurrentLockFile_IOExceptionWrapping404() {
        Mockito.when((Object)this.mockStorage.get(BlobId.of((String)BUCKET_NAME, (String)LOCK_FILE_PATH))).thenReturn((Object)this.mockBlob);
        Mockito.when((Object)this.mockBlob.reader(new Blob.BlobSourceOption[0])).thenThrow(new Throwable[]{new HoodieIOException("IO Error", new IOException((Throwable)new StorageException(404, "storage 404")))});
        Pair result = this.lockService.readCurrentLockFile();
        Assertions.assertEquals((Object)LockGetResult.UNKNOWN_ERROR, (Object)result.getLeft());
        Assertions.assertTrue((boolean)((Option)result.getRight()).isEmpty(), (String)"Should return empty on IO exception wrapping 404 error");
    }

    @Test
    void testClose() throws Exception {
        this.lockService.close();
        ((Storage)Mockito.verify((Object)this.mockStorage)).close();
    }
}

