/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.client;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.avro.Schema;
import org.apache.avro.generic.IndexedRecord;
import org.apache.avro.specific.SpecificRecordBase;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.avro.model.HoodieArchivedMetaEntry;
import org.apache.hudi.avro.model.HoodieMergeArchiveFilePlan;
import org.apache.hudi.client.transaction.TransactionManager;
import org.apache.hudi.client.utils.ArchivalUtils;
import org.apache.hudi.client.utils.MetadataConversionUtils;
import org.apache.hudi.common.config.HoodieMetadataConfig;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.fs.HoodieWrapperFileSystem;
import org.apache.hudi.common.fs.StorageSchemes;
import org.apache.hudi.common.function.SerializableConsumer;
import org.apache.hudi.common.model.HoodieAvroIndexedRecord;
import org.apache.hudi.common.model.HoodieAvroPayload;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.model.HoodieRecord;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.table.timeline.TimelineMetadataUtils;
import org.apache.hudi.common.table.timeline.TimelineUtils;
import org.apache.hudi.common.util.ClusteringUtils;
import org.apache.hudi.common.util.CollectionUtils;
import org.apache.hudi.common.util.CompactionUtils;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieCommitException;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.hudi.table.HoodieTable;
import org.apache.hudi.table.action.compact.CompactionTriggerStrategy;
import org.apache.hudi.table.marker.WriteMarkers;
import org.apache.hudi.table.marker.WriteMarkersFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoodieTimelineArchiver<T extends HoodieAvroPayload, I, K, O> {
    private static final Logger LOG = LoggerFactory.getLogger(HoodieTimelineArchiver.class);
    private final Path archiveFilePath;
    private final HoodieWriteConfig config;
    private HoodieLogFormat.Writer writer;
    private final int maxInstantsToKeep;
    private final int minInstantsToKeep;
    private final HoodieTable<T, I, K, O> table;
    private final HoodieTableMetaClient metaClient;
    private final TransactionManager txnManager;

    public HoodieTimelineArchiver(HoodieWriteConfig config, HoodieTable<T, I, K, O> table) {
        this.config = config;
        this.table = table;
        this.metaClient = table.getMetaClient();
        this.archiveFilePath = HoodieArchivedTimeline.getArchiveLogPath((String)this.metaClient.getArchivePath());
        this.txnManager = new TransactionManager(config, (FileSystem)table.getMetaClient().getFs());
        Pair<Integer, Integer> minAndMaxInstants = ArchivalUtils.getMinAndMaxInstantsToKeep(table, this.metaClient);
        this.minInstantsToKeep = (Integer)minAndMaxInstants.getLeft();
        this.maxInstantsToKeep = (Integer)minAndMaxInstants.getRight();
    }

    private HoodieLogFormat.Writer openWriter() {
        try {
            if (this.writer == null) {
                return HoodieLogFormat.newWriterBuilder().onParentPath(this.archiveFilePath.getParent()).withFileId(this.archiveFilePath.getName()).withFileExtension(".archive").withFs((FileSystem)this.metaClient.getFs()).overBaseCommit("").build();
            }
            return this.writer;
        }
        catch (IOException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", (Throwable)e);
        }
    }

    public HoodieLogFormat.Writer reOpenWriter() {
        try {
            if (this.writer != null) {
                this.writer.close();
                this.writer = null;
            }
            this.writer = this.openWriter();
            return this.writer;
        }
        catch (IOException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", (Throwable)e);
        }
    }

    private void close() {
        try {
            if (this.writer != null) {
                this.writer.close();
            }
        }
        catch (IOException e) {
            throw new HoodieException("Unable to close HoodieLogFormat writer", (Throwable)e);
        }
    }

    public boolean archiveIfRequired(HoodieEngineContext context) throws IOException {
        return this.archiveIfRequired(context, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean archiveIfRequired(HoodieEngineContext context, boolean acquireLock) throws IOException {
        try {
            if (acquireLock) {
                this.txnManager.beginTransaction((Option<HoodieInstant>)Option.empty(), (Option<HoodieInstant>)Option.empty());
            }
            List<HoodieInstant> instantsToArchive = this.getInstantsToArchive().collect(Collectors.toList());
            this.verifyLastMergeArchiveFilesIfNecessary(context);
            boolean success = true;
            if (!instantsToArchive.isEmpty()) {
                this.writer = this.openWriter();
                LOG.info("Archiving instants " + instantsToArchive);
                this.archive(context, instantsToArchive);
                LOG.info("Deleting archived instants " + instantsToArchive);
                success = this.deleteArchivedInstants(instantsToArchive, context);
            } else {
                LOG.info("No Instants to archive");
            }
            if (this.shouldMergeSmallArchiveFiles()) {
                this.mergeArchiveFilesIfNecessary(context);
            }
            boolean bl = success;
            return bl;
        }
        finally {
            this.close();
            if (acquireLock) {
                this.txnManager.endTransaction((Option<HoodieInstant>)Option.empty());
            }
        }
    }

    public boolean shouldMergeSmallArchiveFiles() {
        return this.config.getArchiveMergeEnable() && !StorageSchemes.isAppendSupported((String)this.metaClient.getFs().getScheme());
    }

    private void mergeArchiveFilesIfNecessary(HoodieEngineContext context) throws IOException {
        Path planPath = new Path(this.metaClient.getArchivePath(), "mergeArchivePlan");
        this.reOpenWriter();
        FileStatus[] fsStatuses = this.metaClient.getFs().globStatus(new Path(this.metaClient.getArchivePath() + "/.commits_.archive*"));
        Arrays.sort(fsStatuses, new HoodieArchivedTimeline.ArchiveFileVersionComparator());
        int archiveMergeFilesBatchSize = this.config.getArchiveMergeFilesBatchSize();
        long smallFileLimitBytes = this.config.getArchiveMergeSmallFileLimitBytes();
        List<FileStatus> mergeCandidate = this.getMergeCandidates(smallFileLimitBytes, fsStatuses);
        if (mergeCandidate.size() >= archiveMergeFilesBatchSize) {
            List<String> candidateFiles = mergeCandidate.stream().map(fs -> fs.getPath().toString()).collect(Collectors.toList());
            String logFileName = this.computeLogFileName();
            this.buildArchiveMergePlan(candidateFiles, planPath, logFileName);
            this.mergeArchiveFiles(mergeCandidate);
            this.deleteFilesParallelize(this.metaClient, candidateFiles, context, true);
            LOG.info("Success to delete replaced small archive files.");
            this.metaClient.getFs().delete(planPath, false);
            LOG.info("Success to merge small archive files.");
        }
    }

    private List<FileStatus> getMergeCandidates(long smallFileLimitBytes, FileStatus[] fsStatuses) {
        int index;
        for (index = 0; index < fsStatuses.length && fsStatuses[index].getLen() <= smallFileLimitBytes; ++index) {
        }
        return Arrays.stream(fsStatuses).limit(index).collect(Collectors.toList());
    }

    private String computeLogFileName() throws IOException {
        String logWriteToken = this.writer.getLogFile().getLogWriteToken();
        HoodieLogFile hoodieLogFile = this.writer.getLogFile().rollOver((FileSystem)this.metaClient.getFs(), logWriteToken);
        return hoodieLogFile.getFileName();
    }

    private void verifyLastMergeArchiveFilesIfNecessary(HoodieEngineContext context) throws IOException {
        if (this.shouldMergeSmallArchiveFiles()) {
            Path planPath = new Path(this.metaClient.getArchivePath(), "mergeArchivePlan");
            HoodieWrapperFileSystem fs = this.metaClient.getFs();
            if (fs.exists(planPath)) {
                HoodieMergeArchiveFilePlan plan = null;
                try {
                    plan = (HoodieMergeArchiveFilePlan)TimelineMetadataUtils.deserializeAvroMetadata((byte[])((byte[])FileIOUtils.readDataFromPath((FileSystem)fs, (Path)planPath).get()), HoodieMergeArchiveFilePlan.class);
                }
                catch (IOException e) {
                    LOG.warn("Parsing merge archive plan failed.", (Throwable)e);
                    fs.delete(planPath);
                    return;
                }
                Path mergedArchiveFile = new Path(this.metaClient.getArchivePath(), plan.getMergedArchiveFileName());
                List<Path> candidates = plan.getCandidate().stream().map(Path::new).collect(Collectors.toList());
                if (this.candidateAllExists(candidates)) {
                    if (fs.exists(mergedArchiveFile)) {
                        fs.delete(mergedArchiveFile, false);
                    }
                } else if (fs.exists(mergedArchiveFile)) {
                    this.deleteFilesParallelize(this.metaClient, plan.getCandidate(), context, true);
                }
                fs.delete(planPath);
            }
        }
    }

    private boolean candidateAllExists(List<Path> candidates) throws IOException {
        for (Path archiveFile : candidates) {
            if (this.metaClient.getFs().exists(archiveFile)) continue;
            return false;
        }
        return true;
    }

    public void buildArchiveMergePlan(List<String> compactCandidate, Path planPath, String compactedArchiveFileName) throws IOException {
        LOG.info("Start to build archive merge plan.");
        HoodieMergeArchiveFilePlan plan = HoodieMergeArchiveFilePlan.newBuilder().setCandidate(compactCandidate).setMergedArchiveFileName(compactedArchiveFileName).build();
        Option content = TimelineMetadataUtils.serializeAvroMetadata((SpecificRecordBase)plan, HoodieMergeArchiveFilePlan.class);
        FileIOUtils.createFileInPath((FileSystem)this.metaClient.getFs(), (Path)planPath, (Option)content);
        LOG.info("Success to build archive merge plan");
    }

    public void mergeArchiveFiles(List<FileStatus> compactCandidate) throws IOException {
        LOG.info("Starting to merge small archive files.");
        Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
        try {
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (FileStatus fs : compactCandidate) {
                HoodieLogFormat.Reader reader = HoodieLogFormat.newReader((FileSystem)this.metaClient.getFs(), (HoodieLogFile)new HoodieLogFile(fs.getPath()), (Schema)HoodieArchivedMetaEntry.getClassSchema());
                Throwable throwable = null;
                try {
                    while (reader.hasNext()) {
                        HoodieAvroDataBlock blk = (HoodieAvroDataBlock)reader.next();
                        blk.getRecordIterator(HoodieRecord.HoodieRecordType.AVRO).forEachRemaining(r -> records.add((IndexedRecord)r.getData()));
                        if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                        this.writeToFile(wrapperSchema, records);
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (reader == null) continue;
                    if (throwable != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    reader.close();
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to merge small archive files", e);
        }
        finally {
            this.writer.close();
        }
        LOG.info("Success to merge small archive files.");
    }

    private Map<String, Boolean> deleteFilesParallelize(HoodieTableMetaClient metaClient, List<String> paths, HoodieEngineContext context, boolean ignoreFailed) {
        return FSUtils.parallelizeFilesProcess((HoodieEngineContext)context, (FileSystem)metaClient.getFs(), (int)this.config.getArchiveDeleteParallelism(), (FSUtils.SerializableFunction & Serializable)pairOfSubPathAndConf -> {
            Path file = new Path((String)pairOfSubPathAndConf.getKey());
            try {
                HoodieWrapperFileSystem fs = metaClient.getFs();
                if (fs.exists(file)) {
                    return fs.delete(file, false);
                }
                return true;
            }
            catch (IOException e) {
                if (!ignoreFailed) {
                    throw new HoodieIOException("Failed to delete : " + file, e);
                }
                LOG.warn("Ignore failed deleting : " + file);
                return true;
            }
        }, paths);
    }

    private Stream<HoodieInstant> getCleanInstantsToArchive() {
        HoodieTimeline cleanAndRollbackTimeline = this.table.getActiveTimeline().getTimelineOfActions(CollectionUtils.createSet((Object[])new String[]{"clean", "rollback"})).filterCompletedInstants();
        return cleanAndRollbackTimeline.getInstantsAsStream().collect(Collectors.groupingBy(HoodieInstant::getAction)).values().stream().map(hoodieInstants -> {
            if (hoodieInstants.size() > this.maxInstantsToKeep) {
                return hoodieInstants.subList(0, hoodieInstants.size() - this.minInstantsToKeep);
            }
            return new ArrayList();
        }).flatMap(Collection::stream);
    }

    private Stream<HoodieInstant> getCommitInstantsToArchive() throws IOException {
        Option completedCommitBeforeOldestPendingInstant;
        HoodieTimeline commitTimeline = this.table.getCompletedCommitsTimeline();
        Option oldestPendingInstant = this.table.getActiveTimeline().getWriteTimeline().filter(instant -> !instant.isCompleted()).firstInstant();
        Option oldestCommitToRetain = oldestPendingInstant.isPresent() ? (!(completedCommitBeforeOldestPendingInstant = Option.fromJavaOptional(commitTimeline.getReverseOrderedInstants().filter(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)((HoodieInstant)oldestPendingInstant.get()).getTimestamp())).findFirst())).isPresent() || HoodieTimeline.compareTimestamps((String)((HoodieInstant)oldestPendingInstant.get()).getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)((HoodieInstant)completedCommitBeforeOldestPendingInstant.get()).getTimestamp()) ? oldestPendingInstant : completedCommitBeforeOldestPendingInstant) : Option.empty();
        Option firstSavepoint = this.table.getCompletedSavepointTimeline().firstInstant();
        Set<String> savepointTimestamps = this.table.getSavepointTimestamps();
        if (!commitTimeline.empty() && commitTimeline.countInstants() > this.maxInstantsToKeep) {
            Option oldestInstantToRetainForCompaction = this.metaClient.getTableType() == HoodieTableType.MERGE_ON_READ && (this.config.getInlineCompactTriggerStrategy() == CompactionTriggerStrategy.NUM_COMMITS || this.config.getInlineCompactTriggerStrategy() == CompactionTriggerStrategy.NUM_AND_TIME) ? CompactionUtils.getOldestInstantToRetainForCompaction((HoodieActiveTimeline)this.table.getActiveTimeline(), (int)this.config.getInlineCompactDeltaCommitMax()) : Option.empty();
            Option oldestInstantToRetainForClustering = ClusteringUtils.getOldestInstantToRetainForClustering((HoodieActiveTimeline)this.table.getActiveTimeline(), (HoodieTableMetaClient)this.table.getMetaClient());
            Stream<HoodieInstant> instantToArchiveStream = commitTimeline.getInstantsAsStream().filter(s -> {
                if (this.config.shouldArchiveBeyondSavepoint()) {
                    return !savepointTimestamps.contains(s.getTimestamp());
                }
                return !firstSavepoint.isPresent() || !HoodieTimeline.compareTimestamps((String)((HoodieInstant)firstSavepoint.get()).getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN_OR_EQUALS, (String)s.getTimestamp());
            }).filter(s -> (Boolean)oldestCommitToRetain.map(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.GREATER_THAN, (String)s.getTimestamp())).orElse((Object)true)).filter(s -> (Boolean)oldestInstantToRetainForCompaction.map(instantToRetain -> HoodieTimeline.compareTimestamps((String)s.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)instantToRetain.getTimestamp())).orElse((Object)true)).filter(s -> (Boolean)oldestInstantToRetainForClustering.map(instantToRetain -> HoodieTimeline.compareTimestamps((String)s.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)instantToRetain.getTimestamp())).orElse((Object)true));
            return instantToArchiveStream.limit(commitTimeline.countInstants() - this.minInstantsToKeep);
        }
        return Stream.empty();
    }

    private Stream<HoodieInstant> getInstantsToArchive() throws IOException {
        HoodieTableMetaClient dataMetaClient;
        Option qualifiedEarliestInstant;
        Stream<Object> instants = Stream.concat(this.getCleanInstantsToArchive(), this.getCommitInstantsToArchive());
        if (this.config.isMetaserverEnabled()) {
            return Stream.empty();
        }
        HoodieActiveTimeline rawActiveTimeline = new HoodieActiveTimeline(this.metaClient, false);
        Map<Pair, List<HoodieInstant>> groupByTsAction = rawActiveTimeline.getInstantsAsStream().collect(Collectors.groupingBy(i -> Pair.of((Object)i.getTimestamp(), (Object)HoodieInstant.getComparableAction((String)i.getAction()))));
        if (this.table.getMetaClient().getTableConfig().isMetadataTableAvailable()) {
            try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create((HoodieEngineContext)this.table.getContext(), (HoodieMetadataConfig)this.config.getMetadataConfig(), (String)this.config.getBasePath());){
                Option latestCompactionTime = tableMetadata.getLatestCompactionTime();
                if (!latestCompactionTime.isPresent()) {
                    LOG.info("Not archiving as there is no compaction yet on the metadata table");
                    instants = Stream.empty();
                } else {
                    LOG.info("Limiting archiving of instants to latest compaction on metadata table at " + (String)latestCompactionTime.get());
                    instants = instants.filter(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)((String)latestCompactionTime.get())));
                }
            }
            catch (Exception e) {
                throw new HoodieException("Error limiting instant archival based on metadata table", (Throwable)e);
            }
        }
        if (this.table.isMetadataTable() && (qualifiedEarliestInstant = TimelineUtils.getEarliestInstantForMetadataArchival((HoodieActiveTimeline)(dataMetaClient = HoodieTableMetaClient.builder().setBasePath(HoodieTableMetadata.getDatasetBasePath((String)this.config.getBasePath())).setConf(this.metaClient.getHadoopConf()).build()).getActiveTimeline(), (boolean)this.config.shouldArchiveBeyondSavepoint())).isPresent()) {
            instants = instants.filter(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)((HoodieInstant)qualifiedEarliestInstant.get()).getTimestamp()));
        }
        return instants.flatMap(hoodieInstant -> {
            List instantsToStream = (List)groupByTsAction.get(Pair.of((Object)hoodieInstant.getTimestamp(), (Object)HoodieInstant.getComparableAction((String)hoodieInstant.getAction())));
            if (instantsToStream != null) {
                return instantsToStream.stream();
            }
            return Stream.empty();
        });
    }

    private boolean deleteArchivedInstants(List<HoodieInstant> archivedInstants, HoodieEngineContext context) throws IOException {
        LOG.info("Deleting instants " + archivedInstants);
        ArrayList<HoodieInstant> pendingInstants = new ArrayList<HoodieInstant>();
        ArrayList<HoodieInstant> completedInstants = new ArrayList<HoodieInstant>();
        for (HoodieInstant instant2 : archivedInstants) {
            if (instant2.isCompleted()) {
                completedInstants.add(instant2);
                continue;
            }
            pendingInstants.add(instant2);
        }
        context.setJobStatus(this.getClass().getSimpleName(), "Delete archived instants: " + this.config.getTableName());
        HoodieActiveTimeline activeTimeline = this.metaClient.getActiveTimeline();
        if (!pendingInstants.isEmpty()) {
            context.foreach(pendingInstants, (SerializableConsumer & Serializable)instant -> activeTimeline.deleteInstantFileIfExists(instant), Math.min(pendingInstants.size(), this.config.getArchiveDeleteParallelism()));
        }
        if (!completedInstants.isEmpty()) {
            context.foreach(completedInstants, (SerializableConsumer & Serializable)instant -> activeTimeline.deleteInstantFileIfExists(instant), Math.min(completedInstants.size(), this.config.getArchiveDeleteParallelism()));
        }
        return true;
    }

    public void archive(HoodieEngineContext context, List<HoodieInstant> instants) throws HoodieCommitException {
        try {
            Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
            LOG.info("Wrapper schema " + wrapperSchema.toString());
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (HoodieInstant hoodieInstant : instants) {
                try {
                    this.deleteAnyLeftOverMarkers(context, hoodieInstant);
                    records.add(this.convertToAvroRecord(hoodieInstant));
                    if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                    this.writeToFile(wrapperSchema, records);
                }
                catch (Exception e) {
                    LOG.error("Failed to archive commits, .commit file: " + hoodieInstant.getFileName(), (Throwable)e);
                    if (!this.config.isFailOnTimelineArchivingEnabled()) continue;
                    throw e;
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to archive commits", e);
        }
    }

    private void deleteAnyLeftOverMarkers(HoodieEngineContext context, HoodieInstant instant) {
        WriteMarkers writeMarkers = WriteMarkersFactory.get(this.config.getMarkersType(), this.table, instant.getTimestamp());
        if (writeMarkers.deleteMarkerDir(context, this.config.getMarkersDeleteParallelism())) {
            LOG.info("Cleaned up left over marker directory for instant :" + instant);
        }
    }

    private void writeToFile(Schema wrapperSchema, List<IndexedRecord> records) throws Exception {
        if (records.size() > 0) {
            HashMap<HoodieLogBlock.HeaderMetadataType, String> header = new HashMap<HoodieLogBlock.HeaderMetadataType, String>();
            header.put(HoodieLogBlock.HeaderMetadataType.SCHEMA, wrapperSchema.toString());
            String keyField = this.table.getMetaClient().getTableConfig().getRecordKeyFieldProp();
            List indexRecords = records.stream().map(HoodieAvroIndexedRecord::new).collect(Collectors.toList());
            HoodieAvroDataBlock block = new HoodieAvroDataBlock(indexRecords, header, keyField);
            this.writer.appendBlock((HoodieLogBlock)block);
            records.clear();
        }
    }

    private IndexedRecord convertToAvroRecord(HoodieInstant hoodieInstant) throws IOException {
        return MetadataConversionUtils.createMetaWrapper(hoodieInstant, this.metaClient);
    }
}

