/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.ozone;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.ReadOnlyBufferException;
import java.util.Arrays;
import java.util.EnumSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.GlobalStorageStatistics;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageStatistics;
import org.apache.hadoop.fs.ozone.OzoneClientAdapter;
import org.apache.hadoop.fs.ozone.OzoneFSOutputStream;
import org.apache.hadoop.fs.ozone.OzoneFileSystem;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

public class TestReadWriteStatistics {
    private Path aPath = new Path("/afile");
    private byte[] buff = new byte[512];
    private static final String O3FS_STORAGE_STAT_NAME = "o3fs";
    private static final String STAT_NAME_BYTES_READ = "bytesRead";
    private static final String STAT_NAME_BYTES_WRITTEN = "bytesWritten";
    private static final String STAT_NAME_READ_OPS = "readOps";
    private static final String STAT_NAME_LARGE_READ_OPS = "largeReadOps";
    private static final String STAT_NAME_WRITE_OPS = "writeOps";
    private OzoneFileSystem fs = (OzoneFileSystem)Mockito.spy((Object)new OzoneFileSystem());
    private OzoneClientAdapter fakeAdapter = (OzoneClientAdapter)Mockito.mock(OzoneClientAdapter.class);
    private InputStream fakeInputStream = (InputStream)Mockito.mock(InputStream.class);
    private OutputStream fakeOutputStream = (OutputStream)Mockito.mock(OutputStream.class);

    @Test
    public void testZeroBytesReadWhenExceptionWasThrown() throws Exception {
        this.setupFakeInputStreamToThrowIOExceptionOnRead();
        FSDataInputStream stream = this.fs.open(this.aPath);
        try {
            stream.read();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.assertBytesReadAndReadNumOps(0L, 1L);
    }

    @Test
    public void testZeroBytesReadWhenEOFReached() throws Exception {
        this.setupFakeInputStreamToReadByte(-1);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read();
        this.assertBytesReadAndReadNumOps(0L, 1L);
    }

    @Test
    public void testOneByteReadOnSingleReadCall() throws Exception {
        this.setupFakeInputStreamToReadByte(20);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read();
        this.assertBytesReadAndReadNumOps(1L, 1L);
    }

    @Test
    public void testConsecutiveReadsIncreaseStats() throws Exception {
        this.setupFakeInputStreamToReadByte(20);
        FSDataInputStream stream = this.fs.open(this.aPath);
        for (int i = 1; i <= 5; ++i) {
            stream.read();
            this.assertBytesReadAndReadNumOps(i, 1L);
        }
    }

    @Test
    public void testConsecutiveOpensAndReadsIncreaseStats() throws Exception {
        this.setupFakeInputStreamToReadByte(20);
        for (int i = 0; i < 5; ++i) {
            FSDataInputStream stream = this.fs.open(this.aPath);
            stream.read();
            stream.close();
            this.assertBytesReadAndReadNumOps(i + 1, i + 1);
        }
    }

    @Test
    public void testConsecutiveOpensIncreaseStats() throws Exception {
        this.setupFakeInputStreamToReadByte(20);
        for (int i = 1; i <= 5; ++i) {
            FSDataInputStream stream = this.fs.open(this.aPath);
            stream.close();
            this.assertBytesReadAndReadNumOps(0L, i);
        }
    }

    @Test
    public void testZeroBytesReadOnMultiByteReadWhenExceptionWasThrown() throws Exception {
        this.setupFakeInputStreamToThrowExceptionOnMultiByteRead();
        FSDataInputStream stream = this.fs.open(this.aPath);
        try {
            stream.read(this.buff, 0, this.buff.length);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.assertBytesReadAndReadNumOps(0L, 1L);
    }

    @Test
    public void testZeroBytesReadOnMultiByteReadWhenEOFReachedAtStart() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(-1);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read(this.buff, 0, this.buff.length);
        this.assertBytesReadAndReadNumOps(0L, 1L);
    }

    @Test
    public void testEOFBeforeLengthOnMultiByteRead() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(256);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read(this.buff, 0, this.buff.length);
        this.assertBytesReadAndReadNumOps(256L, 1L);
    }

    @Test
    public void testFullyReadBufferOnMultiByteRead() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(this.buff.length);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read(this.buff, 0, this.buff.length);
        this.assertBytesReadAndReadNumOps(this.buff.length, 1L);
    }

    @Test
    public void testConsecutiveReadsToBufferOnMultiByteRead() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(this.buff.length);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read(this.buff, 0, this.buff.length);
        stream.read(this.buff, 0, this.buff.length);
        this.assertBytesReadAndReadNumOps(2 * this.buff.length, 1L);
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(256);
        stream.read(this.buff, 0, 256);
        this.assertBytesReadAndReadNumOps(2 * this.buff.length + 256, 1L);
    }

    @Test
    public void testZeroBytesWrittenWhenExceptionWasThrown() throws Exception {
        this.setupFakeOutputStreamToThrowIOExceptionOnWrite();
        FSDataOutputStream stream = this.fs.create(this.aPath);
        try {
            stream.write(20);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.assertBytesWrittenAndWriteNumOps(0L, 1L);
    }

    @Test
    public void testOneByteWrittenOnSingleWriteCall() throws Exception {
        FSDataOutputStream stream = this.fs.create(this.aPath);
        stream.write(20);
        this.assertBytesWrittenAndWriteNumOps(1L, 1L);
    }

    @Test
    public void testConsecutiveWritesIncreaseStats() throws Exception {
        FSDataOutputStream stream = this.fs.create(this.aPath);
        for (int i = 1; i <= 5; ++i) {
            stream.write(20);
            this.assertBytesWrittenAndWriteNumOps(i, 1L);
        }
    }

    @Test
    public void testConsecutiveCreatesAndWritesIncreaseStats() throws Exception {
        for (int i = 1; i <= 5; ++i) {
            FSDataOutputStream stream = this.fs.create(this.aPath);
            stream.write(20);
            this.assertBytesWrittenAndWriteNumOps(i, i);
        }
    }

    @Test
    public void testConsecutiveCreatesIncreaseStats() throws Exception {
        for (int i = 1; i <= 5; ++i) {
            this.fs.create(this.aPath);
            this.assertBytesWrittenAndWriteNumOps(0L, i);
        }
    }

    @Test
    public void testBufferReadCallsIncreaseStatistics() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(128);
        ByteBuffer buffer = ByteBuffer.wrap(this.buff);
        FSDataInputStream stream = this.fs.open(this.aPath);
        stream.read(buffer);
        this.assertBytesReadAndReadNumOps(128L, 1L);
    }

    @Test
    public void testReadToReadOnlyBufferDoesNotChangeStats() throws Exception {
        this.setupFakeInputStreamToReadNumBytesOnMultiByteRead(128);
        ByteBuffer buffer = ByteBuffer.wrap(this.buff).asReadOnlyBuffer();
        FSDataInputStream stream = this.fs.open(this.aPath);
        try {
            stream.read(buffer);
        }
        catch (ReadOnlyBufferException readOnlyBufferException) {
            // empty catch block
        }
        this.assertBytesReadAndReadNumOps(0L, 1L);
    }

    @Test
    public void testZeroBytesWrittenOnMultiByteWriteWhenExceptionWasThrown() throws Exception {
        this.setupFakeOutputStreamToThrowIOExceptionOnMultiByteWrite();
        FSDataOutputStream stream = this.fs.create(this.aPath);
        try {
            stream.write(this.buff, 0, this.buff.length);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.assertBytesWrittenAndWriteNumOps(0L, 1L);
    }

    @Test
    public void testBufferFullyWrittenOnMultiByteWrite() throws Exception {
        FSDataOutputStream stream = this.fs.create(this.aPath);
        stream.write(this.buff, 0, this.buff.length);
        this.assertBytesWrittenAndWriteNumOps(this.buff.length, 1L);
    }

    @Test
    public void testBufferPartiallyWrittenOnMultiByteWrite() throws Exception {
        FSDataOutputStream stream = this.fs.create(this.aPath);
        stream.write(this.buff, this.buff.length / 2, this.buff.length / 4);
        this.assertBytesWrittenAndWriteNumOps(this.buff.length / 4, 1L);
    }

    @Test
    public void testConsecutiveMultiByteWritesIncreaseStats() throws Exception {
        FSDataOutputStream stream = this.fs.create(this.aPath);
        for (int i = 1; i <= 5; ++i) {
            stream.write(this.buff, 0, this.buff.length);
            this.assertBytesWrittenAndWriteNumOps(i * this.buff.length, 1L);
        }
        stream.write(this.buff, 0, 128);
        this.assertBytesWrittenAndWriteNumOps(128 + 5 * this.buff.length, 1L);
    }

    @Test
    public void testNonRecursiveCreateIncreaseStats() throws Exception {
        EnumSet<CreateFlag> flags = EnumSet.of(CreateFlag.OVERWRITE);
        for (int i = 1; i <= 5; ++i) {
            FSDataOutputStream stream = this.fs.createNonRecursive(this.aPath, null, flags, 512, (short)3, 512L, null);
            this.assertBytesWrittenAndWriteNumOps(0L, i);
        }
    }

    @Test(expected=UnsupportedOperationException.class)
    public void testsIfAppendGetsSupported() throws Exception {
        this.fs.append(this.aPath, 512, null);
        Assert.fail((String)"Add tests to cover metrics changes on append!");
    }

    private long readValueFromFSStatistics(String valueName) {
        GlobalStorageStatistics stats = FileSystem.getGlobalStorageStatistics();
        StorageStatistics fsStats = stats.get(O3FS_STORAGE_STAT_NAME);
        return fsStats.getLong(valueName);
    }

    private void assertBytesReadAndReadNumOps(long expectedBytesRead, long expectedNumReadOps) {
        long bytesRead = this.readValueFromFSStatistics(STAT_NAME_BYTES_READ);
        long numReadOps = this.readValueFromFSStatistics(STAT_NAME_READ_OPS);
        Assert.assertEquals((String)"Bytes read.", (long)expectedBytesRead, (long)bytesRead);
        Assert.assertEquals((String)"Read op count.", (long)expectedNumReadOps, (long)numReadOps);
    }

    private void assertBytesWrittenAndWriteNumOps(long expectedBytesWritten, long expectedNumWriteOps) {
        long bytesWritten = this.readValueFromFSStatistics(STAT_NAME_BYTES_WRITTEN);
        long numWriteOps = this.readValueFromFSStatistics(STAT_NAME_WRITE_OPS);
        Assert.assertEquals((String)"Bytes written.", (long)expectedBytesWritten, (long)bytesWritten);
        Assert.assertEquals((String)"Write op count.", (long)expectedNumWriteOps, (long)numWriteOps);
    }

    @Before
    public void setupMocks() throws Exception {
        this.setupAdapterToReturnFakeInputStreamOnReadFile();
        this.setupAdapterToReturnFakeOutputStreamOnCreate();
        this.setupFileSystemToUseFakeClientAdapter();
        this.initializeFS();
        Arrays.fill(this.buff, (byte)20);
    }

    private void setupAdapterToReturnFakeInputStreamOnReadFile() throws IOException {
        Mockito.when((Object)this.fakeAdapter.readFile(Mockito.anyString())).thenReturn((Object)this.fakeInputStream);
    }

    private void setupAdapterToReturnFakeOutputStreamOnCreate() throws Exception {
        Mockito.when((Object)this.fakeAdapter.createFile(Mockito.anyString(), Mockito.anyBoolean(), Mockito.anyBoolean())).thenReturn((Object)new OzoneFSOutputStream(this.fakeOutputStream));
    }

    private void setupFileSystemToUseFakeClientAdapter() throws IOException {
        ((OzoneFileSystem)Mockito.doReturn((Object)this.fakeAdapter).when((Object)this.fs)).createAdapter((Configuration)Mockito.any(Configuration.class), Mockito.anyString(), Mockito.anyString(), Mockito.anyString(), Mockito.anyInt(), Mockito.anyBoolean());
    }

    private void initializeFS() throws IOException, URISyntaxException {
        FileSystem.getGlobalStorageStatistics().reset();
        URI fsUri = new URI("o3fs://volume.bucket.localhost");
        Configuration conf = new Configuration();
        this.fs.initialize(fsUri, conf);
    }

    private void setupFakeInputStreamToThrowIOExceptionOnRead() throws IOException {
        Mockito.when((Object)this.fakeInputStream.read()).thenThrow(new Throwable[]{new IOException("Simulated IOE")});
    }

    private void setupFakeInputStreamToReadByte(int byteToReturn) throws IOException {
        Mockito.when((Object)this.fakeInputStream.read()).thenReturn((Object)byteToReturn);
    }

    private void setupFakeInputStreamToThrowExceptionOnMultiByteRead() throws Exception {
        Mockito.when((Object)this.fakeInputStream.read((byte[])Mockito.any(byte[].class), Mockito.anyInt(), Mockito.anyInt())).thenThrow(new Throwable[]{new IOException("Simulated IOE")});
    }

    private void setupFakeInputStreamToReadNumBytesOnMultiByteRead(int numOfBytesToReturn) throws Exception {
        Mockito.when((Object)this.fakeInputStream.read((byte[])Mockito.any(byte[].class), Mockito.anyInt(), Mockito.anyInt())).thenReturn((Object)numOfBytesToReturn);
    }

    private void setupFakeOutputStreamToThrowIOExceptionOnWrite() throws Exception {
        ((OutputStream)Mockito.doThrow((Throwable)new IOException("Simulated IOE")).when((Object)this.fakeOutputStream)).write(Mockito.anyInt());
    }

    private void setupFakeOutputStreamToThrowIOExceptionOnMultiByteWrite() throws Exception {
        ((OutputStream)Mockito.doThrow((Throwable)new IOException("Simulated IOE")).when((Object)this.fakeOutputStream)).write((byte[])Mockito.any(byte[].class), Mockito.anyInt(), Mockito.anyInt());
    }
}

