/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.giraph.comm.requests;

import org.apache.giraph.comm.netty.SaslNettyServer;
import org.apache.log4j.Logger;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

/**
 * Send and receive SASL tokens.
 */
public class SaslTokenMessageRequest extends WritableRequest {
  /** Class logger */
  private static final Logger LOG =
      Logger.getLogger(SaslTokenMessageRequest.class);

  /** Used for client or server's token to send or receive from each other. */
  private byte[] token;

  /**
   * Constructor used for reflection only.
   */
  public SaslTokenMessageRequest() { }

 /**
   * Constructor used to send request.
   *
   * @param token the SASL token, generated by a SaslClient or SaslServer.
   */
  public SaslTokenMessageRequest(byte[] token) {
    this.token = token;
  }

  /**
   * Read accessor for SASL token
   *
   * @return saslToken SASL token
   */
  public byte[] getSaslToken() {
    return token;
  }

  /**
   * Write accessor for SASL token
   *
   * @param token SASL token
   */
  public void setSaslToken(byte[] token) {
    this.token = token;
  }

  @Override
  public RequestType getType() {
    return RequestType.SASL_TOKEN_MESSAGE_REQUEST;
  }

  @Override
  public void readFieldsRequest(DataInput input) throws IOException {
    int tokenSize = input.readInt();
    token = new byte[tokenSize];
    input.readFully(token);
  }

  /**
   * Update server's token in response to the SASL token received from
   * client. Updated token is sent to client by
   * SaslServerHandler.messageReceived().
   *
   * @param saslNettyServer used to create response.
   */

  public void processToken(SaslNettyServer saslNettyServer) {
    if (LOG.isDebugEnabled()) {
      LOG.debug("processToken:  With nettyServer: " + saslNettyServer +
          " and token length: " + token.length);
    }
    token = saslNettyServer.response(token);
    if (LOG.isDebugEnabled()) {
      LOG.debug("processToken: Response token's length is:" + token.length);
    }
  }

  @Override
  public void writeRequest(DataOutput output) throws IOException {
    output.writeInt(token.length);
    output.write(token);
  }

  @Override
  public int getSerializedSize() {
    return super.getSerializedSize() + 4 + token.length;
  }
}
