/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geaflow.cluster.clustermanager;

import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.geaflow.cluster.clustermanager.ClusterContext;
import org.apache.geaflow.cluster.clustermanager.ClusterId;
import org.apache.geaflow.cluster.clustermanager.ContainerExecutorInfo;
import org.apache.geaflow.cluster.clustermanager.ExecutorRegisterException;
import org.apache.geaflow.cluster.clustermanager.ExecutorRegisteredCallback;
import org.apache.geaflow.cluster.clustermanager.IClusterManager;
import org.apache.geaflow.cluster.config.ClusterConfig;
import org.apache.geaflow.cluster.constants.ClusterConstants;
import org.apache.geaflow.cluster.container.ContainerInfo;
import org.apache.geaflow.cluster.driver.DriverInfo;
import org.apache.geaflow.cluster.failover.IFailoverStrategy;
import org.apache.geaflow.cluster.protocol.OpenContainerEvent;
import org.apache.geaflow.cluster.protocol.OpenContainerResponseEvent;
import org.apache.geaflow.cluster.rpc.ConnectAddress;
import org.apache.geaflow.cluster.rpc.RpcClient;
import org.apache.geaflow.cluster.rpc.RpcEndpointRef;
import org.apache.geaflow.cluster.rpc.RpcEndpointRefFactory;
import org.apache.geaflow.cluster.rpc.RpcUtil;
import org.apache.geaflow.cluster.rpc.impl.ContainerEndpointRef;
import org.apache.geaflow.common.config.Configuration;
import org.apache.geaflow.common.serialize.SerializerFactory;
import org.apache.geaflow.common.utils.FutureUtil;
import org.apache.geaflow.rpc.proto.Container;
import org.apache.geaflow.rpc.proto.Master;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractClusterManager
implements IClusterManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractClusterManager.class);
    protected String masterId;
    protected ClusterConfig clusterConfig;
    protected ClusterContext clusterContext;
    protected Configuration config;
    protected ClusterId clusterInfo;
    protected Map<Integer, ContainerInfo> containerInfos;
    protected Map<Integer, DriverInfo> driverInfos;
    protected Map<Integer, Future<DriverInfo>> driverFutureMap;
    protected IFailoverStrategy foStrategy;
    protected long driverTimeoutSec;
    private AtomicInteger idGenerator;

    @Override
    public void init(ClusterContext clusterContext) {
        this.config = clusterContext.getConfig();
        this.clusterConfig = clusterContext.getClusterConfig();
        this.driverTimeoutSec = this.clusterConfig.getDriverRegisterTimeoutSec();
        this.containerInfos = new ConcurrentHashMap<Integer, ContainerInfo>();
        this.driverInfos = new ConcurrentHashMap<Integer, DriverInfo>();
        this.clusterContext = clusterContext;
        this.idGenerator = new AtomicInteger(clusterContext.getMaxComponentId());
        this.masterId = clusterContext.getConfig().getMasterId();
        Preconditions.checkNotNull((Object)this.masterId, (Object)"masterId is not set");
        this.foStrategy = this.buildFailoverStrategy();
        this.driverFutureMap = new ConcurrentHashMap<Integer, Future<DriverInfo>>();
        if (clusterContext.isRecover()) {
            for (Integer driverId : clusterContext.getDriverIds().keySet()) {
                this.driverFutureMap.put(driverId, new CompletableFuture());
            }
        }
        RpcClient.init(clusterContext.getConfig());
    }

    @Override
    public void allocateWorkers(int workerNum) {
        int workersPerContainer = this.clusterConfig.getContainerWorkerNum();
        int containerNum = (workerNum + workersPerContainer - 1) / workersPerContainer;
        LOGGER.info("allocate {} containers with {} workers", (Object)containerNum, (Object)workerNum);
        this.startContainers(containerNum);
        this.doCheckpoint();
    }

    protected void startContainers(int containerNum) {
        this.validateContainerNum(containerNum);
        HashMap<Integer, String> containerIds = new HashMap<Integer, String>();
        for (int i = 0; i < containerNum; ++i) {
            int containerId = this.generateNextComponentId();
            this.createNewContainer(containerId, false);
            containerIds.put(containerId, ClusterConstants.getContainerName(containerId));
        }
        this.clusterContext.getContainerIds().putAll(containerIds);
    }

    @Override
    public Map<String, ConnectAddress> startDrivers() {
        int driverNum = this.clusterConfig.getDriverNum();
        LOGGER.info("start driver number: {}", (Object)driverNum);
        if (!this.clusterContext.isRecover()) {
            HashMap<Integer, String> driverIds = new HashMap<Integer, String>();
            for (int driverIndex = 0; driverIndex < driverNum; ++driverIndex) {
                int driverId = this.generateNextComponentId();
                this.driverFutureMap.put(driverId, new CompletableFuture());
                this.createNewDriver(driverId, driverIndex);
                driverIds.put(driverId, ClusterConstants.getDriverName(driverId));
            }
            this.clusterContext.getDriverIds().putAll(driverIds);
            this.doCheckpoint();
        }
        HashMap<String, ConnectAddress> driverAddresses = new HashMap<String, ConnectAddress>(driverNum);
        List driverInfoList = FutureUtil.wait(this.driverFutureMap.values(), (long)this.driverTimeoutSec, (TimeUnit)TimeUnit.SECONDS);
        driverInfoList.forEach(driverInfo -> driverAddresses.put(driverInfo.getName(), new ConnectAddress(driverInfo.getHost(), driverInfo.getRpcPort())));
        return driverAddresses;
    }

    public void restartAllDrivers() {
        Map<Integer, String> driverIds = this.clusterContext.getDriverIds();
        LOGGER.info("Restart all drivers: {}", driverIds);
        for (Map.Entry<Integer, String> entry : driverIds.entrySet()) {
            this.restartDriver(entry.getKey());
        }
    }

    public void restartAllContainers() {
        Map<Integer, String> containerIds = this.clusterContext.getContainerIds();
        LOGGER.info("Restart all containers: {}", containerIds);
        for (Map.Entry<Integer, String> entry : containerIds.entrySet()) {
            this.restartContainer(entry.getKey());
        }
    }

    public abstract void restartDriver(int var1);

    public abstract void restartContainer(int var1);

    protected abstract void createNewDriver(int var1, int var2);

    protected abstract void createNewContainer(int var1, boolean var2);

    protected abstract IFailoverStrategy buildFailoverStrategy();

    protected void validateContainerNum(int containerNum) {
    }

    @Override
    public void doFailover(int componentId, Throwable cause) {
        this.foStrategy.doFailover(componentId, cause);
    }

    @Override
    public void close() {
        if (this.clusterInfo != null) {
            LOGGER.info("close master {}", (Object)this.masterId);
            RpcClient.getInstance().closeMasterConnection(this.masterId);
        }
        for (ContainerInfo containerInfo : this.containerInfos.values()) {
            LOGGER.info("close container {}", (Object)containerInfo.getName());
            RpcClient.getInstance().closeContainerConnection(containerInfo.getName());
        }
        for (DriverInfo driverInfo : this.driverInfos.values()) {
            LOGGER.info("close driver {}", (Object)driverInfo.getName());
            RpcClient.getInstance().closeDriverConnection(driverInfo.getName());
        }
    }

    private int generateNextComponentId() {
        int id = this.idGenerator.incrementAndGet();
        this.clusterContext.setMaxComponentId(id);
        return id;
    }

    public Master.RegisterResponse registerContainer(ContainerInfo request) {
        LOGGER.info("register container:{}", (Object)request);
        this.containerInfos.put(request.getId(), request);
        RpcUtil.asyncExecute(() -> this.openContainer(request));
        return Master.RegisterResponse.newBuilder().setSuccess(true).build();
    }

    public Master.RegisterResponse registerDriver(DriverInfo driverInfo) {
        LOGGER.info("register driver:{}", (Object)driverInfo);
        this.driverInfos.put(driverInfo.getId(), driverInfo);
        CompletableFuture completableFuture = (CompletableFuture)this.driverFutureMap.get(driverInfo.getId());
        completableFuture.complete(driverInfo);
        return Master.RegisterResponse.newBuilder().setSuccess(true).build();
    }

    protected void openContainer(final ContainerInfo containerInfo) {
        ContainerEndpointRef endpointRef = RpcEndpointRefFactory.getInstance().connectContainer(containerInfo.getHost(), containerInfo.getRpcPort());
        final int workerNum = this.clusterConfig.getContainerWorkerNum();
        endpointRef.process(new OpenContainerEvent(workerNum), new RpcEndpointRef.RpcCallback<Container.Response>(){

            @Override
            public void onSuccess(Container.Response response) {
                byte[] payload = response.getPayload().toByteArray();
                OpenContainerResponseEvent openResult = (OpenContainerResponseEvent)SerializerFactory.getKryoSerializer().deserialize(payload);
                ContainerExecutorInfo executorInfo = new ContainerExecutorInfo(containerInfo, openResult.getFirstWorkerIndex(), workerNum);
                AbstractClusterManager.this.handleRegisterResponse(executorInfo, openResult, null);
            }

            @Override
            public void onFailure(Throwable t) {
                AbstractClusterManager.this.handleRegisterResponse(null, null, t);
            }
        });
    }

    private void handleRegisterResponse(ContainerExecutorInfo executorInfo, OpenContainerResponseEvent response, Throwable e) {
        List<ExecutorRegisteredCallback> callbacks = this.clusterContext.getCallbacks();
        if (e != null || !response.isSuccess()) {
            for (ExecutorRegisteredCallback callback : callbacks) {
                callback.onFailure(new ExecutorRegisterException(e));
            }
        } else {
            for (ExecutorRegisteredCallback callback : callbacks) {
                callback.onSuccess(executorInfo);
            }
        }
    }

    private synchronized void doCheckpoint() {
        this.clusterContext.checkpoint(new ClusterContext.ClusterCheckpointFunction());
    }

    public ClusterContext getClusterContext() {
        return this.clusterContext;
    }

    public int getTotalContainers() {
        return this.clusterContext.getContainerIds().size();
    }

    public int getTotalDrivers() {
        return this.clusterContext.getDriverIds().size();
    }

    public Map<Integer, ContainerInfo> getContainerInfos() {
        return this.containerInfos;
    }

    public Map<Integer, DriverInfo> getDriverInfos() {
        return this.driverInfos;
    }

    public Map<Integer, String> getContainerIds() {
        return this.clusterContext.getContainerIds();
    }

    public Map<Integer, String> getDriverIds() {
        return this.clusterContext.getDriverIds();
    }
}

