/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.sink.hbase;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.flume.Channel;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.EventDeliveryException;
import org.apache.flume.FlumeException;
import org.apache.flume.Sink;
import org.apache.flume.Transaction;
import org.apache.flume.conf.Configurable;
import org.apache.flume.instrumentation.SinkCounter;
import org.apache.flume.sink.AbstractSink;
import org.apache.flume.sink.hbase.HBaseSinkSecurityManager;
import org.apache.flume.sink.hbase.HbaseEventSerializer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HBaseSink
extends AbstractSink
implements Configurable {
    private String tableName;
    private byte[] columnFamily;
    private HTable table;
    private long batchSize;
    private Configuration config;
    private static final Logger logger = LoggerFactory.getLogger(HBaseSink.class);
    private HbaseEventSerializer serializer;
    private String eventSerializerType;
    private Context serializerContext;
    private String kerberosPrincipal;
    private String kerberosKeytab;
    private User hbaseUser;
    private boolean enableWal = true;
    private SinkCounter sinkCounter;

    public HBaseSink() {
        this(HBaseConfiguration.create());
    }

    public HBaseSink(Configuration conf) {
        this.config = conf;
    }

    public void start() {
        Preconditions.checkArgument((this.table == null ? 1 : 0) != 0, (Object)"Please call stop before calling start on an old instance.");
        try {
            if (HBaseSinkSecurityManager.isSecurityEnabled(this.config)) {
                this.hbaseUser = HBaseSinkSecurityManager.login(this.config, null, this.kerberosPrincipal, this.kerberosKeytab);
            }
        }
        catch (Exception ex) {
            this.sinkCounter.incrementConnectionFailedCount();
            throw new FlumeException("Failed to login to HBase using provided credentials.", (Throwable)ex);
        }
        try {
            this.table = this.runPrivileged(new PrivilegedExceptionAction<HTable>(){

                @Override
                public HTable run() throws Exception {
                    HTable table = new HTable(HBaseSink.this.config, HBaseSink.this.tableName);
                    table.setAutoFlush(false);
                    return table;
                }
            });
        }
        catch (Exception e) {
            this.sinkCounter.incrementConnectionFailedCount();
            logger.error("Could not load table, " + this.tableName + " from HBase", (Throwable)e);
            throw new FlumeException("Could not load table, " + this.tableName + " from HBase", (Throwable)e);
        }
        try {
            if (!this.runPrivileged(new PrivilegedExceptionAction<Boolean>(){

                @Override
                public Boolean run() throws IOException {
                    return HBaseSink.this.table.getTableDescriptor().hasFamily(HBaseSink.this.columnFamily);
                }
            }).booleanValue()) {
                throw new IOException("Table " + this.tableName + " has no such column family " + Bytes.toString((byte[])this.columnFamily));
            }
        }
        catch (Exception e) {
            this.sinkCounter.incrementConnectionFailedCount();
            throw new FlumeException("Error getting column family from HBase.Please verify that the table " + this.tableName + " and Column Family, " + Bytes.toString((byte[])this.columnFamily) + " exists in HBase, and the" + " current user has permissions to access that table.", (Throwable)e);
        }
        super.start();
        this.sinkCounter.incrementConnectionCreatedCount();
        this.sinkCounter.start();
    }

    public void stop() {
        try {
            this.table.close();
            this.table = null;
        }
        catch (IOException e) {
            throw new FlumeException("Error closing table.", (Throwable)e);
        }
        this.sinkCounter.incrementConnectionClosedCount();
        this.sinkCounter.stop();
    }

    public void configure(Context context) {
        this.tableName = context.getString("table");
        String cf = context.getString("columnFamily");
        this.batchSize = context.getLong("batchSize", new Long(100L));
        this.serializerContext = new Context();
        this.eventSerializerType = context.getString("serializer");
        Preconditions.checkNotNull((Object)this.tableName, (Object)"Table name cannot be empty, please specify in configuration file");
        Preconditions.checkNotNull((Object)cf, (Object)"Column family cannot be empty, please specify in configuration file");
        if (this.eventSerializerType == null || this.eventSerializerType.isEmpty()) {
            this.eventSerializerType = "org.apache.flume.sink.hbase.SimpleHbaseEventSerializer";
            logger.info("No serializer defined, Will use default");
        }
        this.serializerContext.putAll((Map)context.getSubProperties("serializer."));
        this.columnFamily = cf.getBytes(Charsets.UTF_8);
        try {
            Class<?> clazz = Class.forName(this.eventSerializerType);
            this.serializer = (HbaseEventSerializer)clazz.newInstance();
            this.serializer.configure(this.serializerContext);
        }
        catch (Exception e) {
            logger.error("Could not instantiate event serializer.", (Throwable)e);
            Throwables.propagate((Throwable)e);
        }
        this.kerberosKeytab = context.getString("kerberosKeytab", "");
        this.kerberosPrincipal = context.getString("kerberosPrincipal", "");
        this.enableWal = context.getBoolean("enableWal", Boolean.valueOf(true));
        logger.info("The write to WAL option is set to: " + String.valueOf(this.enableWal));
        if (!this.enableWal) {
            logger.warn("HBase Sink's enableWal configuration is set to false. All writes to HBase will have WAL disabled, and any data in the memstore of this region in the Region Server could be lost!");
        }
        this.sinkCounter = new SinkCounter(this.getName());
    }

    public Sink.Status process() throws EventDeliveryException {
        long i;
        Sink.Status status = Sink.Status.READY;
        Channel channel = this.getChannel();
        Transaction txn = channel.getTransaction();
        LinkedList<Row> actions = new LinkedList<Row>();
        LinkedList<Increment> incs = new LinkedList<Increment>();
        txn.begin();
        for (i = 0L; i < this.batchSize; ++i) {
            Event event = channel.take();
            if (event == null) {
                status = Sink.Status.BACKOFF;
                if (i == 0L) {
                    this.sinkCounter.incrementBatchEmptyCount();
                    break;
                }
                this.sinkCounter.incrementBatchUnderflowCount();
                break;
            }
            this.serializer.initialize(event, this.columnFamily);
            actions.addAll(this.serializer.getActions());
            incs.addAll(this.serializer.getIncrements());
        }
        if (i == this.batchSize) {
            this.sinkCounter.incrementBatchCompleteCount();
        }
        this.sinkCounter.addToEventDrainAttemptCount(i);
        this.putEventsAndCommit(actions, incs, txn);
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void putEventsAndCommit(final List<Row> actions, final List<Increment> incs, Transaction txn) throws EventDeliveryException {
        block8: {
            try {
                this.runPrivileged(new PrivilegedExceptionAction<Void>(){

                    @Override
                    public Void run() throws Exception {
                        for (Row r : actions) {
                            if (r instanceof Put) {
                                ((Put)r).setWriteToWAL(HBaseSink.this.enableWal);
                            }
                            if (!(r instanceof Increment)) continue;
                            ((Increment)r).setWriteToWAL(HBaseSink.this.enableWal);
                        }
                        HBaseSink.this.table.batch(actions);
                        return null;
                    }
                });
                this.runPrivileged(new PrivilegedExceptionAction<Void>(){

                    @Override
                    public Void run() throws Exception {
                        for (Increment i : incs) {
                            i.setWriteToWAL(HBaseSink.this.enableWal);
                            HBaseSink.this.table.increment(i);
                        }
                        return null;
                    }
                });
                txn.commit();
                this.sinkCounter.addToEventDrainSuccessCount((long)actions.size());
            }
            catch (Throwable e) {
                try {
                    txn.rollback();
                }
                catch (Exception e2) {
                    logger.error("Exception in rollback. Rollback might not have beensuccessful.", (Throwable)e2);
                }
                logger.error("Failed to commit transaction.Transaction rolled back.", e);
                if (e instanceof Error || e instanceof RuntimeException) {
                    logger.error("Failed to commit transaction.Transaction rolled back.", e);
                    Throwables.propagate((Throwable)e);
                    break block8;
                }
                logger.error("Failed to commit transaction.Transaction rolled back.", e);
                throw new EventDeliveryException("Failed to commit transaction.Transaction rolled back.", e);
            }
            finally {
                txn.close();
            }
        }
    }

    private <T> T runPrivileged(PrivilegedExceptionAction<T> action) throws Exception {
        if (this.hbaseUser != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Calling runAs as hbase user: " + this.hbaseUser.getName());
            }
            return (T)this.hbaseUser.runAs(action);
        }
        return action.run();
    }
}

