/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.contrib.streaming.state.snapshot;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.contrib.streaming.state.RocksDBKeyedStateBackend.RocksDbKvStateInfo;
import org.apache.flink.core.fs.CloseableRegistry;
import org.apache.flink.runtime.checkpoint.CheckpointOptions;
import org.apache.flink.runtime.state.AbstractSnapshotStrategy;
import org.apache.flink.runtime.state.CheckpointListener;
import org.apache.flink.runtime.state.CheckpointStreamFactory;
import org.apache.flink.runtime.state.DoneFuture;
import org.apache.flink.runtime.state.KeyGroupRange;
import org.apache.flink.runtime.state.KeyedStateHandle;
import org.apache.flink.runtime.state.LocalRecoveryConfig;
import org.apache.flink.runtime.state.SnapshotResult;
import org.apache.flink.runtime.state.SnapshotStrategy;
import org.apache.flink.util.ResourceGuard;

import org.rocksdb.RocksDB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;

import java.util.LinkedHashMap;
import java.util.concurrent.RunnableFuture;

/**
 * Abstract base class for {@link SnapshotStrategy} implementations for RocksDB state backend.
 *
 * @param <K> type of the backend keys.
 */
public abstract class RocksDBSnapshotStrategyBase<K>
	extends AbstractSnapshotStrategy<KeyedStateHandle>
	implements CheckpointListener {

	private static final Logger LOG = LoggerFactory.getLogger(RocksDBSnapshotStrategyBase.class);

	/** RocksDB instance from the backend. */
	@Nonnull
	protected RocksDB db;

	/** Resource guard for the RocksDB instance. */
	@Nonnull
	protected final ResourceGuard rocksDBResourceGuard;

	/** The key serializer of the backend. */
	@Nonnull
	protected final TypeSerializer<K> keySerializer;

	/** Key/Value state meta info from the backend. */
	@Nonnull
	protected final LinkedHashMap<String, RocksDbKvStateInfo> kvStateInformation;

	/** The key-group range for the task. */
	@Nonnull
	protected final KeyGroupRange keyGroupRange;

	/** Number of bytes in the key-group prefix. */
	@Nonnegative
	protected final int keyGroupPrefixBytes;

	/** The configuration for local recovery. */
	@Nonnull
	protected final LocalRecoveryConfig localRecoveryConfig;

	/** A {@link CloseableRegistry} that will be closed when the task is cancelled. */
	@Nonnull
	protected final CloseableRegistry cancelStreamRegistry;

	public RocksDBSnapshotStrategyBase(
		@Nonnull String description,
		@Nonnull RocksDB db,
		@Nonnull ResourceGuard rocksDBResourceGuard,
		@Nonnull TypeSerializer<K> keySerializer,
		@Nonnull LinkedHashMap<String, RocksDbKvStateInfo> kvStateInformation,
		@Nonnull KeyGroupRange keyGroupRange,
		@Nonnegative int keyGroupPrefixBytes,
		@Nonnull LocalRecoveryConfig localRecoveryConfig,
		@Nonnull CloseableRegistry cancelStreamRegistry) {

		super(description);
		this.db = db;
		this.rocksDBResourceGuard = rocksDBResourceGuard;
		this.keySerializer = keySerializer;
		this.kvStateInformation = kvStateInformation;
		this.keyGroupRange = keyGroupRange;
		this.keyGroupPrefixBytes = keyGroupPrefixBytes;
		this.localRecoveryConfig = localRecoveryConfig;
		this.cancelStreamRegistry = cancelStreamRegistry;
	}

	@Nonnull
	@Override
	public final RunnableFuture<SnapshotResult<KeyedStateHandle>> snapshot(
		long checkpointId,
		long timestamp,
		@Nonnull CheckpointStreamFactory streamFactory,
		@Nonnull CheckpointOptions checkpointOptions) throws Exception {

		if (kvStateInformation.isEmpty()) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Asynchronous RocksDB snapshot performed on empty keyed state at {}. Returning null.",
					timestamp);
			}
			return DoneFuture.of(SnapshotResult.empty());
		} else {
			return doSnapshot(checkpointId, timestamp, streamFactory, checkpointOptions);
		}
	}

	/**
	 * This method implements the concrete snapshot logic for a non-empty state.
	 */
	@Nonnull
	protected abstract RunnableFuture<SnapshotResult<KeyedStateHandle>> doSnapshot(
		long checkpointId,
		long timestamp,
		CheckpointStreamFactory streamFactory,
		CheckpointOptions checkpointOptions) throws Exception;
}
