/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.metadata;

import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.ql.metadata.CheckResult;
import org.apache.hadoop.hive.ql.metadata.Hive;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.metadata.Partition;
import org.apache.hadoop.hive.ql.metadata.Table;
import org.apache.hadoop.hive.ql.optimizer.ppr.PartitionPruner;
import org.apache.hadoop.hive.ql.parse.PrunedPartitionList;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveMetaStoreChecker {
    public static final Logger LOG = LoggerFactory.getLogger(HiveMetaStoreChecker.class);
    private final Hive hive;
    private final HiveConf conf;

    public HiveMetaStoreChecker(Hive hive) {
        this.hive = hive;
        this.conf = hive.getConf();
    }

    public void checkMetastore(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws HiveException, IOException {
        if (dbName == null || "".equalsIgnoreCase(dbName)) {
            dbName = "default";
        }
        try {
            if (tableName == null || "".equals(tableName)) {
                List<String> tables = this.hive.getTablesForDb(dbName, ".*");
                for (String currentTableName : tables) {
                    this.checkTable(dbName, currentTableName, null, result);
                }
                this.findUnknownTables(dbName, tables, result);
            } else if (partitions == null || partitions.isEmpty()) {
                this.checkTable(dbName, tableName, null, result);
            } else {
                this.checkTable(dbName, tableName, partitions, result);
            }
            LOG.info("Number of partitionsNotInMs=" + result.getPartitionsNotInMs() + ", partitionsNotOnFs=" + result.getPartitionsNotOnFs() + ", tablesNotInMs=" + result.getTablesNotInMs() + ", tablesNotOnFs=" + result.getTablesNotOnFs());
        }
        catch (MetaException e) {
            throw new HiveException(e);
        }
        catch (TException e) {
            throw new HiveException(e);
        }
    }

    void findUnknownTables(String dbName, List<String> tables, CheckResult result) throws IOException, MetaException, TException, HiveException {
        HashSet<Path> dbPaths = new HashSet<Path>();
        HashSet<String> tableNames = new HashSet<String>(tables);
        for (String tableName : tables) {
            Table table = this.hive.getTable(dbName, tableName);
            String isExternal = table.getParameters().get("EXTERNAL");
            if (isExternal != null && "TRUE".equalsIgnoreCase(isExternal)) continue;
            dbPaths.add(table.getPath().getParent());
        }
        for (Path dbPath : dbPaths) {
            FileStatus[] statuses;
            FileSystem fs = dbPath.getFileSystem((Configuration)this.conf);
            for (FileStatus status : statuses = fs.listStatus(dbPath, FileUtils.HIDDEN_FILES_PATH_FILTER)) {
                if (!status.isDir() || tableNames.contains(status.getPath().getName())) continue;
                result.getTablesNotInMs().add(status.getPath().getName());
            }
        }
    }

    void checkTable(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws MetaException, IOException, HiveException {
        Table table = null;
        try {
            table = this.hive.getTable(dbName, tableName);
        }
        catch (HiveException e) {
            result.getTablesNotInMs().add(tableName);
            return;
        }
        ArrayList<Partition> parts = new ArrayList<Partition>();
        boolean findUnknownPartitions = true;
        if (table.isPartitioned()) {
            if (partitions == null || partitions.isEmpty()) {
                PrunedPartitionList prunedPartList = PartitionPruner.prune(table, null, this.conf, this.toString(), null);
                parts.addAll(prunedPartList.getPartitions());
            } else {
                findUnknownPartitions = false;
                for (Map<String, String> map : partitions) {
                    Partition part = this.hive.getPartition(table, map, false);
                    if (part == null) {
                        CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                        pr.setTableName(tableName);
                        pr.setPartitionName(Warehouse.makePartPath(map));
                        result.getPartitionsNotInMs().add(pr);
                        continue;
                    }
                    parts.add(part);
                }
            }
        }
        this.checkTable(table, parts, findUnknownPartitions, result);
    }

    void checkTable(Table table, List<Partition> parts, boolean findUnknownPartitions, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        FileSystem fs = tablePath.getFileSystem((Configuration)this.conf);
        if (!fs.exists(tablePath)) {
            result.getTablesNotOnFs().add(table.getTableName());
            return;
        }
        HashSet<Path> partPaths = new HashSet<Path>();
        for (Partition partition : parts) {
            if (partition == null) continue;
            Path partPath = partition.getDataLocation();
            fs = partPath.getFileSystem((Configuration)this.conf);
            if (!fs.exists(partPath)) {
                CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                pr.setPartitionName(partition.getName());
                pr.setTableName(partition.getTable().getTableName());
                result.getPartitionsNotOnFs().add(pr);
            }
            for (int i = 0; i < partition.getSpec().size(); ++i) {
                partPaths.add(partPath.makeQualified(fs));
                partPath = partPath.getParent();
            }
        }
        if (findUnknownPartitions) {
            this.findUnknownPartitions(table, partPaths, result);
        }
    }

    void findUnknownPartitions(Table table, Set<Path> partPaths, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        HashSet<Path> allPartDirs = new HashSet<Path>();
        this.checkPartitionDirs(tablePath, allPartDirs, table.getPartCols().size());
        allPartDirs.remove(tablePath);
        allPartDirs.removeAll(partPaths);
        HashSet<String> partColNames = Sets.newHashSet();
        for (FieldSchema fSchema : table.getPartCols()) {
            partColNames.add(fSchema.getName());
        }
        for (Path partPath : allPartDirs) {
            FileSystem fs = partPath.getFileSystem((Configuration)this.conf);
            String partitionName = HiveMetaStoreChecker.getPartitionName(fs.makeQualified(tablePath), partPath, partColNames);
            LOG.debug("PartitionName: " + partitionName);
            if (partitionName == null) continue;
            CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
            pr.setPartitionName(partitionName);
            pr.setTableName(table.getTableName());
            result.getPartitionsNotInMs().add(pr);
        }
        LOG.debug("Number of partitions not in metastore : " + result.getPartitionsNotInMs().size());
    }

    static String getPartitionName(Path tablePath, Path partitionPath, Set<String> partCols) {
        String result = null;
        LOG.debug("tablePath:" + tablePath + ", partCols: " + partCols);
        for (Path currPath = partitionPath; currPath != null && !tablePath.equals((Object)currPath); currPath = currPath.getParent()) {
            String[] parts = currPath.getName().split("=");
            if (parts != null && parts.length > 0) {
                if (parts.length != 2) {
                    LOG.warn(currPath.getName() + " is not a valid partition name");
                    return result;
                }
                String partitionName = parts[0];
                if (partCols.contains(partitionName)) {
                    result = result == null ? currPath.getName() : currPath.getName() + "/" + result;
                }
            }
            LOG.debug("currPath=" + currPath);
        }
        return result;
    }

    private void checkPartitionDirs(Path basePath, Set<Path> allDirs, int maxDepth) throws IOException, HiveException {
        ThreadPoolExecutor pool;
        ConcurrentLinkedQueue<Path> basePaths = new ConcurrentLinkedQueue<Path>();
        basePaths.add(basePath);
        Set<Path> dirSet = Collections.newSetFromMap(new ConcurrentHashMap());
        int poolSize = this.conf.getInt(HiveConf.ConfVars.HIVE_MOVE_FILES_THREAD_COUNT.varname, 15);
        poolSize = poolSize == 0 ? poolSize : Math.max(poolSize, Runtime.getRuntime().availableProcessors() * 2);
        ThreadPoolExecutor threadPoolExecutor = pool = poolSize > 0 ? (ThreadPoolExecutor)Executors.newFixedThreadPool(poolSize, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MSCK-GetPaths-%d").build()) : null;
        if (pool == null) {
            LOG.debug("Not-using threaded version of MSCK-GetPaths");
        } else {
            LOG.debug("Using threaded version of MSCK-GetPaths with number of threads " + pool.getMaximumPoolSize());
        }
        this.checkPartitionDirs(pool, basePaths, dirSet, basePath.getFileSystem((Configuration)this.conf), maxDepth, maxDepth);
        if (pool != null) {
            pool.shutdown();
        }
        allDirs.addAll(dirSet);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkPartitionDirs(final ThreadPoolExecutor pool, ConcurrentLinkedQueue<Path> basePaths, final Set<Path> allDirs, final FileSystem fs, final int depth, final int maxDepth) throws IOException, HiveException {
        final ConcurrentLinkedQueue<Path> nextLevel = new ConcurrentLinkedQueue<Path>();
        boolean useThreadPool = false;
        if (pool != null) {
            ThreadPoolExecutor threadPoolExecutor = pool;
            synchronized (threadPoolExecutor) {
                if (pool.getActiveCount() < pool.getMaximumPoolSize()) {
                    useThreadPool = true;
                }
                if (!useThreadPool && LOG.isDebugEnabled()) {
                    LOG.debug("Not using threadPool as active count:" + pool.getActiveCount() + ", max:" + pool.getMaximumPoolSize());
                }
            }
        }
        if (null == pool || !useThreadPool) {
            for (Path path : basePaths) {
                FileStatus[] fileStatusArray = fs.listStatus(path, FileUtils.HIDDEN_FILES_PATH_FILTER);
                boolean fileFound = false;
                for (FileStatus status : fileStatusArray) {
                    if (status.isDirectory()) {
                        nextLevel.add(status.getPath());
                        continue;
                    }
                    fileFound = true;
                }
                if (depth != 0) {
                    if (fileFound) {
                        if ("throw".equals(HiveConf.getVar(this.conf, HiveConf.ConfVars.HIVE_MSCK_PATH_VALIDATION))) {
                            throw new HiveException("MSCK finds a file rather than a folder when it searches for " + path.toString());
                        }
                        LOG.warn("MSCK finds a file rather than a folder when it searches for " + path.toString());
                    }
                    if (!nextLevel.isEmpty()) {
                        this.checkPartitionDirs(pool, nextLevel, allDirs, fs, depth - 1, maxDepth);
                        continue;
                    }
                    if (depth == maxDepth) continue;
                    if ("throw".equals(HiveConf.getVar(this.conf, HiveConf.ConfVars.HIVE_MSCK_PATH_VALIDATION))) {
                        throw new HiveException("MSCK is missing partition columns under " + path.toString());
                    }
                    LOG.warn("MSCK is missing partition columns under " + path.toString());
                    continue;
                }
                allDirs.add(path);
            }
        } else {
            LinkedList<Future<Void>> futures = new LinkedList<Future<Void>>();
            for (final Path path : basePaths) {
                futures.add(pool.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        FileStatus[] statuses = fs.listStatus(path, FileUtils.HIDDEN_FILES_PATH_FILTER);
                        boolean fileFound = false;
                        for (FileStatus status : statuses) {
                            if (status.isDirectory()) {
                                nextLevel.add(status.getPath());
                                continue;
                            }
                            fileFound = true;
                        }
                        if (depth != 0) {
                            if (fileFound) {
                                if ("throw".equals(HiveConf.getVar(HiveMetaStoreChecker.this.conf, HiveConf.ConfVars.HIVE_MSCK_PATH_VALIDATION))) {
                                    throw new HiveException("MSCK finds a file rather than a folder when it searches for " + path.toString());
                                }
                                LOG.warn("MSCK finds a file rather than a folder when it searches for " + path.toString());
                            }
                            if (!nextLevel.isEmpty()) {
                                HiveMetaStoreChecker.this.checkPartitionDirs(pool, nextLevel, allDirs, fs, depth - 1, maxDepth);
                            } else if (depth != maxDepth) {
                                if ("throw".equals(HiveConf.getVar(HiveMetaStoreChecker.this.conf, HiveConf.ConfVars.HIVE_MSCK_PATH_VALIDATION))) {
                                    throw new HiveException("MSCK is missing partition columns under " + path.toString());
                                }
                                LOG.warn("MSCK is missing partition columns under " + path.toString());
                            }
                        } else {
                            allDirs.add(path);
                        }
                        return null;
                    }
                }));
            }
            for (Future future : futures) {
                try {
                    future.get();
                }
                catch (Exception e) {
                    LOG.error(e.getMessage());
                    pool.shutdownNow();
                    throw new HiveException(e.getCause());
                }
            }
            if (!nextLevel.isEmpty() && depth != 0) {
                this.checkPartitionDirs(pool, nextLevel, allDirs, fs, depth - 1, maxDepth);
            }
        }
    }
}

