/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.connector.hbase.util;

import java.io.Serializable;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import javax.annotation.Nullable;
import org.apache.flink.connector.hbase.util.HBaseTableSchema;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.client.Delete;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.client.Get;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.client.Put;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.client.Result;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.client.Scan;
import org.apache.flink.hbase.shaded.org.apache.hadoop.hbase.util.Bytes;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeFamily;
import org.apache.flink.table.types.logical.utils.LogicalTypeChecks;
import org.apache.flink.util.Preconditions;

public class HBaseSerde {
    private static final byte[] EMPTY_BYTES = new byte[0];
    private static final int MIN_TIMESTAMP_PRECISION = 0;
    private static final int MAX_TIMESTAMP_PRECISION = 3;
    private static final int MIN_TIME_PRECISION = 0;
    private static final int MAX_TIME_PRECISION = 3;
    private final byte[] nullStringBytes;
    private final int rowkeyIndex;
    private final byte[][] families;
    private final byte[][][] qualifiers;
    private final int fieldLength;
    private GenericRowData reusedRow;
    private GenericRowData[] reusedFamilyRows;
    @Nullable
    private final FieldEncoder keyEncoder;
    @Nullable
    private final FieldDecoder keyDecoder;
    private final FieldEncoder[][] qualifierEncoders;
    private final FieldDecoder[][] qualifierDecoders;
    private final GenericRowData rowWithRowKey;

    public HBaseSerde(HBaseTableSchema hbaseSchema, String nullStringLiteral) {
        this.families = hbaseSchema.getFamilyKeys();
        this.rowkeyIndex = hbaseSchema.getRowKeyIndex();
        LogicalType rowkeyType = hbaseSchema.getRowKeyDataType().map(DataType::getLogicalType).orElse(null);
        if (this.rowkeyIndex != -1 && rowkeyType != null) {
            this.fieldLength = this.families.length + 1;
            this.keyEncoder = HBaseSerde.createFieldEncoder(rowkeyType);
            this.keyDecoder = HBaseSerde.createFieldDecoder(rowkeyType);
        } else {
            this.fieldLength = this.families.length;
            this.keyEncoder = null;
            this.keyDecoder = null;
        }
        this.nullStringBytes = nullStringLiteral.getBytes(StandardCharsets.UTF_8);
        this.reusedRow = new GenericRowData(this.fieldLength);
        this.reusedFamilyRows = new GenericRowData[this.families.length];
        this.qualifiers = new byte[this.families.length][][];
        this.qualifierEncoders = new FieldEncoder[this.families.length][];
        this.qualifierDecoders = new FieldDecoder[this.families.length][];
        String[] familyNames = hbaseSchema.getFamilyNames();
        for (int f = 0; f < this.families.length; ++f) {
            this.qualifiers[f] = hbaseSchema.getQualifierKeys(familyNames[f]);
            DataType[] dataTypes = hbaseSchema.getQualifierDataTypes(familyNames[f]);
            this.qualifierEncoders[f] = (FieldEncoder[])Arrays.stream(dataTypes).map(DataType::getLogicalType).map(t -> HBaseSerde.createNullableFieldEncoder(t, this.nullStringBytes)).toArray(FieldEncoder[]::new);
            this.qualifierDecoders[f] = (FieldDecoder[])Arrays.stream(dataTypes).map(DataType::getLogicalType).map(t -> HBaseSerde.createNullableFieldDecoder(t, this.nullStringBytes)).toArray(FieldDecoder[]::new);
            this.reusedFamilyRows[f] = new GenericRowData(dataTypes.length);
        }
        this.rowWithRowKey = new GenericRowData(1);
    }

    @Nullable
    public Put createPutMutation(RowData row) {
        Preconditions.checkArgument((this.keyEncoder != null ? 1 : 0) != 0, (Object)"row key is not set.");
        byte[] rowkey = this.keyEncoder.encode(row, this.rowkeyIndex);
        if (rowkey.length == 0) {
            return null;
        }
        Put put = new Put(rowkey);
        for (int i = 0; i < this.fieldLength; ++i) {
            if (i == this.rowkeyIndex) continue;
            int f = i > this.rowkeyIndex ? i - 1 : i;
            byte[] familyKey = this.families[f];
            RowData familyRow = row.getRow(i, this.qualifiers[f].length);
            for (int q = 0; q < this.qualifiers[f].length; ++q) {
                byte[] qualifier = this.qualifiers[f][q];
                byte[] value = this.qualifierEncoders[f][q].encode(familyRow, q);
                put.addColumn(familyKey, qualifier, value);
            }
        }
        return put;
    }

    @Nullable
    public Delete createDeleteMutation(RowData row) {
        Preconditions.checkArgument((this.keyEncoder != null ? 1 : 0) != 0, (Object)"row key is not set.");
        byte[] rowkey = this.keyEncoder.encode(row, this.rowkeyIndex);
        if (rowkey.length == 0) {
            return null;
        }
        Delete delete = new Delete(rowkey);
        for (int i = 0; i < this.fieldLength; ++i) {
            if (i == this.rowkeyIndex) continue;
            int f = i > this.rowkeyIndex ? i - 1 : i;
            byte[] familyKey = this.families[f];
            for (int q = 0; q < this.qualifiers[f].length; ++q) {
                byte[] qualifier = this.qualifiers[f][q];
                delete.addColumn(familyKey, qualifier);
            }
        }
        return delete;
    }

    public Scan createScan() {
        Scan scan = new Scan();
        for (int f = 0; f < this.families.length; ++f) {
            byte[] family = this.families[f];
            for (int q = 0; q < this.qualifiers[f].length; ++q) {
                byte[] quantifier = this.qualifiers[f][q];
                scan.addColumn(family, quantifier);
            }
        }
        return scan;
    }

    public Get createGet(Object rowKey) {
        Preconditions.checkArgument((this.keyEncoder != null ? 1 : 0) != 0, (Object)"row key is not set.");
        this.rowWithRowKey.setField(0, rowKey);
        byte[] rowkey = this.keyEncoder.encode((RowData)this.rowWithRowKey, 0);
        if (rowkey.length == 0) {
            return null;
        }
        Get get = new Get(rowkey);
        for (int f = 0; f < this.families.length; ++f) {
            byte[] family = this.families[f];
            for (byte[] qualifier : this.qualifiers[f]) {
                get.addColumn(family, qualifier);
            }
        }
        return get;
    }

    public RowData convertToNewRow(Result result) {
        GenericRowData resultRow = new GenericRowData(this.fieldLength);
        GenericRowData[] familyRows = new GenericRowData[this.families.length];
        for (int f = 0; f < this.families.length; ++f) {
            familyRows[f] = new GenericRowData(this.qualifiers[f].length);
        }
        return this.convertToRow(result, resultRow, familyRows);
    }

    public RowData convertToReusedRow(Result result) {
        return this.convertToRow(result, this.reusedRow, this.reusedFamilyRows);
    }

    private RowData convertToRow(Result result, GenericRowData resultRow, GenericRowData[] familyRows) {
        for (int i = 0; i < this.fieldLength; ++i) {
            if (this.rowkeyIndex == i) {
                assert (this.keyDecoder != null);
                Object rowkey = this.keyDecoder.decode(result.getRow());
                resultRow.setField(this.rowkeyIndex, rowkey);
                continue;
            }
            int f = this.rowkeyIndex != -1 && i > this.rowkeyIndex ? i - 1 : i;
            byte[] familyKey = this.families[f];
            GenericRowData familyRow = familyRows[f];
            for (int q = 0; q < this.qualifiers[f].length; ++q) {
                byte[] qualifier = this.qualifiers[f][q];
                byte[] value = result.getValue(familyKey, qualifier);
                familyRow.setField(q, this.qualifierDecoders[f][q].decode(value));
            }
            resultRow.setField(i, (Object)familyRow);
        }
        return resultRow;
    }

    @Deprecated
    public RowData convertToRow(Result result) {
        for (int i = 0; i < this.fieldLength; ++i) {
            if (this.rowkeyIndex == i) {
                assert (this.keyDecoder != null);
                Object rowkey = this.keyDecoder.decode(result.getRow());
                this.reusedRow.setField(this.rowkeyIndex, rowkey);
                continue;
            }
            int f = this.rowkeyIndex != -1 && i > this.rowkeyIndex ? i - 1 : i;
            byte[] familyKey = this.families[f];
            GenericRowData familyRow = this.reusedFamilyRows[f];
            for (int q = 0; q < this.qualifiers[f].length; ++q) {
                byte[] qualifier = this.qualifiers[f][q];
                byte[] value = result.getValue(familyKey, qualifier);
                familyRow.setField(q, this.qualifierDecoders[f][q].decode(value));
            }
            this.reusedRow.setField(i, (Object)familyRow);
        }
        return this.reusedRow;
    }

    private static FieldEncoder createNullableFieldEncoder(LogicalType fieldType, byte[] nullStringBytes) {
        FieldEncoder encoder = HBaseSerde.createFieldEncoder(fieldType);
        if (fieldType.isNullable()) {
            if (LogicalTypeChecks.hasFamily((LogicalType)fieldType, (LogicalTypeFamily)LogicalTypeFamily.CHARACTER_STRING)) {
                return (row, pos) -> {
                    if (row.isNullAt(pos)) {
                        return nullStringBytes;
                    }
                    return encoder.encode(row, pos);
                };
            }
            return (row, pos) -> {
                if (row.isNullAt(pos)) {
                    return EMPTY_BYTES;
                }
                return encoder.encode(row, pos);
            };
        }
        return encoder;
    }

    private static FieldEncoder createFieldEncoder(LogicalType fieldType) {
        switch (fieldType.getTypeRoot()) {
            case CHAR: 
            case VARCHAR: {
                return (row, pos) -> row.getString(pos).toBytes();
            }
            case BOOLEAN: {
                return (row, pos) -> Bytes.toBytes(row.getBoolean(pos));
            }
            case BINARY: 
            case VARBINARY: {
                return RowData::getBinary;
            }
            case DECIMAL: {
                return HBaseSerde.createDecimalEncoder((DecimalType)fieldType);
            }
            case TINYINT: {
                return (row, pos) -> new byte[]{row.getByte(pos)};
            }
            case SMALLINT: {
                return (row, pos) -> Bytes.toBytes(row.getShort(pos));
            }
            case INTEGER: 
            case DATE: 
            case INTERVAL_YEAR_MONTH: {
                return (row, pos) -> Bytes.toBytes(row.getInt(pos));
            }
            case TIME_WITHOUT_TIME_ZONE: {
                int timePrecision = LogicalTypeChecks.getPrecision((LogicalType)fieldType);
                if (timePrecision < 0 || timePrecision > 3) {
                    throw new UnsupportedOperationException(String.format("The precision %s of TIME type is out of the range [%s, %s] supported by HBase connector", timePrecision, 0, 3));
                }
                return (row, pos) -> Bytes.toBytes(row.getInt(pos));
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return (row, pos) -> Bytes.toBytes(row.getLong(pos));
            }
            case FLOAT: {
                return (row, pos) -> Bytes.toBytes(row.getFloat(pos));
            }
            case DOUBLE: {
                return (row, pos) -> Bytes.toBytes(row.getDouble(pos));
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: 
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                int timestampPrecision = LogicalTypeChecks.getPrecision((LogicalType)fieldType);
                if (timestampPrecision < 0 || timestampPrecision > 3) {
                    throw new UnsupportedOperationException(String.format("The precision %s of TIMESTAMP type is out of the range [%s, %s] supported by HBase connector", timestampPrecision, 0, 3));
                }
                return HBaseSerde.createTimestampEncoder(timestampPrecision);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + fieldType);
    }

    private static FieldEncoder createDecimalEncoder(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return (row, pos) -> {
            BigDecimal decimal = row.getDecimal(pos, precision, scale).toBigDecimal();
            return Bytes.toBytes(decimal);
        };
    }

    private static FieldEncoder createTimestampEncoder(int precision) {
        return (row, pos) -> {
            long millisecond = row.getTimestamp(pos, precision).getMillisecond();
            return Bytes.toBytes(millisecond);
        };
    }

    private static FieldDecoder createNullableFieldDecoder(LogicalType fieldType, byte[] nullStringBytes) {
        FieldDecoder decoder = HBaseSerde.createFieldDecoder(fieldType);
        if (fieldType.isNullable()) {
            if (LogicalTypeChecks.hasFamily((LogicalType)fieldType, (LogicalTypeFamily)LogicalTypeFamily.CHARACTER_STRING)) {
                return value -> {
                    if (value == null || Arrays.equals(value, nullStringBytes)) {
                        return null;
                    }
                    return decoder.decode(value);
                };
            }
            return value -> {
                if (value == null || value.length == 0) {
                    return null;
                }
                return decoder.decode(value);
            };
        }
        return decoder;
    }

    private static FieldDecoder createFieldDecoder(LogicalType fieldType) {
        switch (fieldType.getTypeRoot()) {
            case CHAR: 
            case VARCHAR: {
                return StringData::fromBytes;
            }
            case BOOLEAN: {
                return Bytes::toBoolean;
            }
            case BINARY: 
            case VARBINARY: {
                return value -> value;
            }
            case DECIMAL: {
                return HBaseSerde.createDecimalDecoder((DecimalType)fieldType);
            }
            case TINYINT: {
                return value -> value[0];
            }
            case SMALLINT: {
                return Bytes::toShort;
            }
            case INTEGER: 
            case DATE: 
            case INTERVAL_YEAR_MONTH: {
                return Bytes::toInt;
            }
            case TIME_WITHOUT_TIME_ZONE: {
                int timePrecision = LogicalTypeChecks.getPrecision((LogicalType)fieldType);
                if (timePrecision < 0 || timePrecision > 3) {
                    throw new UnsupportedOperationException(String.format("The precision %s of TIME type is out of the range [%s, %s] supported by HBase connector", timePrecision, 0, 3));
                }
                return Bytes::toInt;
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return Bytes::toLong;
            }
            case FLOAT: {
                return Bytes::toFloat;
            }
            case DOUBLE: {
                return Bytes::toDouble;
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: 
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                int timestampPrecision = LogicalTypeChecks.getPrecision((LogicalType)fieldType);
                if (timestampPrecision < 0 || timestampPrecision > 3) {
                    throw new UnsupportedOperationException(String.format("The precision %s of TIMESTAMP type is out of the range [%s, %s] supported by HBase connector", timestampPrecision, 0, 3));
                }
                return HBaseSerde.createTimestampDecoder();
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + fieldType);
    }

    private static FieldDecoder createDecimalDecoder(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return value -> {
            BigDecimal decimal = Bytes.toBigDecimal(value);
            return DecimalData.fromBigDecimal((BigDecimal)decimal, (int)precision, (int)scale);
        };
    }

    private static FieldDecoder createTimestampDecoder() {
        return value -> {
            long milliseconds = Bytes.toLong(value);
            return TimestampData.fromEpochMillis((long)milliseconds);
        };
    }

    @FunctionalInterface
    private static interface FieldDecoder
    extends Serializable {
        @Nullable
        public Object decode(byte[] var1);
    }

    @FunctionalInterface
    private static interface FieldEncoder
    extends Serializable {
        public byte[] encode(RowData var1, int var2);
    }
}

