/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.client.cli;

import java.io.IOError;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.flink.table.api.TableSchema;
import org.apache.flink.table.client.SqlClientException;
import org.apache.flink.table.client.cli.CliChangelogResultView;
import org.apache.flink.table.client.cli.CliResultView;
import org.apache.flink.table.client.cli.CliStrings;
import org.apache.flink.table.client.cli.CliTableResultView;
import org.apache.flink.table.client.cli.SqlCommandParser;
import org.apache.flink.table.client.cli.SqlCompleter;
import org.apache.flink.table.client.cli.SqlMultiLineParser;
import org.apache.flink.table.client.config.entries.ViewEntry;
import org.apache.flink.table.client.gateway.Executor;
import org.apache.flink.table.client.gateway.ProgramTargetDescriptor;
import org.apache.flink.table.client.gateway.ResultDescriptor;
import org.apache.flink.table.client.gateway.SessionContext;
import org.apache.flink.table.client.gateway.SqlExecutionException;
import org.jline.reader.EndOfFileException;
import org.jline.reader.LineReader;
import org.jline.reader.LineReaderBuilder;
import org.jline.reader.MaskingCallback;
import org.jline.reader.UserInterruptException;
import org.jline.terminal.Terminal;
import org.jline.terminal.TerminalBuilder;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;
import org.jline.utils.InfoCmp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CliClient {
    private static final Logger LOG = LoggerFactory.getLogger(CliClient.class);
    private final Executor executor;
    private final SessionContext context;
    private final Terminal terminal;
    private final LineReader lineReader;
    private final String prompt;
    private boolean isRunning;
    private static final int PLAIN_TERMINAL_WIDTH = 80;
    private static final int PLAIN_TERMINAL_HEIGHT = 30;
    private static final int SOURCE_MAX_SIZE = 50000;

    public CliClient(SessionContext context, Executor executor) {
        this.context = context;
        this.executor = executor;
        try {
            this.terminal = TerminalBuilder.builder().name("Flink SQL CLI Client").build();
            this.terminal.writer().println();
            this.terminal.writer().flush();
        }
        catch (IOException e2) {
            throw new SqlClientException("Error opening command line interface.", e2);
        }
        this.lineReader = LineReaderBuilder.builder().terminal(this.terminal).appName("Flink SQL CLI Client").parser(new SqlMultiLineParser()).completer(new SqlCompleter(context, executor)).build();
        this.lineReader.option(LineReader.Option.DISABLE_EVENT_EXPANSION, true);
        this.lineReader.setVariable("errors", 1);
        this.lineReader.option(LineReader.Option.CASE_INSENSITIVE, true);
        this.prompt = new AttributedStringBuilder().style(AttributedStyle.DEFAULT.foreground(2)).append("Flink SQL").style(AttributedStyle.DEFAULT).append("> ").toAnsi();
    }

    public Terminal getTerminal() {
        return this.terminal;
    }

    public SessionContext getContext() {
        return this.context;
    }

    public void clearTerminal() {
        if (this.isPlainTerminal()) {
            for (int i = 0; i < 200; ++i) {
                this.terminal.writer().println();
            }
        } else {
            this.terminal.puts(InfoCmp.Capability.clear_screen, new Object[0]);
        }
    }

    public boolean isPlainTerminal() {
        return this.terminal.getWidth() == 0 && this.terminal.getHeight() == 0;
    }

    public int getWidth() {
        if (this.isPlainTerminal()) {
            return 80;
        }
        return this.terminal.getWidth();
    }

    public int getHeight() {
        if (this.isPlainTerminal()) {
            return 30;
        }
        return this.terminal.getHeight();
    }

    public Executor getExecutor() {
        return this.executor;
    }

    public void open() {
        this.isRunning = true;
        this.terminal.writer().append(CliStrings.MESSAGE_WELCOME);
        while (this.isRunning) {
            String line;
            this.terminal.writer().append("\n");
            this.terminal.flush();
            try {
                line = this.lineReader.readLine(this.prompt, null, (MaskingCallback)null, null);
            }
            catch (UserInterruptException e2) {
                continue;
            }
            catch (IOError | EndOfFileException e3) {
                break;
            }
            catch (Throwable t) {
                throw new SqlClientException("Could not read from command line.", t);
            }
            if (line == null) continue;
            Optional<SqlCommandParser.SqlCommandCall> cmdCall = this.parseCommand(line);
            cmdCall.ifPresent(this::callCommand);
        }
    }

    public boolean submitUpdate(String statement) {
        this.terminal.writer().println(CliStrings.messageInfo("Executing the following statement:").toAnsi());
        this.terminal.writer().println(new AttributedString(statement).toString());
        this.terminal.flush();
        Optional<SqlCommandParser.SqlCommandCall> parsedStatement = this.parseCommand(statement);
        return parsedStatement.map(cmdCall -> {
            switch (cmdCall.command) {
                case INSERT_INTO: {
                    return this.callInsertInto((SqlCommandParser.SqlCommandCall)cmdCall);
                }
            }
            this.printError("Unsupported SQL statement.");
            return false;
        }).orElse(false);
    }

    private Optional<SqlCommandParser.SqlCommandCall> parseCommand(String line) {
        Optional<SqlCommandParser.SqlCommandCall> parsedLine = SqlCommandParser.parse(line);
        if (!parsedLine.isPresent()) {
            this.printError("Unknown or invalid SQL statement.");
        }
        return parsedLine;
    }

    private void callCommand(SqlCommandParser.SqlCommandCall cmdCall) {
        switch (cmdCall.command) {
            case QUIT: {
                this.callQuit();
                break;
            }
            case CLEAR: {
                this.callClear();
                break;
            }
            case RESET: {
                this.callReset();
                break;
            }
            case SET: {
                this.callSet(cmdCall);
                break;
            }
            case HELP: {
                this.callHelp();
                break;
            }
            case SHOW_TABLES: {
                this.callShowTables();
                break;
            }
            case SHOW_FUNCTIONS: {
                this.callShowFunctions();
                break;
            }
            case DESCRIBE: {
                this.callDescribe(cmdCall);
                break;
            }
            case EXPLAIN: {
                this.callExplain(cmdCall);
                break;
            }
            case SELECT: {
                this.callSelect(cmdCall);
                break;
            }
            case INSERT_INTO: {
                this.callInsertInto(cmdCall);
                break;
            }
            case CREATE_VIEW: {
                this.callCreateView(cmdCall);
                break;
            }
            case DROP_VIEW: {
                this.callDropView(cmdCall);
                break;
            }
            case SOURCE: {
                this.callSource(cmdCall);
                break;
            }
            default: {
                throw new SqlClientException("Unsupported command: " + (Object)((Object)cmdCall.command));
            }
        }
    }

    private void callQuit() {
        this.printInfo("Exiting Flink SQL CLI Client...");
        this.isRunning = false;
    }

    private void callClear() {
        this.clearTerminal();
    }

    private void callReset() {
        this.context.resetSessionProperties();
        this.printInfo("All session properties have been set to their default values.");
    }

    private void callSet(SqlCommandParser.SqlCommandCall cmdCall) {
        if (cmdCall.operands.length == 0) {
            Map<String, String> properties;
            try {
                properties = this.executor.getSessionProperties(this.context);
            }
            catch (SqlExecutionException e3) {
                this.printExecutionException(e3);
                return;
            }
            if (properties.isEmpty()) {
                this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
            } else {
                properties.entrySet().stream().map(e2 -> (String)e2.getKey() + "=" + (String)e2.getValue()).sorted().forEach(p -> this.terminal.writer().println((String)p));
            }
        } else {
            this.context.setSessionProperty(cmdCall.operands[0], cmdCall.operands[1]);
            this.terminal.writer().println(CliStrings.messageInfo("Session property has been set.").toAnsi());
        }
        this.terminal.flush();
    }

    private void callHelp() {
        this.terminal.writer().println(CliStrings.MESSAGE_HELP);
        this.terminal.flush();
    }

    private void callShowTables() {
        List<String> tables;
        try {
            tables = this.executor.listTables(this.context);
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return;
        }
        if (tables.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            tables.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callShowFunctions() {
        List<String> functions;
        try {
            functions = this.executor.listUserDefinedFunctions(this.context);
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return;
        }
        if (functions.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            functions.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callDescribe(SqlCommandParser.SqlCommandCall cmdCall) {
        TableSchema schema;
        try {
            schema = this.executor.getTableSchema(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return;
        }
        this.terminal.writer().println(schema.toString());
        this.terminal.flush();
    }

    private void callExplain(SqlCommandParser.SqlCommandCall cmdCall) {
        String explanation;
        try {
            explanation = this.executor.explainStatement(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return;
        }
        this.terminal.writer().println(explanation);
        this.terminal.flush();
    }

    private void callSelect(SqlCommandParser.SqlCommandCall cmdCall) {
        ResultDescriptor resultDesc;
        try {
            resultDesc = this.executor.executeQuery(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return;
        }
        CliResultView view = resultDesc.isMaterialized() ? new CliTableResultView(this, resultDesc) : new CliChangelogResultView(this, resultDesc);
        try {
            view.open();
            this.printInfo("Result retrieval cancelled.");
        }
        catch (SqlExecutionException e3) {
            this.printExecutionException(e3);
        }
    }

    private boolean callInsertInto(SqlCommandParser.SqlCommandCall cmdCall) {
        this.printInfo("Submitting SQL update statement to the cluster...");
        try {
            ProgramTargetDescriptor programTarget = this.executor.executeUpdate(this.context, cmdCall.operands[0]);
            this.terminal.writer().println(CliStrings.messageInfo("Table update statement has been successfully submitted to the cluster:").toAnsi());
            this.terminal.writer().println(programTarget.toString());
            this.terminal.flush();
        }
        catch (SqlExecutionException e2) {
            this.printExecutionException(e2);
            return false;
        }
        return true;
    }

    private void callCreateView(SqlCommandParser.SqlCommandCall cmdCall) {
        String name = cmdCall.operands[0];
        String query = cmdCall.operands[1];
        ViewEntry previousView = this.context.getViews().get(name);
        if (previousView != null) {
            this.printExecutionError("A view with this name has already been defined in the current CLI session.");
            return;
        }
        try {
            this.context.addView(ViewEntry.create(name, query));
            this.executor.validateSession(this.context);
            this.printInfo("View has been created.");
        }
        catch (SqlExecutionException e2) {
            this.context.removeView(name);
            this.printExecutionException(e2);
        }
    }

    private void callDropView(SqlCommandParser.SqlCommandCall cmdCall) {
        String name = cmdCall.operands[0];
        ViewEntry view = this.context.getViews().get(name);
        if (view == null) {
            this.printExecutionError("The given view does not exist in the current CLI session. Only views created with a CREATE VIEW statement can be accessed.");
            return;
        }
        try {
            this.context.removeView(name);
            this.executor.validateSession(this.context);
            this.printInfo("View has been removed.");
        }
        catch (SqlExecutionException e2) {
            this.context.addView(view);
            this.printExecutionException("The given view cannot be removed without affecting other views.", e2);
        }
    }

    private void callSource(SqlCommandParser.SqlCommandCall cmdCall) {
        String stmt;
        String pathString = cmdCall.operands[0];
        try {
            Path path = Paths.get(pathString, new String[0]);
            byte[] encoded = Files.readAllBytes(path);
            stmt = new String(encoded, Charset.defaultCharset());
        }
        catch (IOException e2) {
            this.printExecutionException(e2);
            return;
        }
        if (stmt.length() > 50000) {
            this.printExecutionError("The given file exceeds the maximum number of characters.");
            return;
        }
        this.terminal.writer().println(CliStrings.messageInfo("Executing the following statement:").toAnsi());
        this.terminal.writer().println(new AttributedString(stmt).toString());
        this.terminal.flush();
        Optional<SqlCommandParser.SqlCommandCall> call = this.parseCommand(stmt);
        call.ifPresent(this::callCommand);
    }

    private void printExecutionException(Throwable t) {
        this.printExecutionException(null, t);
    }

    private void printExecutionException(String message, Throwable t) {
        String finalMessage = message == null ? "Could not execute SQL statement." : "Could not execute SQL statement. " + message;
        this.printException(finalMessage, t);
    }

    private void printExecutionError(String message) {
        this.terminal.writer().println(CliStrings.messageError("Could not execute SQL statement.", message).toAnsi());
        this.terminal.flush();
    }

    private void printException(String message, Throwable t) {
        LOG.warn(message, t);
        this.terminal.writer().println(CliStrings.messageError(message, t).toAnsi());
        this.terminal.flush();
    }

    private void printError(String message) {
        this.terminal.writer().println(CliStrings.messageError(message).toAnsi());
        this.terminal.flush();
    }

    private void printInfo(String message) {
        this.terminal.writer().println(CliStrings.messageInfo(message).toAnsi());
        this.terminal.flush();
    }
}

