/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.client.cli;

import java.io.IOError;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;
import org.apache.flink.table.api.TableSchema;
import org.apache.flink.table.client.SqlClientException;
import org.apache.flink.table.client.cli.CliChangelogResultView;
import org.apache.flink.table.client.cli.CliResultView;
import org.apache.flink.table.client.cli.CliStrings;
import org.apache.flink.table.client.cli.CliTableResultView;
import org.apache.flink.table.client.cli.SqlCommandParser;
import org.apache.flink.table.client.gateway.Executor;
import org.apache.flink.table.client.gateway.ResultDescriptor;
import org.apache.flink.table.client.gateway.SessionContext;
import org.apache.flink.table.client.gateway.SqlExecutionException;
import org.jline.reader.EndOfFileException;
import org.jline.reader.LineReader;
import org.jline.reader.LineReaderBuilder;
import org.jline.reader.MaskingCallback;
import org.jline.reader.UserInterruptException;
import org.jline.reader.impl.DefaultParser;
import org.jline.terminal.Terminal;
import org.jline.terminal.TerminalBuilder;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;
import org.jline.utils.InfoCmp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CliClient {
    private static final Logger LOG = LoggerFactory.getLogger(CliClient.class);
    private final Executor executor;
    private final SessionContext context;
    private final Terminal terminal;
    private final LineReader lineReader;
    private final String prompt;
    private boolean isRunning;
    private static final int PLAIN_TERMINAL_WIDTH = 80;
    private static final int PLAIN_TERMINAL_HEIGHT = 30;
    private static final int SOURCE_MAX_SIZE = 50000;

    public CliClient(SessionContext context, Executor executor) {
        this.context = context;
        this.executor = executor;
        try {
            this.terminal = TerminalBuilder.builder().name("Flink SQL CLI Client").build();
        }
        catch (IOException e2) {
            throw new SqlClientException("Error opening command line interface.", e2);
        }
        DefaultParser parser = new DefaultParser();
        parser.setEofOnEscapedNewLine(true);
        this.lineReader = LineReaderBuilder.builder().terminal(this.terminal).appName("Flink SQL CLI Client").parser(parser).build();
        this.prompt = new AttributedStringBuilder().style(AttributedStyle.DEFAULT.foreground(2)).append("Flink SQL").style(AttributedStyle.DEFAULT).append("> ").toAnsi();
    }

    public Terminal getTerminal() {
        return this.terminal;
    }

    public SessionContext getContext() {
        return this.context;
    }

    public void clearTerminal() {
        if (this.isPlainTerminal()) {
            for (int i = 0; i < 200; ++i) {
                this.terminal.writer().println();
            }
        } else {
            this.terminal.puts(InfoCmp.Capability.clear_screen, new Object[0]);
        }
    }

    public boolean isPlainTerminal() {
        return this.terminal.getWidth() == 0 && this.terminal.getHeight() == 0;
    }

    public int getWidth() {
        if (this.isPlainTerminal()) {
            return 80;
        }
        return this.terminal.getWidth();
    }

    public int getHeight() {
        if (this.isPlainTerminal()) {
            return 30;
        }
        return this.terminal.getHeight();
    }

    public Executor getExecutor() {
        return this.executor;
    }

    public void open() {
        this.isRunning = true;
        this.terminal.writer().append(CliStrings.MESSAGE_WELCOME);
        while (this.isRunning) {
            String line;
            this.terminal.writer().append("\n");
            this.terminal.flush();
            try {
                line = this.lineReader.readLine(this.prompt, null, (MaskingCallback)null, null);
            }
            catch (IOError | EndOfFileException | UserInterruptException e2) {
                break;
            }
            catch (Throwable t) {
                throw new SqlClientException("Could not read from command line.", t);
            }
            if (line == null || line.equals("")) continue;
            SqlCommandParser.SqlCommandCall cmdCall = SqlCommandParser.parse(line);
            if (cmdCall == null) {
                this.terminal.writer().println(CliStrings.messageError("Unknown SQL statement."));
                continue;
            }
            switch (cmdCall.command) {
                case QUIT: 
                case EXIT: {
                    this.callQuit(cmdCall);
                    break;
                }
                case CLEAR: {
                    this.callClear(cmdCall);
                    break;
                }
                case RESET: {
                    this.callReset(cmdCall);
                    break;
                }
                case SET: {
                    this.callSet(cmdCall);
                    break;
                }
                case HELP: {
                    this.callHelp(cmdCall);
                    break;
                }
                case SHOW_TABLES: {
                    this.callShowTables(cmdCall);
                    break;
                }
                case DESCRIBE: {
                    this.callDescribe(cmdCall);
                    break;
                }
                case EXPLAIN: {
                    this.callExplain(cmdCall);
                    break;
                }
                case SELECT: {
                    this.callSelect(cmdCall);
                    break;
                }
                case SOURCE: {
                    this.callSource(cmdCall);
                }
            }
        }
    }

    private void callQuit(SqlCommandParser.SqlCommandCall cmdCall) {
        this.terminal.writer().println("Exiting Flink SQL CLI Client...");
        this.terminal.flush();
        this.isRunning = false;
    }

    private void callClear(SqlCommandParser.SqlCommandCall cmdCall) {
        this.clearTerminal();
    }

    private void callReset(SqlCommandParser.SqlCommandCall cmdCall) {
        this.context.resetSessionProperties();
        this.terminal.writer().println(CliStrings.messageInfo("All session properties have been set to their default values.").toAnsi());
    }

    private void callSet(SqlCommandParser.SqlCommandCall cmdCall) {
        if (cmdCall.operands.length == 0) {
            Map<String, String> properties2;
            try {
                properties2 = this.executor.getSessionProperties(this.context);
            }
            catch (SqlExecutionException e3) {
                this.printException(e3);
                return;
            }
            if (properties2.isEmpty()) {
                this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
            } else {
                properties2.entrySet().stream().map(e2 -> (String)e2.getKey() + "=" + (String)e2.getValue()).sorted().forEach(p -> this.terminal.writer().println((String)p));
            }
        } else {
            this.context.setSessionProperty(cmdCall.operands[0], cmdCall.operands[1]);
            this.terminal.writer().println(CliStrings.messageInfo("Session property has been set.").toAnsi());
        }
        this.terminal.flush();
    }

    private void callHelp(SqlCommandParser.SqlCommandCall cmdCall) {
        this.terminal.writer().println(CliStrings.MESSAGE_HELP);
        this.terminal.flush();
    }

    private void callShowTables(SqlCommandParser.SqlCommandCall cmdCall) {
        List<String> tables;
        try {
            tables = this.executor.listTables(this.context);
        }
        catch (SqlExecutionException e2) {
            this.printException(e2);
            return;
        }
        if (tables.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            tables.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callDescribe(SqlCommandParser.SqlCommandCall cmdCall) {
        TableSchema schema2;
        try {
            schema2 = this.executor.getTableSchema(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printException(e2);
            return;
        }
        this.terminal.writer().println(schema2.toString());
        this.terminal.flush();
    }

    private void callExplain(SqlCommandParser.SqlCommandCall cmdCall) {
        String explanation;
        try {
            explanation = this.executor.explainStatement(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printException(e2);
            return;
        }
        this.terminal.writer().println(explanation);
        this.terminal.flush();
    }

    private void callSelect(SqlCommandParser.SqlCommandCall cmdCall) {
        ResultDescriptor resultDesc;
        try {
            resultDesc = this.executor.executeQuery(this.context, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e2) {
            this.printException(e2);
            return;
        }
        CliResultView view = resultDesc.isMaterialized() ? new CliTableResultView(this, resultDesc) : new CliChangelogResultView(this, resultDesc);
        try {
            view.open();
            this.terminal.writer().println(CliStrings.messageInfo("Result retrieval cancelled.").toAnsi());
            this.terminal.flush();
        }
        catch (SqlExecutionException e3) {
            this.printException(e3);
        }
    }

    private void callSource(SqlCommandParser.SqlCommandCall cmdCall) {
        String stmt;
        String pathString = cmdCall.operands[0];
        if (pathString.isEmpty()) {
            this.printError("Path is invalid.");
            return;
        }
        try {
            Path path = Paths.get(pathString, new String[0]);
            byte[] encoded = Files.readAllBytes(path);
            stmt = new String(encoded, Charset.defaultCharset());
        }
        catch (IOException e2) {
            this.printException(e2);
            return;
        }
        if (stmt.length() > 50000) {
            this.printError("The given file exceeds the maximum number of characters.");
            return;
        }
        this.terminal.writer().println(CliStrings.messageInfo("Executing the following statement:").toAnsi());
        this.terminal.writer().println(new AttributedString(stmt).toString());
        this.terminal.flush();
        this.callSelect(new SqlCommandParser.SqlCommandCall(SqlCommandParser.SqlCommand.SELECT, new String[]{stmt}));
    }

    private void printException(Throwable t) {
        LOG.warn("Could not execute SQL statement.", t);
        this.terminal.writer().println(CliStrings.messageError("Could not execute SQL statement.", t).toAnsi());
        this.terminal.flush();
    }

    private void printError(String e2) {
        this.terminal.writer().println(CliStrings.messageError("Could not execute SQL statement.", e2).toAnsi());
        this.terminal.flush();
    }
}

