/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.fs;

import java.io.IOException;
import java.net.URI;
import java.util.Random;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.FileSystemKind;
import org.apache.flink.core.fs.Path;
import org.apache.flink.util.StringUtils;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Assumptions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public abstract class FileSystemBehaviorTestSuite {
    private static final Random RND = new Random();
    private FileSystem fs;
    private Path basePath;

    protected abstract FileSystem getFileSystem() throws Exception;

    protected abstract Path getBasePath() throws Exception;

    protected abstract FileSystemKind getFileSystemKind();

    @BeforeEach
    void prepare() throws Exception {
        this.fs = this.getFileSystem();
        this.basePath = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(this.basePath);
    }

    @AfterEach
    void cleanup() throws Exception {
        this.fs.delete(this.basePath, true);
    }

    @Test
    void testFileSystemKind() {
        Assertions.assertThat((Comparable)this.fs.getKind()).isEqualTo((Object)this.getFileSystemKind());
    }

    @Test
    void testPathAndScheme() throws Exception {
        Assertions.assertThat((URI)this.fs.getUri()).isEqualTo((Object)this.getBasePath().getFileSystem().getUri());
        Assertions.assertThat((String)this.fs.getUri().getScheme()).isEqualTo(this.getBasePath().toUri().getScheme());
    }

    @Test
    void testHomeDirScheme() {
        Assertions.assertThat((String)this.fs.getHomeDirectory().toUri().getScheme()).isEqualTo(this.fs.getUri().getScheme());
    }

    @Test
    void testWorkDirScheme() {
        Assertions.assertThat((String)this.fs.getWorkingDirectory().toUri().getScheme()).isEqualTo(this.fs.getUri().getScheme());
    }

    @Test
    void testFileExists() throws IOException {
        Path filePath = this.createRandomFileInDirectory(this.basePath);
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isTrue();
    }

    @Test
    void testFileDoesNotExist() throws IOException {
        Assertions.assertThat((boolean)this.fs.exists(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()))).isFalse();
    }

    @Test
    void testExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), false);
    }

    @Test
    void testExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), true);
    }

    @Test
    void testNotExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), false);
    }

    @Test
    void testNotExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), true);
    }

    @Test
    void testExistingEmptyDirectoryDeletion() throws IOException {
        Path path = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, false);
    }

    @Test
    void testExistingEmptyDirectoryRecursiveDeletion() throws IOException {
        Path path = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, true);
    }

    private void testSuccessfulDeletion(Path path, boolean recursionEnabled) throws IOException {
        this.fs.delete(path, recursionEnabled);
        Assertions.assertThat((boolean)this.fs.exists(path)).isFalse();
    }

    @Test
    void testExistingNonEmptyDirectoryDeletion() throws IOException {
        Path directoryPath = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path filePath = this.createRandomFileInDirectory(directoryPath);
        Assertions.assertThatThrownBy(() -> this.fs.delete(directoryPath, false)).isInstanceOf(IOException.class);
        Assertions.assertThat((boolean)this.fs.exists(directoryPath)).isTrue();
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isTrue();
    }

    @Test
    void testExistingNonEmptyDirectoryRecursiveDeletion() throws IOException {
        Path directoryPath = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path filePath = this.createRandomFileInDirectory(directoryPath);
        this.fs.delete(directoryPath, true);
        Assertions.assertThat((boolean)this.fs.exists(directoryPath)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isFalse();
    }

    @Test
    void testExistingNonEmptyDirectoryWithSubDirRecursiveDeletion() throws IOException {
        Path level1SubDirWithFile = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path fileInLevel1Subdir = this.createRandomFileInDirectory(level1SubDirWithFile);
        Path level2SubDirWithFile = new Path(level1SubDirWithFile, FileSystemBehaviorTestSuite.randomName());
        Path fileInLevel2Subdir = this.createRandomFileInDirectory(level2SubDirWithFile);
        this.testSuccessfulDeletion(level1SubDirWithFile, true);
        Assertions.assertThat((boolean)this.fs.exists(fileInLevel1Subdir)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(level2SubDirWithFile)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(fileInLevel2Subdir)).isFalse();
    }

    @Test
    void testMkdirsReturnsTrueWhenCreatingDirectory() throws Exception {
        Path directory = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
        if (this.getFileSystemKind() != FileSystemKind.OBJECT_STORE) {
            Assertions.assertThat((boolean)this.fs.exists(directory)).isTrue();
        }
    }

    @Test
    void testMkdirsCreatesParentDirectories() throws Exception {
        Path directory = new Path(new Path(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName());
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
        if (this.getFileSystemKind() != FileSystemKind.OBJECT_STORE) {
            Assertions.assertThat((boolean)this.fs.exists(directory)).isTrue();
        }
    }

    @Test
    void testMkdirsReturnsTrueForExistingDirectory() throws Exception {
        Path directory = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.createRandomFileInDirectory(directory);
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
    }

    @Test
    protected void testMkdirsFailsForExistingFile() throws Exception {
        this.assumeNotObjectStore();
        Path file = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        Assertions.assertThatThrownBy(() -> this.fs.mkdirs(file)).isInstanceOf(IOException.class);
    }

    @Test
    void testMkdirsFailsWithExistingParentFile() throws Exception {
        this.assumeNotObjectStore();
        Path file = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        Path dirUnderFile = new Path(file, FileSystemBehaviorTestSuite.randomName());
        Assertions.assertThatThrownBy(() -> this.fs.mkdirs(dirUnderFile)).isInstanceOf(IOException.class);
    }

    private static String randomName() {
        return StringUtils.getRandomString((Random)RND, (int)16, (int)16, (char)'a', (char)'z');
    }

    private void createFile(Path file) throws IOException {
        try (FSDataOutputStream out = this.fs.create(file, FileSystem.WriteMode.NO_OVERWRITE);){
            out.write(new byte[]{1, 2, 3, 4, 5, 6, 7, 8});
        }
    }

    private Path createRandomFileInDirectory(Path directory) throws IOException {
        this.fs.mkdirs(directory);
        Path filePath = new Path(directory, FileSystemBehaviorTestSuite.randomName());
        this.createFile(filePath);
        return filePath;
    }

    private void assumeNotObjectStore() {
        ((AbstractBooleanAssert)Assumptions.assumeThat((this.getFileSystemKind() != FileSystemKind.OBJECT_STORE ? 1 : 0) != 0).describedAs("Test does not apply to object stores", new Object[0])).isTrue();
    }
}

