/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.store;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.flink.streaming.connectors.elasticsearch.shaded.org.apache.lucene.store.StoreRateLimiting;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateNonMasterUpdateTask;
import org.elasticsearch.cluster.ClusterStateObserver;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.node.settings.NodeSettingsService;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class IndicesStore
extends AbstractComponent
implements ClusterStateListener,
Closeable {
    public static final String INDICES_STORE_THROTTLE_TYPE = "indices.store.throttle.type";
    public static final String INDICES_STORE_THROTTLE_MAX_BYTES_PER_SEC = "indices.store.throttle.max_bytes_per_sec";
    public static final String INDICES_STORE_DELETE_SHARD_TIMEOUT = "indices.store.delete.shard.timeout";
    public static final String ACTION_SHARD_EXISTS = "internal:index/shard/exists";
    private static final EnumSet<IndexShardState> ACTIVE_STATES = EnumSet.of(IndexShardState.STARTED, IndexShardState.RELOCATED);
    public static final TimeValue DEFAULT_SHARD_DELETE_TIMEOUT = new TimeValue(30L, TimeUnit.SECONDS);
    private final NodeEnvironment nodeEnv;
    private final NodeSettingsService nodeSettingsService;
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private volatile String rateLimitingType;
    private volatile ByteSizeValue rateLimitingThrottle;
    private final StoreRateLimiting rateLimiting = new StoreRateLimiting();
    private final ApplySettings applySettings = new ApplySettings();
    private TimeValue deleteShardTimeout;

    @Inject
    public IndicesStore(Settings settings, NodeEnvironment nodeEnv, NodeSettingsService nodeSettingsService, IndicesService indicesService, ClusterService clusterService, TransportService transportService) {
        super(settings);
        this.nodeEnv = nodeEnv;
        this.nodeSettingsService = nodeSettingsService;
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        transportService.registerHandler(ACTION_SHARD_EXISTS, new ShardActiveRequestHandler());
        this.rateLimitingType = this.componentSettings.get("throttle.type", StoreRateLimiting.Type.MERGE.name());
        this.rateLimiting.setType(this.rateLimitingType);
        this.rateLimitingThrottle = this.componentSettings.getAsBytesSize("throttle.max_bytes_per_sec", new ByteSizeValue(20L, ByteSizeUnit.MB));
        this.rateLimiting.setMaxRate(this.rateLimitingThrottle);
        this.deleteShardTimeout = settings.getAsTime(INDICES_STORE_DELETE_SHARD_TIMEOUT, DEFAULT_SHARD_DELETE_TIMEOUT);
        this.logger.debug("using indices.store.throttle.type [{}], with index.store.throttle.max_bytes_per_sec [{}]", this.rateLimitingType, this.rateLimitingThrottle);
        nodeSettingsService.addListener(this.applySettings);
        clusterService.addLast(this);
    }

    IndicesStore() {
        super(ImmutableSettings.EMPTY);
        this.nodeEnv = null;
        this.nodeSettingsService = null;
        this.indicesService = null;
        this.clusterService = null;
        this.transportService = null;
    }

    public StoreRateLimiting rateLimiting() {
        return this.rateLimiting;
    }

    @Override
    public void close() {
        this.nodeSettingsService.removeListener(this.applySettings);
        this.clusterService.remove(this);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.routingTableChanged()) {
            return;
        }
        if (event.state().blocks().disableStatePersistence()) {
            return;
        }
        for (IndexRoutingTable indexRoutingTable : event.state().routingTable()) {
            for (IndexShardRoutingTable indexShardRoutingTable : indexRoutingTable) {
                ShardId shardId;
                if (!this.shardCanBeDeleted(event.state(), indexShardRoutingTable) || !this.indicesService.canDeleteShardContent(shardId = indexShardRoutingTable.shardId(), event.state().getMetaData().index(shardId.getIndex()))) continue;
                this.deleteShardIfExistElseWhere(event.state(), indexShardRoutingTable);
            }
        }
    }

    boolean shardCanBeDeleted(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        if (indexShardRoutingTable.size() == 0) {
            return false;
        }
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            if (!shardRouting.started()) {
                return false;
            }
            DiscoveryNode node = state.nodes().get(shardRouting.currentNodeId());
            if (node == null) {
                return false;
            }
            if (shardRouting.relocatingNodeId() != null && (node = state.nodes().get(shardRouting.relocatingNodeId())) == null) {
                return false;
            }
            String localNodeId = state.getNodes().localNode().id();
            if (!localNodeId.equals(shardRouting.currentNodeId()) && !localNodeId.equals(shardRouting.relocatingNodeId())) continue;
            return false;
        }
        return true;
    }

    private void deleteShardIfExistElseWhere(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>> requests = new ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>>(indexShardRoutingTable.size());
        String indexUUID = state.getMetaData().index(indexShardRoutingTable.shardId().getIndex()).getUUID();
        ClusterName clusterName = state.getClusterName();
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            DiscoveryNode discoveryNode = state.nodes().get(shardRouting.currentNodeId());
            assert (discoveryNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(discoveryNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
            if (shardRouting.relocatingNodeId() == null) continue;
            DiscoveryNode relocatingNode = state.nodes().get(shardRouting.relocatingNodeId());
            assert (relocatingNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(relocatingNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
        }
        ShardActiveResponseHandler responseHandler = new ShardActiveResponseHandler(indexShardRoutingTable.shardId(), state, requests.size());
        for (Tuple tuple : requests) {
            this.logger.trace("{} sending shard active check to {}", ((ShardActiveRequest)tuple.v2()).shardId, tuple.v1());
            this.transportService.sendRequest((DiscoveryNode)tuple.v1(), ACTION_SHARD_EXISTS, (TransportRequest)tuple.v2(), responseHandler);
        }
    }

    private static class ShardActiveResponse
    extends TransportResponse {
        private boolean shardActive;
        private DiscoveryNode node;

        ShardActiveResponse() {
        }

        ShardActiveResponse(boolean shardActive, DiscoveryNode node) {
            this.shardActive = shardActive;
            this.node = node;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardActive = in.readBoolean();
            this.node = DiscoveryNode.readNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeBoolean(this.shardActive);
            this.node.writeTo(out);
        }
    }

    protected static class ShardActiveRequest
    extends TransportRequest {
        protected TimeValue timeout = DEFAULT_SHARD_DELETE_TIMEOUT;
        protected ClusterName clusterName;
        protected String indexUUID;
        protected ShardId shardId;

        ShardActiveRequest() {
        }

        ShardActiveRequest(ClusterName clusterName, String indexUUID, ShardId shardId, TimeValue timeout) {
            this.shardId = shardId;
            this.indexUUID = indexUUID;
            this.clusterName = clusterName;
            this.timeout = timeout;
            assert (timeout != null);
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.clusterName = ClusterName.readClusterName(in);
            this.indexUUID = in.readString();
            this.shardId = ShardId.readShardId(in);
            if (in.getVersion().onOrAfter(Version.V_1_6_0)) {
                this.timeout = new TimeValue(in.readLong(), TimeUnit.MILLISECONDS);
            }
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.clusterName.writeTo(out);
            out.writeString(this.indexUUID);
            this.shardId.writeTo(out);
            if (out.getVersion().onOrAfter(Version.V_1_6_0)) {
                out.writeLong(this.timeout.millis());
            }
        }
    }

    private class ShardActiveRequestHandler
    extends BaseTransportRequestHandler<ShardActiveRequest> {
        private ShardActiveRequestHandler() {
        }

        @Override
        public ShardActiveRequest newInstance() {
            return new ShardActiveRequest();
        }

        @Override
        public String executor() {
            return "same";
        }

        @Override
        public void messageReceived(final ShardActiveRequest request, final TransportChannel channel) throws Exception {
            assert (request.timeout != null);
            IndexShard indexShard = this.getShard(request);
            if (indexShard == null) {
                IndicesStore.this.logger.trace("{} can't wait for shard being active - not allocated", request.shardId);
                channel.sendResponse(new ShardActiveResponse(false, IndicesStore.this.clusterService.localNode()));
            } else {
                ClusterStateObserver observer = new ClusterStateObserver(IndicesStore.this.clusterService, request.timeout, IndicesStore.this.logger);
                boolean shardActive = this.shardActive(indexShard);
                if (shardActive) {
                    IndicesStore.this.logger.trace("{} shard is already active", request.shardId, shardActive);
                    channel.sendResponse(new ShardActiveResponse(true, IndicesStore.this.clusterService.localNode()));
                } else {
                    IndicesStore.this.logger.trace("{} waiting for next cluster-state update with timeout {}", request.shardId, request.timeout);
                    observer.waitForNextChange(new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        @Override
                        public void onClusterServiceClose() {
                            this.sendResult(false);
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            IndicesStore.this.logger.trace("{} waiting for shard being active timed out after {}", request.shardId, timeout);
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        public void sendResult(boolean shardActive) {
                            try {
                                IndicesStore.this.logger.trace("{} reply with shard active: {}", request.shardId, shardActive);
                                channel.sendResponse(new ShardActiveResponse(shardActive, IndicesStore.this.clusterService.localNode()));
                            }
                            catch (IOException e) {
                                IndicesStore.this.logger.error("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", e, request.shardId);
                            }
                            catch (EsRejectedExecutionException e) {
                                IndicesStore.this.logger.error("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", e, request.shardId);
                            }
                        }
                    }, new ClusterStateObserver.ValidationPredicate(){

                        @Override
                        protected boolean validate(ClusterState newState) {
                            IndexShard indexShard = ShardActiveRequestHandler.this.getShard(request);
                            return indexShard == null || ShardActiveRequestHandler.this.shardActive(indexShard);
                        }
                    });
                }
            }
        }

        private boolean shardActive(IndexShard indexShard) {
            if (indexShard != null) {
                return ACTIVE_STATES.contains((Object)indexShard.state());
            }
            return false;
        }

        private IndexShard getShard(ShardActiveRequest request) {
            ClusterName thisClusterName = IndicesStore.this.clusterService.state().getClusterName();
            if (!thisClusterName.equals(request.clusterName)) {
                IndicesStore.this.logger.trace("shard exists request meant for cluster[{}], but this is cluster[{}], ignoring request", request.clusterName, thisClusterName);
                return null;
            }
            ShardId shardId = request.shardId;
            IndexService indexService = IndicesStore.this.indicesService.indexService(shardId.index().getName());
            if (indexService != null && indexService.indexUUID().equals(request.indexUUID)) {
                return indexService.shard(shardId.id());
            }
            return null;
        }
    }

    private class ShardActiveResponseHandler
    implements TransportResponseHandler<ShardActiveResponse> {
        private final ShardId shardId;
        private final int expectedActiveCopies;
        private final ClusterState clusterState;
        private final AtomicInteger awaitingResponses;
        private final AtomicInteger activeCopies;

        public ShardActiveResponseHandler(ShardId shardId, ClusterState clusterState, int expectedActiveCopies) {
            this.shardId = shardId;
            this.expectedActiveCopies = expectedActiveCopies;
            this.clusterState = clusterState;
            this.awaitingResponses = new AtomicInteger(expectedActiveCopies);
            this.activeCopies = new AtomicInteger();
        }

        @Override
        public ShardActiveResponse newInstance() {
            return new ShardActiveResponse();
        }

        @Override
        public void handleResponse(ShardActiveResponse response) {
            IndicesStore.this.logger.trace("{} is {}active on node {}", this.shardId, response.shardActive ? "" : "not ", response.node);
            if (response.shardActive) {
                this.activeCopies.incrementAndGet();
            }
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public void handleException(TransportException exp) {
            IndicesStore.this.logger.debug("shards active request failed for {}", exp, this.shardId);
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public String executor() {
            return "same";
        }

        private void allNodesResponded() {
            if (this.activeCopies.get() != this.expectedActiveCopies) {
                IndicesStore.this.logger.trace("not deleting shard {}, expected {} active copies, but only {} found active copies", this.shardId, this.expectedActiveCopies, this.activeCopies.get());
                return;
            }
            ClusterState latestClusterState = IndicesStore.this.clusterService.state();
            if (this.clusterState.getVersion() != latestClusterState.getVersion()) {
                IndicesStore.this.logger.trace("not deleting shard {}, the latest cluster state version[{}] is not equal to cluster state before shard active api call [{}]", this.shardId, latestClusterState.getVersion(), this.clusterState.getVersion());
                return;
            }
            IndicesStore.this.clusterService.submitStateUpdateTask("indices_store ([" + this.shardId + "] active fully on other nodes)", new ClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    if (ShardActiveResponseHandler.this.clusterState.getVersion() != currentState.getVersion()) {
                        IndicesStore.this.logger.trace("not deleting shard {}, the update task state version[{}] is not equal to cluster state before shard active api call [{}]", ShardActiveResponseHandler.this.shardId, currentState.getVersion(), ShardActiveResponseHandler.this.clusterState.getVersion());
                        return currentState;
                    }
                    try {
                        IndicesStore.this.indicesService.deleteShardStore("no longer used", ShardActiveResponseHandler.this.shardId, currentState);
                    }
                    catch (Throwable ex) {
                        IndicesStore.this.logger.debug("{} failed to delete unallocated shard, ignoring", ex, ShardActiveResponseHandler.this.shardId);
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    IndicesStore.this.logger.error("{} unexpected error during deletion of unallocated shard", t, ShardActiveResponseHandler.this.shardId);
                }
            });
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            ByteSizeValue rateLimitingThrottle;
            String rateLimitingType = settings.get(IndicesStore.INDICES_STORE_THROTTLE_TYPE, IndicesStore.this.rateLimitingType);
            StoreRateLimiting.Type.fromString(rateLimitingType);
            if (!rateLimitingType.equals(IndicesStore.this.rateLimitingType)) {
                IndicesStore.this.logger.info("updating indices.store.throttle.type from [{}] to [{}]", IndicesStore.this.rateLimitingType, rateLimitingType);
                IndicesStore.this.rateLimitingType = rateLimitingType;
                IndicesStore.this.rateLimiting.setType(rateLimitingType);
            }
            if (!(rateLimitingThrottle = settings.getAsBytesSize(IndicesStore.INDICES_STORE_THROTTLE_MAX_BYTES_PER_SEC, IndicesStore.this.rateLimitingThrottle)).equals(IndicesStore.this.rateLimitingThrottle)) {
                IndicesStore.this.logger.info("updating indices.store.throttle.max_bytes_per_sec from [{}] to [{}], note, type is [{}]", IndicesStore.this.rateLimitingThrottle, rateLimitingThrottle, IndicesStore.this.rateLimitingType);
                IndicesStore.this.rateLimitingThrottle = rateLimitingThrottle;
                IndicesStore.this.rateLimiting.setMaxRate(rateLimitingThrottle);
            }
        }
    }
}

