/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.EngineConfig;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
extends AbstractLifecycleComponent<IndexingMemoryController> {
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final ByteSizeValue translogBuffer;
    private final ByteSizeValue minShardTranslogBufferSize;
    private final ByteSizeValue maxShardTranslogBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private volatile ScheduledFuture scheduler;
    private static final EnumSet<IndexShardState> CAN_UPDATE_INDEX_BUFFER_STATES = EnumSet.of(IndexShardState.RECOVERING, IndexShardState.POST_RECOVERY, IndexShardState.STARTED, IndexShardState.RELOCATED);

    @Inject
    public IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        super(settings);
        ByteSizeValue translogBuffer;
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.componentSettings.get("index_buffer_size", "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.componentSettings.getAsBytesSize("min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.componentSettings.getAsBytesSize("max_index_buffer_size", null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.componentSettings.getAsBytesSize("min_shard_index_buffer_size", new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = this.componentSettings.getAsBytesSize("max_shard_index_buffer_size", new ByteSizeValue(512L, ByteSizeUnit.MB));
        String translogBufferSetting = this.componentSettings.get("translog_buffer_size", "1%");
        if (translogBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(translogBufferSetting.substring(0, translogBufferSetting.length() - 1));
            translogBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minTranslogBuffer = this.componentSettings.getAsBytesSize("min_translog_buffer_size", new ByteSizeValue(256L, ByteSizeUnit.KB));
            ByteSizeValue maxTranslogBuffer = this.componentSettings.getAsBytesSize("max_translog_buffer_size", null);
            if (translogBuffer.bytes() < minTranslogBuffer.bytes()) {
                translogBuffer = minTranslogBuffer;
            }
            if (maxTranslogBuffer != null && translogBuffer.bytes() > maxTranslogBuffer.bytes()) {
                translogBuffer = maxTranslogBuffer;
            }
        } else {
            translogBuffer = ByteSizeValue.parseBytesSizeValue(translogBufferSetting, null);
        }
        this.translogBuffer = translogBuffer;
        this.minShardTranslogBufferSize = this.componentSettings.getAsBytesSize("min_shard_translog_buffer_size", new ByteSizeValue(2L, ByteSizeUnit.KB));
        this.maxShardTranslogBufferSize = this.componentSettings.getAsBytesSize("max_shard_translog_buffer_size", new ByteSizeValue(64L, ByteSizeUnit.KB));
        this.inactiveTime = this.componentSettings.getAsTime("shard_inactive_time", TimeValue.timeValueMinutes(5L));
        this.interval = this.componentSettings.getAsTime("interval", TimeValue.timeValueSeconds(30L));
        this.logger.debug("using index_buffer_size [{}], with min_shard_index_buffer_size [{}], max_shard_index_buffer_size [{}], shard_inactive_time [{}]", this.indexingBuffer, this.minShardIndexBufferSize, this.maxShardIndexBufferSize, this.inactiveTime);
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        this.scheduler = this.threadPool.scheduleWithFixedDelay(new ShardsIndicesStatusChecker(), this.interval);
    }

    @Override
    protected void doStop() throws ElasticsearchException {
        FutureUtils.cancel(this.scheduler);
        this.scheduler = null;
    }

    @Override
    protected void doClose() throws ElasticsearchException {
    }

    public ByteSizeValue indexingBufferSize() {
        return this.indexingBuffer;
    }

    static class ShardIndexingStatus {
        long translogId = -1L;
        int translogNumberOfOperations = -1;
        boolean activeIndexing = true;
        long time = -1L;

        ShardIndexingStatus() {
        }
    }

    private static enum ShardStatusChangeType {
        ADDED,
        DELETED,
        BECAME_ACTIVE,
        BECAME_INACTIVE;

    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        private final Map<ShardId, ShardIndexingStatus> shardsIndicesStatus = new HashMap<ShardId, ShardIndexingStatus>();

        ShardsIndicesStatusChecker() {
        }

        @Override
        public void run() {
            EnumSet<ShardStatusChangeType> changes = EnumSet.noneOf(ShardStatusChangeType.class);
            changes.addAll(this.purgeDeletedAndClosedShards());
            ArrayList<IndexShard> activeToInactiveIndexingShards = Lists.newArrayList();
            int activeShards = this.updateShardStatuses(changes, activeToInactiveIndexingShards);
            for (IndexShard indexShard : activeToInactiveIndexingShards) {
                try {
                    indexShard.markAsInactive();
                }
                catch (EngineClosedException e) {
                }
                catch (FlushNotAllowedEngineException e) {}
            }
            if (!changes.isEmpty()) {
                this.calcAndSetShardBuffers(activeShards, "[" + changes + "]");
            }
        }

        private int updateShardStatuses(EnumSet<ShardStatusChangeType> changes, List<IndexShard> activeToInactiveIndexingShards) {
            int activeShards = 0;
            for (IndexService indexService : IndexingMemoryController.this.indicesService) {
                for (IndexShard indexShard : indexService) {
                    if (!CAN_UPDATE_INDEX_BUFFER_STATES.contains((Object)indexShard.state())) continue;
                    long time = IndexingMemoryController.this.threadPool.estimatedTimeInMillis();
                    Translog translog = indexShard.translog();
                    ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                    if (status == null) {
                        status = new ShardIndexingStatus();
                        this.shardsIndicesStatus.put(indexShard.shardId(), status);
                        changes.add(ShardStatusChangeType.ADDED);
                    }
                    if (status.translogId == translog.currentId() && translog.estimatedNumberOfOperations() == 0) {
                        if (status.time == -1L) {
                            status.time = time;
                        }
                        if (status.activeIndexing && time - status.time > IndexingMemoryController.this.inactiveTime.millis() && indexShard.mergeStats().getCurrent() == 0L) {
                            activeToInactiveIndexingShards.add(indexShard);
                            status.activeIndexing = false;
                            changes.add(ShardStatusChangeType.BECAME_INACTIVE);
                            IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as inactive (inactive_time[{}]) indexing wise, setting size to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), IndexingMemoryController.this.inactiveTime, EngineConfig.INACTIVE_SHARD_INDEXING_BUFFER);
                        }
                    } else {
                        if (!status.activeIndexing) {
                            status.activeIndexing = true;
                            changes.add(ShardStatusChangeType.BECAME_ACTIVE);
                            IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as active indexing wise", indexShard.shardId().index().name(), indexShard.shardId().id());
                        }
                        status.time = -1L;
                    }
                    status.translogId = translog.currentId();
                    status.translogNumberOfOperations = translog.estimatedNumberOfOperations();
                    if (!status.activeIndexing) continue;
                    ++activeShards;
                }
            }
            return activeShards;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private EnumSet<ShardStatusChangeType> purgeDeletedAndClosedShards() {
            EnumSet<ShardStatusChangeType> changes = EnumSet.noneOf(ShardStatusChangeType.class);
            Iterator<ShardId> statusShardIdIterator = this.shardsIndicesStatus.keySet().iterator();
            while (statusShardIdIterator.hasNext()) {
                ShardId statusShardId = statusShardIdIterator.next();
                IndexService indexService = IndexingMemoryController.this.indicesService.indexService(statusShardId.getIndex());
                boolean remove = false;
                try {
                    if (indexService == null) {
                        remove = true;
                        continue;
                    }
                    IndexShard indexShard = indexService.shard(statusShardId.id());
                    if (indexShard == null) {
                        remove = true;
                        continue;
                    }
                    remove = !CAN_UPDATE_INDEX_BUFFER_STATES.contains((Object)indexShard.state());
                }
                finally {
                    if (!remove) continue;
                    changes.add(ShardStatusChangeType.DELETED);
                    statusShardIdIterator.remove();
                }
            }
            return changes;
        }

        private void calcAndSetShardBuffers(int activeShards, String reason) {
            ByteSizeValue shardTranslogBufferSize;
            if (activeShards == 0) {
                return;
            }
            ByteSizeValue shardIndexingBufferSize = new ByteSizeValue(IndexingMemoryController.this.indexingBuffer.bytes() / (long)activeShards);
            if (shardIndexingBufferSize.bytes() < IndexingMemoryController.this.minShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.minShardIndexBufferSize;
            }
            if (shardIndexingBufferSize.bytes() > IndexingMemoryController.this.maxShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.maxShardIndexBufferSize;
            }
            if ((shardTranslogBufferSize = new ByteSizeValue(IndexingMemoryController.this.translogBuffer.bytes() / (long)activeShards)).bytes() < IndexingMemoryController.this.minShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.minShardTranslogBufferSize;
            }
            if (shardTranslogBufferSize.bytes() > IndexingMemoryController.this.maxShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.maxShardTranslogBufferSize;
            }
            IndexingMemoryController.this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to indexing=[{}], translog=[{}]", reason, IndexingMemoryController.this.indexingBuffer, activeShards, shardIndexingBufferSize, shardTranslogBufferSize);
            for (IndexService indexService : IndexingMemoryController.this.indicesService) {
                for (IndexShard indexShard : indexService) {
                    IndexShardState state = indexShard.state();
                    if (!CAN_UPDATE_INDEX_BUFFER_STATES.contains((Object)state)) {
                        IndexingMemoryController.this.logger.trace("shard [{}] is not yet ready for index buffer update. index shard state: [{}]", new Object[]{indexShard.shardId(), state});
                        continue;
                    }
                    ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                    if (status != null && !status.activeIndexing) continue;
                    try {
                        indexShard.updateBufferSize(shardIndexingBufferSize, shardTranslogBufferSize);
                    }
                    catch (EngineClosedException e) {
                    }
                    catch (FlushNotAllowedEngineException e) {
                    }
                    catch (Exception e) {
                        IndexingMemoryController.this.logger.warn("failed to set shard {} index buffer to [{}]", indexShard.shardId(), shardIndexingBufferSize);
                    }
                }
            }
        }
    }
}

