/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import org.apache.flink.streaming.connectors.elasticsearch5.shaded.org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.flink.streaming.connectors.elasticsearch5.shaded.org.apache.lucene.util.IOUtils;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptResponse;
import org.elasticsearch.action.admin.cluster.storedscripts.GetStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.breaker.CircuitBreakingException;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.env.Environment;
import org.elasticsearch.script.CompiledScript;
import org.elasticsearch.script.ExecutableScript;
import org.elasticsearch.script.GeneralScriptException;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptContextRegistry;
import org.elasticsearch.script.ScriptEngineRegistry;
import org.elasticsearch.script.ScriptEngineService;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptMetaData;
import org.elasticsearch.script.ScriptMetrics;
import org.elasticsearch.script.ScriptModes;
import org.elasticsearch.script.ScriptSettings;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.ScriptType;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class ScriptService
extends AbstractComponent
implements Closeable,
ClusterStateListener {
    static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    public static final Setting<Integer> SCRIPT_CACHE_SIZE_SETTING = Setting.intSetting("script.cache.max_size", 100, 0, Setting.Property.NodeScope);
    public static final Setting<TimeValue> SCRIPT_CACHE_EXPIRE_SETTING = Setting.positiveTimeSetting("script.cache.expire", TimeValue.timeValueMillis(0L), Setting.Property.NodeScope);
    public static final Setting<Boolean> SCRIPT_AUTO_RELOAD_ENABLED_SETTING = Setting.boolSetting("script.auto_reload_enabled", true, Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_SIZE_IN_BYTES = Setting.intSetting("script.max_size_in_bytes", 65535, Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_COMPILATIONS_PER_MINUTE = Setting.intSetting("script.max_compilations_per_minute", 15, 0, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final Collection<ScriptEngineService> scriptEngines;
    private final Map<String, ScriptEngineService> scriptEnginesByLang;
    private final Map<String, ScriptEngineService> scriptEnginesByExt;
    private final ConcurrentMap<CacheKey, CompiledScript> staticCache = ConcurrentCollections.newConcurrentMap();
    private final Cache<CacheKey, CompiledScript> cache;
    private final Path scriptsDirectory;
    private final ScriptModes scriptModes;
    private final ScriptContextRegistry scriptContextRegistry;
    private final ScriptMetrics scriptMetrics = new ScriptMetrics();
    private ClusterState clusterState;
    private int totalCompilesPerMinute;
    private long lastInlineCompileTime;
    private double scriptsPerMinCounter;
    private double compilesAllowedPerNano;

    public ScriptService(Settings settings, Environment env, ResourceWatcherService resourceWatcherService, ScriptEngineRegistry scriptEngineRegistry, ScriptContextRegistry scriptContextRegistry, ScriptSettings scriptSettings) throws IOException {
        super(settings);
        TimeValue cacheExpire;
        Objects.requireNonNull(scriptEngineRegistry);
        Objects.requireNonNull(scriptContextRegistry);
        Objects.requireNonNull(scriptSettings);
        if (Strings.hasLength(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING))) {
            throw new IllegalArgumentException("script.disable_dynamic is not a supported setting, replace with fine-grained script settings. \nDynamic scripts can be enabled for all languages and all operations by replacing `script.disable_dynamic: false` with `script.inline: true` and `script.stored: true` in elasticsearch.yml");
        }
        this.scriptEngines = scriptEngineRegistry.getRegisteredLanguages().values();
        this.scriptContextRegistry = scriptContextRegistry;
        int cacheMaxSize = SCRIPT_CACHE_SIZE_SETTING.get(settings);
        CacheBuilder<CacheKey, CompiledScript> cacheBuilder = CacheBuilder.builder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.setMaximumWeight(cacheMaxSize);
        }
        if ((cacheExpire = SCRIPT_CACHE_EXPIRE_SETTING.get(settings)).getNanos() != 0L) {
            cacheBuilder.setExpireAfterAccess(cacheExpire);
        }
        this.logger.debug("using script cache with max_size [{}], expire [{}]", (Object)cacheMaxSize, (Object)cacheExpire);
        this.cache = cacheBuilder.removalListener(new ScriptCacheRemovalListener()).build();
        HashMap<String, ScriptEngineService> enginesByLangBuilder = new HashMap<String, ScriptEngineService>();
        HashMap<String, ScriptEngineService> enginesByExtBuilder = new HashMap<String, ScriptEngineService>();
        for (ScriptEngineService scriptEngine : this.scriptEngines) {
            String language = scriptEngineRegistry.getLanguage(scriptEngine.getClass());
            enginesByLangBuilder.put(language, scriptEngine);
            enginesByExtBuilder.put(scriptEngine.getExtension(), scriptEngine);
        }
        this.scriptEnginesByLang = Collections.unmodifiableMap(enginesByLangBuilder);
        this.scriptEnginesByExt = Collections.unmodifiableMap(enginesByExtBuilder);
        this.scriptModes = new ScriptModes(scriptSettings, settings);
        this.scriptsDirectory = env.scriptsFile();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Using scripts directory [{}] ", (Object)this.scriptsDirectory);
        }
        FileWatcher fileWatcher = new FileWatcher(this.scriptsDirectory);
        fileWatcher.addListener(new ScriptChangesListener());
        if (SCRIPT_AUTO_RELOAD_ENABLED_SETTING.get(settings).booleanValue()) {
            resourceWatcherService.add(fileWatcher);
        } else {
            fileWatcher.init();
        }
        this.lastInlineCompileTime = System.nanoTime();
        this.setMaxCompilationsPerMinute(SCRIPT_MAX_COMPILATIONS_PER_MINUTE.get(settings));
    }

    void registerClusterSettingsListeners(ClusterSettings clusterSettings) {
        clusterSettings.addSettingsUpdateConsumer(SCRIPT_MAX_COMPILATIONS_PER_MINUTE, this::setMaxCompilationsPerMinute);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.scriptEngines);
    }

    private ScriptEngineService getScriptEngineServiceForLang(String lang) {
        ScriptEngineService scriptEngineService = this.scriptEnginesByLang.get(lang);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        return scriptEngineService;
    }

    private ScriptEngineService getScriptEngineServiceForFileExt(String fileExtension) {
        ScriptEngineService scriptEngineService = this.scriptEnginesByExt.get(fileExtension);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script file extension not supported [" + fileExtension + "]");
        }
        return scriptEngineService;
    }

    void setMaxCompilationsPerMinute(Integer newMaxPerMinute) {
        this.totalCompilesPerMinute = newMaxPerMinute;
        this.scriptsPerMinCounter = this.totalCompilesPerMinute;
        this.compilesAllowedPerNano = (double)this.totalCompilesPerMinute / (double)TimeValue.timeValueMinutes(1L).nanos();
    }

    public CompiledScript compile(Script script, ScriptContext scriptContext, Map<String, String> params) {
        if (script == null) {
            throw new IllegalArgumentException("The parameter script (Script) must not be null.");
        }
        if (scriptContext == null) {
            throw new IllegalArgumentException("The parameter scriptContext (ScriptContext) must not be null.");
        }
        String lang = script.getLang();
        ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(lang);
        if (!this.canExecuteScript(lang, script.getType(), scriptContext)) {
            throw new IllegalStateException("scripts of type [" + script.getType() + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are disabled");
        }
        boolean expression = "expression".equals(script.getLang());
        boolean notSupported = scriptContext.getKey().equals(ScriptContext.Standard.UPDATE.getKey());
        if (expression && notSupported) {
            throw new UnsupportedOperationException("scripts of type [" + script.getType() + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are not supported");
        }
        return this.compileInternal(script, params);
    }

    void checkCompilationLimit() {
        long now = System.nanoTime();
        long timePassed = now - this.lastInlineCompileTime;
        this.lastInlineCompileTime = now;
        this.scriptsPerMinCounter += (double)timePassed * this.compilesAllowedPerNano;
        if (this.scriptsPerMinCounter > (double)this.totalCompilesPerMinute) {
            this.scriptsPerMinCounter = this.totalCompilesPerMinute;
        }
        if (this.scriptsPerMinCounter >= 1.0) {
            this.scriptsPerMinCounter -= 1.0;
        } else {
            throw new CircuitBreakingException("[script] Too many dynamic script compilations within one minute, max: [" + this.totalCompilesPerMinute + "/min]; please use on-disk, indexed, or scripts with parameters instead; this limit can be changed by the [" + SCRIPT_MAX_COMPILATIONS_PER_MINUTE.getKey() + "] setting");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CompiledScript compileInternal(Script script, Map<String, String> params) {
        CacheKey cacheKey;
        CompiledScript compiledScript;
        if (script == null) {
            throw new IllegalArgumentException("The parameter script (Script) must not be null.");
        }
        String lang = script.getLang();
        ScriptType type = script.getType();
        String name = script.getIdOrCode();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Compiling lang: [{}] type: [{}] script: {}", (Object)lang, (Object)type, (Object)name);
        }
        ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(lang);
        if (type == ScriptType.FILE) {
            CacheKey cacheKey2 = new CacheKey(scriptEngineService, name, null, params);
            CompiledScript compiledScript2 = (CompiledScript)this.staticCache.get(cacheKey2);
            if (compiledScript2 == null) {
                throw new IllegalArgumentException("Unable to find on disk file script [" + name + "] using lang [" + lang + "]");
            }
            return compiledScript2;
        }
        String code = script.getIdOrCode();
        if (type == ScriptType.STORED) {
            IndexedScript indexedScript = new IndexedScript(lang, name);
            name = indexedScript.id;
            code = this.getScriptFromClusterState(indexedScript.lang, indexedScript.id);
        }
        if ((compiledScript = this.cache.get(cacheKey = new CacheKey(scriptEngineService, type == ScriptType.INLINE ? null : name, code, params))) != null) {
            return compiledScript;
        }
        ScriptService scriptService = this;
        synchronized (scriptService) {
            compiledScript = this.cache.get(cacheKey);
            if (compiledScript == null) {
                try {
                    String actualName;
                    String string = actualName = type == ScriptType.INLINE ? null : name;
                    if (this.logger.isTraceEnabled()) {
                        this.logger.trace("compiling script, type: [{}], lang: [{}], params: [{}]", (Object)type, (Object)lang, (Object)params);
                    }
                    this.checkCompilationLimit();
                    compiledScript = new CompiledScript(type, name, lang, scriptEngineService.compile(actualName, code, params));
                }
                catch (ScriptException good) {
                    throw good;
                }
                catch (Exception exception) {
                    throw new GeneralScriptException("Failed to compile " + type + " script [" + name + "] using lang [" + lang + "]", exception);
                }
                this.scriptMetrics.onCompilation();
                this.cache.put(cacheKey, compiledScript);
            }
            return compiledScript;
        }
    }

    private String validateScriptLanguage(String scriptLang) {
        Objects.requireNonNull(scriptLang);
        if (!this.scriptEnginesByLang.containsKey(scriptLang)) {
            throw new IllegalArgumentException("script_lang not supported [" + scriptLang + "]");
        }
        return scriptLang;
    }

    String getScriptFromClusterState(String scriptLang, String id) {
        scriptLang = this.validateScriptLanguage(scriptLang);
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            throw new ResourceNotFoundException("Unable to find script [" + scriptLang + "/" + id + "] in cluster state", new Object[0]);
        }
        String script = scriptMetadata.getScript(scriptLang, id);
        if (script == null) {
            throw new ResourceNotFoundException("Unable to find script [" + scriptLang + "/" + id + "] in cluster state", new Object[0]);
        }
        return script;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    void validateStoredScript(String id, String scriptLang, BytesReference scriptBytes) {
        this.validateScriptSize(id, scriptBytes.length());
        String script = ScriptMetaData.parseStoredScript(scriptBytes);
        if (!Strings.hasLength(scriptBytes)) throw new IllegalArgumentException("Unable to find script in : " + scriptBytes.utf8ToString());
        try {
            ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(scriptLang);
            if (this.isAnyScriptContextEnabled(scriptLang, ScriptType.STORED)) {
                Object compiled = scriptEngineService.compile(id, script, Collections.emptyMap());
                if (compiled != null) return;
                throw new IllegalArgumentException("Unable to parse [" + script + "] lang [" + scriptLang + "] (ScriptService.compile returned null)");
            }
            this.logger.warn("skipping compile of script [{}], lang [{}] as all scripted operations are disabled for indexed scripts", (Object)script, (Object)scriptLang);
            return;
        }
        catch (ScriptException good) {
            throw good;
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Unable to parse [" + script + "] lang [" + scriptLang + "]", e);
        }
    }

    public void storeScript(ClusterService clusterService, final PutStoredScriptRequest request, ActionListener<PutStoredScriptResponse> listener) {
        final String scriptLang = this.validateScriptLanguage(request.scriptLang());
        this.validateStoredScript(request.id(), scriptLang, request.script());
        clusterService.submitStateUpdateTask("put-script-" + request.id(), new AckedClusterStateUpdateTask<PutStoredScriptResponse>((AckedRequest)request, listener){

            @Override
            protected PutStoredScriptResponse newResponse(boolean acknowledged) {
                return new PutStoredScriptResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return ScriptService.innerStoreScript(currentState, scriptLang, request);
            }
        });
    }

    static ClusterState innerStoreScript(ClusterState currentState, String validatedScriptLang, PutStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
        ScriptMetaData.Builder scriptMetadataBuilder = new ScriptMetaData.Builder(scriptMetadata);
        scriptMetadataBuilder.storeScript(validatedScriptLang, request.id(), request.script());
        MetaData.Builder metaDataBuilder = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", scriptMetadataBuilder.build());
        return ClusterState.builder(currentState).metaData(metaDataBuilder).build();
    }

    public void deleteStoredScript(ClusterService clusterService, final DeleteStoredScriptRequest request, ActionListener<DeleteStoredScriptResponse> listener) {
        final String scriptLang = this.validateScriptLanguage(request.scriptLang());
        clusterService.submitStateUpdateTask("delete-script-" + request.id(), new AckedClusterStateUpdateTask<DeleteStoredScriptResponse>((AckedRequest)request, listener){

            @Override
            protected DeleteStoredScriptResponse newResponse(boolean acknowledged) {
                return new DeleteStoredScriptResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return ScriptService.innerDeleteScript(currentState, scriptLang, request);
            }
        });
    }

    static ClusterState innerDeleteScript(ClusterState currentState, String validatedLang, DeleteStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
        ScriptMetaData.Builder scriptMetadataBuilder = new ScriptMetaData.Builder(scriptMetadata);
        scriptMetadataBuilder.deleteScript(validatedLang, request.id());
        MetaData.Builder metaDataBuilder = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", scriptMetadataBuilder.build());
        return ClusterState.builder(currentState).metaData(metaDataBuilder).build();
    }

    public String getStoredScript(ClusterState state, GetStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)state.metaData().custom("stored_scripts");
        if (scriptMetadata != null) {
            return scriptMetadata.getScript(request.lang(), request.id());
        }
        return null;
    }

    public ExecutableScript executable(Script script, ScriptContext scriptContext) {
        return this.executable(this.compile(script, scriptContext, script.getOptions()), script.getParams());
    }

    public ExecutableScript executable(CompiledScript compiledScript, Map<String, Object> params) {
        return this.getScriptEngineServiceForLang(compiledScript.lang()).executable(compiledScript, params);
    }

    public SearchScript search(SearchLookup lookup, Script script, ScriptContext scriptContext) {
        CompiledScript compiledScript = this.compile(script, scriptContext, script.getOptions());
        return this.search(lookup, compiledScript, script.getParams());
    }

    public SearchScript search(SearchLookup lookup, CompiledScript compiledScript, Map<String, Object> params) {
        return this.getScriptEngineServiceForLang(compiledScript.lang()).search(compiledScript, lookup, params);
    }

    private boolean isAnyScriptContextEnabled(String lang, ScriptType scriptType) {
        for (ScriptContext scriptContext : this.scriptContextRegistry.scriptContexts()) {
            if (!this.canExecuteScript(lang, scriptType, scriptContext)) continue;
            return true;
        }
        return false;
    }

    private boolean canExecuteScript(String lang, ScriptType scriptType, ScriptContext scriptContext) {
        assert (lang != null);
        if (!this.scriptContextRegistry.isSupportedContext(scriptContext)) {
            throw new IllegalArgumentException("script context [" + scriptContext.getKey() + "] not supported");
        }
        return this.scriptModes.getScriptEnabled(lang, scriptType, scriptContext);
    }

    public ScriptStats stats() {
        return this.scriptMetrics.stats();
    }

    private void validateScriptSize(String identifier, int scriptSizeInBytes) {
        int allowedScriptSizeInBytes = SCRIPT_MAX_SIZE_IN_BYTES.get(this.settings);
        if (scriptSizeInBytes > allowedScriptSizeInBytes) {
            String message = LoggerMessageFormat.format("Limit of script size in bytes [{}] has been exceeded for script [{}] with size [{}]", allowedScriptSizeInBytes, identifier, scriptSizeInBytes);
            throw new IllegalArgumentException(message);
        }
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        this.clusterState = event.state();
    }

    private static class IndexedScript {
        private final String lang;
        private final String id;

        IndexedScript(String lang, String script) {
            this.lang = lang;
            String[] parts = script.split("/");
            if (parts.length == 1) {
                this.id = script;
            } else {
                if (parts.length != 3) {
                    throw new IllegalArgumentException("Illegal index script format [" + script + "] should be /lang/id");
                }
                if (!parts[1].equals(this.lang)) {
                    throw new IllegalStateException("Conflicting script language, found [" + parts[1] + "] expected + [" + this.lang + "]");
                }
                this.id = parts[2];
            }
        }
    }

    private static final class CacheKey {
        final String lang;
        final String name;
        final String code;
        final Map<String, String> params;

        private CacheKey(ScriptEngineService service, String name, String code, Map<String, String> params) {
            this.lang = service.getType();
            this.name = name;
            this.code = code;
            this.params = params;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            if (!this.lang.equals(cacheKey.lang)) {
                return false;
            }
            if (this.name != null ? !this.name.equals(cacheKey.name) : cacheKey.name != null) {
                return false;
            }
            if (this.code != null ? !this.code.equals(cacheKey.code) : cacheKey.code != null) {
                return false;
            }
            return this.params.equals(cacheKey.params);
        }

        public int hashCode() {
            int result = this.lang.hashCode();
            result = 31 * result + (this.name != null ? this.name.hashCode() : 0);
            result = 31 * result + (this.code != null ? this.code.hashCode() : 0);
            result = 31 * result + this.params.hashCode();
            return result;
        }
    }

    private class ScriptChangesListener
    implements FileChangesListener {
        private ScriptChangesListener() {
        }

        private Tuple<String, String> getScriptNameExt(Path file) {
            Path scriptPath = ScriptService.this.scriptsDirectory.relativize(file);
            int extIndex = scriptPath.toString().lastIndexOf(46);
            if (extIndex <= 0) {
                return null;
            }
            String ext = scriptPath.toString().substring(extIndex + 1);
            if (ext.isEmpty()) {
                return null;
            }
            String scriptName = scriptPath.toString().substring(0, extIndex).replace(scriptPath.getFileSystem().getSeparator(), "_");
            return new Tuple<String, String>(scriptName, ext);
        }

        @Override
        public void onFileInit(Path file) {
            block19: {
                ScriptEngineService engineService;
                Tuple<String, String> scriptNameExt = this.getScriptNameExt(file);
                if (scriptNameExt == null) {
                    ScriptService.this.logger.debug("Skipped script with invalid extension : [{}]", (Object)file);
                    return;
                }
                if (ScriptService.this.logger.isTraceEnabled()) {
                    ScriptService.this.logger.trace("Loading script file : [{}]", (Object)file);
                }
                if ((engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2())) == null) {
                    ScriptService.this.logger.warn("No script engine found for [{}]", (Object)scriptNameExt.v2());
                } else {
                    try {
                        if (ScriptService.this.isAnyScriptContextEnabled(engineService.getType(), ScriptType.FILE)) {
                            ScriptService.this.logger.info("compiling script file [{}]", (Object)file.toAbsolutePath());
                            try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(file, new OpenOption[0]), StandardCharsets.UTF_8);){
                                String script = Streams.copyToString(reader);
                                String name = scriptNameExt.v1();
                                CacheKey cacheKey = new CacheKey(engineService, name, null, Collections.emptyMap());
                                Object executable = engineService.compile(file.getFileName().toString(), script, Collections.emptyMap());
                                CompiledScript compiledScript = new CompiledScript(ScriptType.FILE, name, engineService.getType(), executable);
                                ScriptService.this.staticCache.put(cacheKey, compiledScript);
                                ScriptService.this.scriptMetrics.onCompilation();
                                break block19;
                            }
                        }
                        ScriptService.this.logger.warn("skipping compile of script file [{}] as all scripted operations are disabled for file scripts", (Object)file.toAbsolutePath());
                    }
                    catch (Exception e) {
                        ScriptService.this.logger.warn(() -> new ParameterizedMessage("failed to load/compile script [{}]", scriptNameExt.v1()), (Throwable)e);
                    }
                }
            }
        }

        @Override
        public void onFileCreated(Path file) {
            this.onFileInit(file);
        }

        @Override
        public void onFileDeleted(Path file) {
            Tuple<String, String> scriptNameExt = this.getScriptNameExt(file);
            if (scriptNameExt != null) {
                ScriptEngineService engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2());
                assert (engineService != null);
                ScriptService.this.logger.info("removing script file [{}]", (Object)file.toAbsolutePath());
                ScriptService.this.staticCache.remove(new CacheKey(engineService, scriptNameExt.v1(), null, Collections.emptyMap()));
            }
        }

        @Override
        public void onFileChanged(Path file) {
            this.onFileInit(file);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, CompiledScript> {
        private ScriptCacheRemovalListener() {
        }

        @Override
        public void onRemoval(RemovalNotification<CacheKey, CompiledScript> notification) {
            if (ScriptService.this.logger.isDebugEnabled()) {
                ScriptService.this.logger.debug("removed {} from cache, reason: {}", (Object)notification.getValue(), (Object)notification.getRemovalReason());
            }
            ScriptService.this.scriptMetrics.onCacheEviction();
        }
    }
}

