/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import java.util.Set;
import org.apache.flink.streaming.connectors.elasticsearch5.shaded.com.carrotsearch.hppc.cursors.ObjectCursor;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;

public class DiskThresholdDecider
extends AllocationDecider {
    public static final String NAME = "disk_threshold";
    private final DiskThresholdSettings diskThresholdSettings;

    public DiskThresholdDecider(Settings settings, ClusterSettings clusterSettings) {
        super(settings);
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
    }

    static long sizeOfRelocatingShards(RoutingNode node, RoutingAllocation allocation, boolean subtractShardsMovingAway, String dataPath) {
        ClusterInfo clusterInfo = allocation.clusterInfo();
        long totalSize = 0L;
        for (ShardRouting routing : node.shardsWithState(ShardRoutingState.RELOCATING, ShardRoutingState.INITIALIZING)) {
            String actualPath = clusterInfo.getDataPath(routing);
            if (!dataPath.equals(actualPath)) continue;
            if (routing.initializing() && routing.relocatingNodeId() != null) {
                totalSize += DiskThresholdDecider.getExpectedShardSize(routing, allocation, 0L);
                continue;
            }
            if (!subtractShardsMovingAway || !routing.relocating()) continue;
            totalSize -= DiskThresholdDecider.getExpectedShardSize(routing, allocation, 0L);
        }
        return totalSize;
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        boolean skipLowTresholdChecks;
        ClusterInfo clusterInfo = allocation.clusterInfo();
        ImmutableOpenMap<String, DiskUsage> usages = clusterInfo.getNodeMostAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        double usedDiskThresholdLow = 100.0 - this.diskThresholdSettings.getFreeDiskThresholdLow();
        double usedDiskThresholdHigh = 100.0 - this.diskThresholdSettings.getFreeDiskThresholdHigh();
        DiskUsage usage = this.getDiskUsage(node, allocation, usages, false);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        double usedDiskPercentage = usage.getUsedDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("node [{}] has {}% used disk", (Object)node.nodeId(), (Object)usedDiskPercentage);
        }
        boolean bl = skipLowTresholdChecks = shardRouting.primary() && !shardRouting.active() && shardRouting.recoverySource().getType() == RecoverySource.Type.EMPTY_STORE;
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes()) {
            if (!skipLowTresholdChecks) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation", (Object)this.diskThresholdSettings.getFreeBytesThresholdLow(), (Object)freeBytes, (Object)node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "the node is above the low watermark and has less than required [%s] free, free: [%s]", this.diskThresholdSettings.getFreeBytesThresholdLow(), new ByteSizeValue(freeBytes));
            }
            if (freeBytes > this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, but allowing allocation because primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdLow(), (Object)freeBytes, (Object)node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "the node is above the low watermark, but this primary shard has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytes, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the node is above the high watermark even though this shard has never been allocated and has less than required [%s] free on node, free: [%s]", this.diskThresholdSettings.getFreeBytesThresholdHigh(), new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
            if (!skipLowTresholdChecks) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("more than the allowed {} used disk threshold ({} used) on node [{}], preventing allocation", (Object)Strings.format1Decimals(usedDiskThresholdLow, "%"), (Object)Strings.format1Decimals(usedDiskPercentage, "%"), (Object)node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "the node is above the low watermark and has more than allowed [%s%%] used disk, free: [%s%%]", usedDiskThresholdLow, freeDiskPercentage);
            }
            if (freeDiskPercentage > this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("more than the allowed {} used disk threshold ({} used) on node [{}], but allowing allocation because primary has never been allocated", (Object)Strings.format1Decimals(usedDiskThresholdLow, "%"), (Object)Strings.format1Decimals(usedDiskPercentage, "%"), (Object)node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "the node is above the low watermark, but this primary shard has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", (Object)Strings.format1Decimals(this.diskThresholdSettings.getFreeDiskThresholdHigh(), "%"), (Object)Strings.format1Decimals(freeDiskPercentage, "%"), (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the node is above the high watermark even though this shard has never been allocated and has more than allowed [%s%%] used disk, free: [%s%%]", usedDiskThresholdHigh, freeDiskPercentage);
        }
        long shardSize = DiskThresholdDecider.getExpectedShardSize(shardRouting, allocation, 0L);
        double freeSpaceAfterShard = this.freeDiskPercentageAfterShardAssigned(usage, shardSize);
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
            this.logger.warn("after allocating, node [{}] would have less than the required {} free bytes threshold ({} bytes free), preventing allocation", (Object)node.nodeId(), (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytesAfterShard);
            return allocation.decision(Decision.NO, NAME, "after allocating the shard to this node, it would be above the high watermark and have less than required [%s] free, free: [%s]", this.diskThresholdSettings.getFreeBytesThresholdLow(), new ByteSizeValue(freeBytesAfterShard));
        }
        if (freeSpaceAfterShard < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
            this.logger.warn("after allocating, node [{}] would have more than the allowed {} free disk threshold ({} free), preventing allocation", (Object)node.nodeId(), (Object)Strings.format1Decimals(this.diskThresholdSettings.getFreeDiskThresholdHigh(), "%"), (Object)Strings.format1Decimals(freeSpaceAfterShard, "%"));
            return allocation.decision(Decision.NO, NAME, "after allocating the shard to this node, it would be above the high watermark and have more than allowed [%s%%] used disk, free: [%s%%]", usedDiskThresholdLow, freeSpaceAfterShard);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s], shard size: [%s], free after allocating shard: [%s]", new ByteSizeValue(freeBytes), new ByteSizeValue(shardSize), new ByteSizeValue(freeBytesAfterShard));
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!shardRouting.currentNodeId().equals(node.nodeId())) {
            throw new IllegalArgumentException("Shard [" + shardRouting + "] is not allocated on node: [" + node.nodeId() + "]");
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        ImmutableOpenMap<String, DiskUsage> usages = clusterInfo.getNodeLeastAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        DiskUsage usage = this.getDiskUsage(node, allocation, usages, true);
        String dataPath = clusterInfo.getDataPath(shardRouting);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("node [{}] has {}% free disk ({} bytes)", (Object)node.nodeId(), (Object)freeDiskPercentage, (Object)freeBytes);
        }
        if (dataPath == null || !usage.getPath().equals(dataPath)) {
            return allocation.decision(Decision.YES, NAME, "this shard is not allocated on the most utilized disk and can remain", new Object[0]);
        }
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, shard cannot remain", (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytes, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocating this shard this node would be above the high watermark and there would be less than required [%s] free on node, free: [%s]", this.diskThresholdSettings.getFreeBytesThresholdHigh(), new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {}% free disk threshold ({}% free) on node {}, shard cannot remain", (Object)this.diskThresholdSettings.getFreeDiskThresholdHigh(), (Object)freeDiskPercentage, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocating this shard this node would be above the high watermark and there would be less than required [%s%%] free disk on node, free: [%s%%]", this.diskThresholdSettings.getFreeDiskThresholdHigh(), freeDiskPercentage);
        }
        return allocation.decision(Decision.YES, NAME, "there is enough disk on this node for the shard to remain, free: [%s]", new ByteSizeValue(freeBytes));
    }

    private DiskUsage getDiskUsage(RoutingNode node, RoutingAllocation allocation, ImmutableOpenMap<String, DiskUsage> usages, boolean subtractLeavingShards) {
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", (Object)node.nodeId(), (Object)usage.getTotalBytes(), (Object)usage.getFreeBytes(), (Object)usage.getFreeDiskAsPercentage());
            }
        }
        if (this.diskThresholdSettings.includeRelocations()) {
            long relocatingShardsSize = DiskThresholdDecider.sizeOfRelocatingShards(node, allocation, subtractLeavingShards, usage.getPath());
            DiskUsage usageIncludingRelocations = new DiskUsage(node.nodeId(), node.node().getName(), usage.getPath(), usage.getTotalBytes(), usage.getFreeBytes() - relocatingShardsSize);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("usage without relocations: {}", (Object)usage);
                this.logger.trace("usage with relocations: [{} bytes] {}", (Object)relocatingShardsSize, (Object)usageIncludingRelocations);
            }
            usage = usageIncludingRelocations;
        }
        return usage;
    }

    DiskUsage averageUsage(RoutingNode node, ImmutableOpenMap<String, DiskUsage> usages) {
        if (usages.size() == 0) {
            return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", 0L, 0L);
        }
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (ObjectCursor<DiskUsage> objectCursor : usages.values()) {
            totalBytes += ((DiskUsage)objectCursor.value).getTotalBytes();
            freeBytes += ((DiskUsage)objectCursor.value).getFreeBytes();
        }
        return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    double freeDiskPercentageAfterShardAssigned(DiskUsage usage, Long shardSize) {
        shardSize = shardSize == null ? 0L : shardSize;
        DiskUsage newUsage = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getPath(), usage.getTotalBytes(), usage.getFreeBytes() - shardSize);
        return newUsage.getFreeDiskAsPercentage();
    }

    private Decision earlyTerminate(RoutingAllocation allocation, ImmutableOpenMap<String, DiskUsage> usages) {
        if (!this.diskThresholdSettings.isEnabled()) {
            return allocation.decision(Decision.YES, NAME, "the disk threshold decider is disabled", new Object[0]);
        }
        if (allocation.nodes().getDataNodes().size() <= 1) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("only a single data node is present, allowing allocation");
            }
            return allocation.decision(Decision.YES, NAME, "there is only a single data node present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("cluster info unavailable for disk threshold decider, allowing allocation.");
            }
            return allocation.decision(Decision.YES, NAME, "the cluster info is unavailable", new Object[0]);
        }
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("unable to determine disk usages for disk-aware allocation, allowing allocation");
            }
            return allocation.decision(Decision.YES, NAME, "disk usages are unavailable", new Object[0]);
        }
        return null;
    }

    public static long getExpectedShardSize(ShardRouting shard, RoutingAllocation allocation, long defaultValue) {
        IndexMetaData metaData = allocation.metaData().getIndexSafe(shard.index());
        ClusterInfo info = allocation.clusterInfo();
        if (metaData.getMergeSourceIndex() != null && !shard.active() && shard.recoverySource().getType() == RecoverySource.Type.LOCAL_SHARDS) {
            long targetShardSize = 0L;
            Index mergeSourceIndex = metaData.getMergeSourceIndex();
            IndexMetaData sourceIndexMeta = allocation.metaData().getIndexSafe(mergeSourceIndex);
            Set<ShardId> shardIds = IndexMetaData.selectShrinkShards(shard.id(), sourceIndexMeta, metaData.getNumberOfShards());
            for (IndexShardRoutingTable shardRoutingTable : allocation.routingTable().index(mergeSourceIndex.getName())) {
                if (!shardIds.contains(shardRoutingTable.shardId())) continue;
                targetShardSize += info.getShardSize(shardRoutingTable.primaryShard(), 0L);
            }
            return targetShardSize == 0L ? defaultValue : targetShardSize;
        }
        return info.getShardSize(shard, defaultValue);
    }
}

