/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import org.apache.flink.streaming.connectors.elasticsearch5.shaded.org.apache.lucene.util.IOUtils;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptResponse;
import org.elasticsearch.action.admin.cluster.storedscripts.GetStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.breaker.CircuitBreakingException;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.env.Environment;
import org.elasticsearch.script.CompiledScript;
import org.elasticsearch.script.ExecutableScript;
import org.elasticsearch.script.GeneralScriptException;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptContextRegistry;
import org.elasticsearch.script.ScriptEngineRegistry;
import org.elasticsearch.script.ScriptEngineService;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptMetaData;
import org.elasticsearch.script.ScriptMetrics;
import org.elasticsearch.script.ScriptModes;
import org.elasticsearch.script.ScriptSettings;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.ScriptType;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.script.StoredScriptSource;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class ScriptService
extends AbstractComponent
implements Closeable,
ClusterStateListener {
    static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    public static final Setting<Integer> SCRIPT_CACHE_SIZE_SETTING = Setting.intSetting("script.cache.max_size", 100, 0, Setting.Property.NodeScope);
    public static final Setting<TimeValue> SCRIPT_CACHE_EXPIRE_SETTING = Setting.positiveTimeSetting("script.cache.expire", TimeValue.timeValueMillis(0L), Setting.Property.NodeScope);
    public static final Setting<Boolean> SCRIPT_AUTO_RELOAD_ENABLED_SETTING = Setting.boolSetting("script.auto_reload_enabled", true, Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_SIZE_IN_BYTES = Setting.intSetting("script.max_size_in_bytes", 65535, Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_COMPILATIONS_PER_MINUTE = Setting.intSetting("script.max_compilations_per_minute", 15, 0, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final Collection<ScriptEngineService> scriptEngines;
    private final Map<String, ScriptEngineService> scriptEnginesByLang;
    private final Map<String, ScriptEngineService> scriptEnginesByExt;
    private final ConcurrentMap<CacheKey, CompiledScript> staticCache = ConcurrentCollections.newConcurrentMap();
    private final Cache<CacheKey, CompiledScript> cache;
    private final Path scriptsDirectory;
    private final ScriptModes scriptModes;
    private final ScriptContextRegistry scriptContextRegistry;
    private final ScriptMetrics scriptMetrics = new ScriptMetrics();
    private ClusterState clusterState;
    private int totalCompilesPerMinute;
    private long lastInlineCompileTime;
    private double scriptsPerMinCounter;
    private double compilesAllowedPerNano;

    public ScriptService(Settings settings, Environment env, ResourceWatcherService resourceWatcherService, ScriptEngineRegistry scriptEngineRegistry, ScriptContextRegistry scriptContextRegistry, ScriptSettings scriptSettings) throws IOException {
        super(settings);
        TimeValue cacheExpire;
        Objects.requireNonNull(scriptEngineRegistry);
        Objects.requireNonNull(scriptContextRegistry);
        Objects.requireNonNull(scriptSettings);
        if (Strings.hasLength(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING))) {
            throw new IllegalArgumentException("script.disable_dynamic is not a supported setting, replace with fine-grained script settings. \nDynamic scripts can be enabled for all languages and all operations by replacing `script.disable_dynamic: false` with `script.inline: true` and `script.stored: true` in elasticsearch.yml");
        }
        this.scriptEngines = scriptEngineRegistry.getRegisteredLanguages().values();
        this.scriptContextRegistry = scriptContextRegistry;
        int cacheMaxSize = SCRIPT_CACHE_SIZE_SETTING.get(settings);
        CacheBuilder<CacheKey, CompiledScript> cacheBuilder = CacheBuilder.builder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.setMaximumWeight(cacheMaxSize);
        }
        if ((cacheExpire = SCRIPT_CACHE_EXPIRE_SETTING.get(settings)).getNanos() != 0L) {
            cacheBuilder.setExpireAfterAccess(cacheExpire);
        }
        this.logger.debug("using script cache with max_size [{}], expire [{}]", (Object)cacheMaxSize, (Object)cacheExpire);
        this.cache = cacheBuilder.removalListener(new ScriptCacheRemovalListener()).build();
        HashMap<String, ScriptEngineService> enginesByLangBuilder = new HashMap<String, ScriptEngineService>();
        HashMap<String, ScriptEngineService> enginesByExtBuilder = new HashMap<String, ScriptEngineService>();
        for (ScriptEngineService scriptEngine : this.scriptEngines) {
            String language = scriptEngineRegistry.getLanguage(scriptEngine.getClass());
            enginesByLangBuilder.put(language, scriptEngine);
            enginesByExtBuilder.put(scriptEngine.getExtension(), scriptEngine);
        }
        this.scriptEnginesByLang = Collections.unmodifiableMap(enginesByLangBuilder);
        this.scriptEnginesByExt = Collections.unmodifiableMap(enginesByExtBuilder);
        this.scriptModes = new ScriptModes(scriptSettings, settings);
        this.scriptsDirectory = env.scriptsFile();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Using scripts directory [{}] ", (Object)this.scriptsDirectory);
        }
        FileWatcher fileWatcher = new FileWatcher(this.scriptsDirectory);
        fileWatcher.addListener(new ScriptChangesListener());
        if (SCRIPT_AUTO_RELOAD_ENABLED_SETTING.get(settings).booleanValue()) {
            resourceWatcherService.add(fileWatcher);
        } else {
            fileWatcher.init();
        }
        this.lastInlineCompileTime = System.nanoTime();
        this.setMaxCompilationsPerMinute(SCRIPT_MAX_COMPILATIONS_PER_MINUTE.get(settings));
    }

    void registerClusterSettingsListeners(ClusterSettings clusterSettings) {
        clusterSettings.addSettingsUpdateConsumer(SCRIPT_MAX_COMPILATIONS_PER_MINUTE, this::setMaxCompilationsPerMinute);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.scriptEngines);
    }

    private ScriptEngineService getScriptEngineServiceForLang(String lang) {
        ScriptEngineService scriptEngineService = this.scriptEnginesByLang.get(lang);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        return scriptEngineService;
    }

    private ScriptEngineService getScriptEngineServiceForFileExt(String fileExtension) {
        ScriptEngineService scriptEngineService = this.scriptEnginesByExt.get(fileExtension);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script file extension not supported [" + fileExtension + "]");
        }
        return scriptEngineService;
    }

    void setMaxCompilationsPerMinute(Integer newMaxPerMinute) {
        this.totalCompilesPerMinute = newMaxPerMinute;
        this.scriptsPerMinCounter = this.totalCompilesPerMinute;
        this.compilesAllowedPerNano = (double)this.totalCompilesPerMinute / (double)TimeValue.timeValueMinutes(1L).nanos();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompiledScript compile(Script script, ScriptContext scriptContext) {
        Objects.requireNonNull(script);
        Objects.requireNonNull(scriptContext);
        ScriptType type = script.getType();
        String lang = script.getLang();
        String idOrCode = script.getIdOrCode();
        Map<String, String> options = script.getOptions();
        String id = idOrCode;
        if (type == ScriptType.STORED) {
            String[] path = id.split("/");
            if (path.length == 3) {
                if (lang != null && !lang.equals(path[1])) {
                    throw new IllegalStateException("conflicting script languages, found [" + path[1] + "] but expected [" + lang + "]");
                }
                id = path[2];
                this.deprecationLogger.deprecated("use of </lang/id> [" + idOrCode + "] for looking up stored scripts/templates has been deprecated, use only <id> [" + id + "] instead", new Object[0]);
            } else if (path.length != 1) {
                throw new IllegalArgumentException("illegal stored script format [" + id + "] use only <id>");
            }
            StoredScriptSource source = this.getScriptFromClusterState(id, lang);
            lang = source.getLang();
            idOrCode = source.getCode();
            options = source.getOptions();
        }
        boolean expression = "expression".equals(script.getLang());
        boolean notSupported = scriptContext.getKey().equals(ScriptContext.Standard.UPDATE.getKey());
        if (expression && notSupported) {
            throw new UnsupportedOperationException("scripts of type [" + script.getType() + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are not supported");
        }
        ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(lang);
        if (!this.canExecuteScript(lang, type, scriptContext)) {
            throw new IllegalStateException("scripts of type [" + script.getType() + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are disabled");
        }
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("compiling lang: [{}] type: [{}] script: {}", (Object)lang, (Object)type, (Object)idOrCode);
        }
        if (type == ScriptType.FILE) {
            CacheKey cacheKey = new CacheKey(lang, idOrCode, options);
            CompiledScript compiledScript = (CompiledScript)this.staticCache.get(cacheKey);
            if (compiledScript == null) {
                throw new IllegalArgumentException("unable to find file script [" + idOrCode + "] using lang [" + lang + "]");
            }
            return compiledScript;
        }
        CacheKey cacheKey = new CacheKey(lang, idOrCode, options);
        CompiledScript compiledScript = this.cache.get(cacheKey);
        if (compiledScript != null) {
            return compiledScript;
        }
        ScriptService scriptService = this;
        synchronized (scriptService) {
            compiledScript = this.cache.get(cacheKey);
            if (compiledScript == null) {
                try {
                    if (this.logger.isTraceEnabled()) {
                        this.logger.trace("compiling script, type: [{}], lang: [{}], options: [{}]", (Object)type, (Object)lang, options);
                    }
                    this.checkCompilationLimit();
                    compiledScript = new CompiledScript(type, id, lang, scriptEngineService.compile(id, idOrCode, options));
                }
                catch (ScriptException good) {
                    throw good;
                }
                catch (Exception exception) {
                    throw new GeneralScriptException("Failed to compile " + type + " script [" + id + "] using lang [" + lang + "]", exception);
                }
                this.scriptMetrics.onCompilation();
                this.cache.put(cacheKey, compiledScript);
            }
            return compiledScript;
        }
    }

    void checkCompilationLimit() {
        long now = System.nanoTime();
        long timePassed = now - this.lastInlineCompileTime;
        this.lastInlineCompileTime = now;
        this.scriptsPerMinCounter += (double)timePassed * this.compilesAllowedPerNano;
        if (this.scriptsPerMinCounter > (double)this.totalCompilesPerMinute) {
            this.scriptsPerMinCounter = this.totalCompilesPerMinute;
        }
        if (this.scriptsPerMinCounter >= 1.0) {
            this.scriptsPerMinCounter -= 1.0;
        } else {
            throw new CircuitBreakingException("[script] Too many dynamic script compilations within one minute, max: [" + this.totalCompilesPerMinute + "/min]; please use on-disk, indexed, or scripts with parameters instead; this limit can be changed by the [" + SCRIPT_MAX_COMPILATIONS_PER_MINUTE.getKey() + "] setting");
        }
    }

    public boolean isLangSupported(String lang) {
        Objects.requireNonNull(lang);
        return this.scriptEnginesByLang.containsKey(lang);
    }

    StoredScriptSource getScriptFromClusterState(String id, String lang) {
        if (lang != null && !this.isLangSupported(lang)) {
            throw new IllegalArgumentException("unable to get stored script with unsupported lang [" + lang + "]");
        }
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "]" + (lang == null ? "" : " using lang [" + lang + "]") + " in cluster state", new Object[0]);
        }
        StoredScriptSource source = scriptMetadata.getStoredScript(id, lang);
        if (source == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "]" + (lang == null ? "" : " using lang [" + lang + "]") + " in cluster state", new Object[0]);
        }
        return source;
    }

    public void putStoredScript(ClusterService clusterService, final PutStoredScriptRequest request, ActionListener<PutStoredScriptResponse> listener) {
        StoredScriptSource source;
        block7: {
            int max = SCRIPT_MAX_SIZE_IN_BYTES.get(this.settings);
            if (request.content().length() > max) {
                throw new IllegalArgumentException("exceeded max allowed stored script size in bytes [" + max + "] with size [" + request.content().length() + "] for script [" + request.id() + "]");
            }
            source = StoredScriptSource.parse(request.lang(), request.content(), request.xContentType());
            if (!this.isLangSupported(source.getLang())) {
                throw new IllegalArgumentException("unable to put stored script with unsupported lang [" + source.getLang() + "]");
            }
            try {
                ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(source.getLang());
                if (this.isAnyScriptContextEnabled(source.getLang(), ScriptType.STORED)) {
                    Object compiled = scriptEngineService.compile(request.id(), source.getCode(), Collections.emptyMap());
                    if (compiled == null) {
                        throw new IllegalArgumentException("failed to parse/compile stored script [" + request.id() + "]" + (source.getCode() == null ? "" : " using code [" + source.getCode() + "]"));
                    }
                    break block7;
                }
                throw new IllegalArgumentException("cannot put stored script [" + request.id() + "], stored scripts cannot be run under any context");
            }
            catch (ScriptException good) {
                throw good;
            }
            catch (Exception exception) {
                throw new IllegalArgumentException("failed to parse/compile stored script [" + request.id() + "]", exception);
            }
        }
        clusterService.submitStateUpdateTask("put-script-" + request.id(), new AckedClusterStateUpdateTask<PutStoredScriptResponse>((AckedRequest)request, listener){

            @Override
            protected PutStoredScriptResponse newResponse(boolean acknowledged) {
                return new PutStoredScriptResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.putStoredScript(smd, request.id(), source);
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public void deleteStoredScript(ClusterService clusterService, final DeleteStoredScriptRequest request, ActionListener<DeleteStoredScriptResponse> listener) {
        if (request.lang() != null && !this.isLangSupported(request.lang())) {
            throw new IllegalArgumentException("unable to delete stored script with unsupported lang [" + request.lang() + "]");
        }
        clusterService.submitStateUpdateTask("delete-script-" + request.id(), new AckedClusterStateUpdateTask<DeleteStoredScriptResponse>((AckedRequest)request, listener){

            @Override
            protected DeleteStoredScriptResponse newResponse(boolean acknowledged) {
                return new DeleteStoredScriptResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.deleteStoredScript(smd, request.id(), request.lang());
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public StoredScriptSource getStoredScript(ClusterState state, GetStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)state.metaData().custom("stored_scripts");
        if (scriptMetadata != null) {
            return scriptMetadata.getStoredScript(request.id(), request.lang());
        }
        return null;
    }

    public ExecutableScript executable(Script script, ScriptContext scriptContext) {
        return this.executable(this.compile(script, scriptContext), script.getParams());
    }

    public ExecutableScript executable(CompiledScript compiledScript, Map<String, Object> params) {
        return this.getScriptEngineServiceForLang(compiledScript.lang()).executable(compiledScript, params);
    }

    public SearchScript search(SearchLookup lookup, Script script, ScriptContext scriptContext) {
        CompiledScript compiledScript = this.compile(script, scriptContext);
        return this.search(lookup, compiledScript, script.getParams());
    }

    public SearchScript search(SearchLookup lookup, CompiledScript compiledScript, Map<String, Object> params) {
        return this.getScriptEngineServiceForLang(compiledScript.lang()).search(compiledScript, lookup, params);
    }

    private boolean isAnyScriptContextEnabled(String lang, ScriptType scriptType) {
        for (ScriptContext scriptContext : this.scriptContextRegistry.scriptContexts()) {
            if (!this.canExecuteScript(lang, scriptType, scriptContext)) continue;
            return true;
        }
        return false;
    }

    private boolean canExecuteScript(String lang, ScriptType scriptType, ScriptContext scriptContext) {
        assert (lang != null);
        if (!this.scriptContextRegistry.isSupportedContext(scriptContext)) {
            throw new IllegalArgumentException("script context [" + scriptContext.getKey() + "] not supported");
        }
        return this.scriptModes.getScriptEnabled(lang, scriptType, scriptContext);
    }

    public ScriptStats stats() {
        return this.scriptMetrics.stats();
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        this.clusterState = event.state();
    }

    private static final class CacheKey {
        final String lang;
        final String idOrCode;
        final Map<String, String> options;

        private CacheKey(String lang, String idOrCode, Map<String, String> options) {
            this.lang = lang;
            this.idOrCode = idOrCode;
            this.options = options;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            if (this.lang != null ? !this.lang.equals(cacheKey.lang) : cacheKey.lang != null) {
                return false;
            }
            if (!this.idOrCode.equals(cacheKey.idOrCode)) {
                return false;
            }
            return this.options != null ? this.options.equals(cacheKey.options) : cacheKey.options == null;
        }

        public int hashCode() {
            int result = this.lang != null ? this.lang.hashCode() : 0;
            result = 31 * result + this.idOrCode.hashCode();
            result = 31 * result + (this.options != null ? this.options.hashCode() : 0);
            return result;
        }
    }

    private class ScriptChangesListener
    implements FileChangesListener {
        private ScriptChangesListener() {
        }

        private Tuple<String, String> getScriptNameExt(Path file) {
            Path scriptPath = ScriptService.this.scriptsDirectory.relativize(file);
            int extIndex = scriptPath.toString().lastIndexOf(46);
            if (extIndex <= 0) {
                return null;
            }
            String ext = scriptPath.toString().substring(extIndex + 1);
            if (ext.isEmpty()) {
                return null;
            }
            String scriptName = scriptPath.toString().substring(0, extIndex).replace(scriptPath.getFileSystem().getSeparator(), "_");
            return new Tuple<String, String>(scriptName, ext);
        }

        @Override
        public void onFileInit(Path file) {
            block34: {
                ScriptEngineService engineService;
                Tuple<String, String> scriptNameExt = this.getScriptNameExt(file);
                if (scriptNameExt == null) {
                    ScriptService.this.logger.debug("Skipped script with invalid extension : [{}]", (Object)file);
                    return;
                }
                if (ScriptService.this.logger.isTraceEnabled()) {
                    ScriptService.this.logger.trace("Loading script file : [{}]", (Object)file);
                }
                if ((engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2())) == null) {
                    ScriptService.this.logger.warn("No script engine found for [{}]", (Object)scriptNameExt.v2());
                } else {
                    try {
                        if (ScriptService.this.isAnyScriptContextEnabled(engineService.getType(), ScriptType.FILE)) {
                            ScriptService.this.logger.info("compiling script file [{}]", (Object)file.toAbsolutePath());
                            try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(file, new OpenOption[0]), StandardCharsets.UTF_8);){
                                String script = Streams.copyToString(reader);
                                String id = scriptNameExt.v1();
                                CacheKey cacheKey = new CacheKey(engineService.getType(), id, null);
                                Object executable = engineService.compile(file.getFileName().toString(), script, Collections.emptyMap());
                                CompiledScript compiledScript = new CompiledScript(ScriptType.FILE, id, engineService.getType(), executable);
                                ScriptService.this.staticCache.put(cacheKey, compiledScript);
                                ScriptService.this.scriptMetrics.onCompilation();
                                break block34;
                            }
                        }
                        ScriptService.this.logger.warn("skipping compile of script file [{}] as all scripted operations are disabled for file scripts", (Object)file.toAbsolutePath());
                    }
                    catch (ScriptException e) {
                        try (XContentBuilder builder = JsonXContent.contentBuilder();){
                            builder.prettyPrint();
                            builder.startObject();
                            ElasticsearchException.generateThrowableXContent(builder, ToXContent.EMPTY_PARAMS, e);
                            builder.endObject();
                            ScriptService.this.logger.warn("failed to load/compile script [{}]: {}", (Object)scriptNameExt.v1(), (Object)builder.string());
                        }
                        catch (IOException ioe) {
                            ioe.addSuppressed(e);
                            ScriptService.this.logger.warn(() -> new ParameterizedMessage("failed to log an appropriate warning after failing to load/compile script [{}]", scriptNameExt.v1()), (Throwable)ioe);
                        }
                        ScriptService.this.logger.debug(() -> new ParameterizedMessage("failed to load/compile script [{}]. full exception:", scriptNameExt.v1()), (Throwable)e);
                    }
                    catch (Exception e) {
                        ScriptService.this.logger.warn(() -> new ParameterizedMessage("failed to load/compile script [{}]", scriptNameExt.v1()), (Throwable)e);
                    }
                }
            }
        }

        @Override
        public void onFileCreated(Path file) {
            this.onFileInit(file);
        }

        @Override
        public void onFileDeleted(Path file) {
            Tuple<String, String> scriptNameExt = this.getScriptNameExt(file);
            if (scriptNameExt != null) {
                ScriptEngineService engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2());
                assert (engineService != null);
                ScriptService.this.logger.info("removing script file [{}]", (Object)file.toAbsolutePath());
                ScriptService.this.staticCache.remove(new CacheKey(engineService.getType(), scriptNameExt.v1(), null));
            }
        }

        @Override
        public void onFileChanged(Path file) {
            this.onFileInit(file);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, CompiledScript> {
        private ScriptCacheRemovalListener() {
        }

        @Override
        public void onRemoval(RemovalNotification<CacheKey, CompiledScript> notification) {
            if (ScriptService.this.logger.isDebugEnabled()) {
                ScriptService.this.logger.debug("removed {} from cache, reason: {}", (Object)notification.getValue(), (Object)notification.getRemovalReason());
            }
            ScriptService.this.scriptMetrics.onCacheEviction();
        }
    }
}

