/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.shard;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Map;
import org.apache.flink.streaming.connectors.elasticsearch5.shaded.org.apache.lucene.util.IOUtils;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardStateMetaData;

public final class ShardPath {
    public static final String INDEX_FOLDER_NAME = "index";
    public static final String TRANSLOG_FOLDER_NAME = "translog";
    private final Path path;
    private final ShardId shardId;
    private final Path shardStatePath;
    private final boolean isCustomDataPath;

    public ShardPath(boolean isCustomDataPath, Path dataPath, Path shardStatePath, ShardId shardId) {
        assert (dataPath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "dataPath must end with the shard ID but didn't: " + dataPath.toString();
        assert (shardStatePath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "shardStatePath must end with the shard ID but didn't: " + dataPath.toString();
        assert (dataPath.getParent().getFileName().toString().equals(shardId.getIndex().getUUID())) : "dataPath must end with index path id but didn't: " + dataPath.toString();
        assert (shardStatePath.getParent().getFileName().toString().equals(shardId.getIndex().getUUID())) : "shardStatePath must end with index path id but didn't: " + dataPath.toString();
        if (isCustomDataPath && dataPath.equals(shardStatePath)) {
            throw new IllegalArgumentException("shard state path must be different to the data path when using custom data paths");
        }
        this.isCustomDataPath = isCustomDataPath;
        this.path = dataPath;
        this.shardId = shardId;
        this.shardStatePath = shardStatePath;
    }

    public Path resolveTranslog() {
        return this.path.resolve(TRANSLOG_FOLDER_NAME);
    }

    public Path resolveIndex() {
        return this.path.resolve(INDEX_FOLDER_NAME);
    }

    public Path getDataPath() {
        return this.path;
    }

    public boolean exists() {
        return Files.exists(this.path, new LinkOption[0]);
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public Path getShardStatePath() {
        return this.shardStatePath;
    }

    public Path getRootDataPath() {
        Path noIndexShardId = this.getDataPath().getParent().getParent();
        return this.isCustomDataPath ? noIndexShardId : noIndexShardId.getParent();
    }

    public Path getRootStatePath() {
        return this.getShardStatePath().getParent().getParent().getParent();
    }

    public boolean isCustomDataPath() {
        return this.isCustomDataPath;
    }

    public static ShardPath loadShardPath(Logger logger, NodeEnvironment env, ShardId shardId, IndexSettings indexSettings) throws IOException {
        String indexUUID = indexSettings.getUUID();
        Path[] paths = env.availableShardPaths(shardId);
        Path loadedPath = null;
        for (Path path : paths) {
            ShardStateMetaData load = ShardStateMetaData.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, path);
            if (load == null) continue;
            if (!load.indexUUID.equals(indexUUID) && !"_na_".equals(load.indexUUID)) {
                logger.warn("{} found shard on path: [{}] with a different index UUID - this shard seems to be leftover from a different index with the same name. Remove the leftover shard in order to reuse the path with the current index", (Object)shardId, (Object)path);
                throw new IllegalStateException(shardId + " index UUID in shard state was: " + load.indexUUID + " expected: " + indexUUID + " on shard path: " + path);
            }
            if (loadedPath == null) {
                loadedPath = path;
                continue;
            }
            throw new IllegalStateException(shardId + " more than one shard state found");
        }
        if (loadedPath == null) {
            return null;
        }
        Path statePath = loadedPath;
        Path dataPath = indexSettings.hasCustomDataPath() ? env.resolveCustomLocation(indexSettings, shardId) : statePath;
        logger.debug("{} loaded data path [{}], state path [{}]", (Object)shardId, (Object)dataPath, (Object)statePath);
        return new ShardPath(indexSettings.hasCustomDataPath(), dataPath, statePath, shardId);
    }

    public static void deleteLeftoverShardDirectory(Logger logger, NodeEnvironment env, ShardLock lock, IndexSettings indexSettings) throws IOException {
        Path[] paths;
        String indexUUID = indexSettings.getUUID();
        for (Path path : paths = env.availableShardPaths(lock.getShardId())) {
            ShardStateMetaData load = ShardStateMetaData.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, path);
            if (load == null || load.indexUUID.equals(indexUUID) || "_na_".equals(load.indexUUID)) continue;
            logger.warn("{} deleting leftover shard on path: [{}] with a different index UUID", (Object)lock.getShardId(), (Object)path);
            assert (Files.isDirectory(path, new LinkOption[0])) : path + " is not a directory";
            NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
            IOUtils.rm(path);
        }
    }

    public static ShardPath selectNewPathForShard(NodeEnvironment env, ShardId shardId, IndexSettings indexSettings, long avgShardSizeInBytes, Map<Path, Integer> dataPathToShardCount) throws IOException {
        Path statePath;
        Path dataPath;
        if (indexSettings.hasCustomDataPath()) {
            dataPath = env.resolveCustomLocation(indexSettings, shardId);
            statePath = env.nodePaths()[0].resolve(shardId);
        } else {
            BigInteger totFreeSpace = BigInteger.ZERO;
            for (NodeEnvironment.NodePath nodePath : env.nodePaths()) {
                totFreeSpace = totFreeSpace.add(BigInteger.valueOf(nodePath.fileStore.getUsableSpace()));
            }
            BigInteger estShardSizeInBytes = BigInteger.valueOf(avgShardSizeInBytes).max(totFreeSpace.divide(BigInteger.valueOf(20L)));
            NodeEnvironment.NodePath[] paths = env.nodePaths();
            NodeEnvironment.NodePath bestPath = null;
            BigInteger maxUsableBytes = BigInteger.valueOf(Long.MIN_VALUE);
            for (NodeEnvironment.NodePath nodePath : paths) {
                FileStore fileStore = nodePath.fileStore;
                BigInteger usableBytes = BigInteger.valueOf(fileStore.getUsableSpace());
                assert (usableBytes.compareTo(BigInteger.ZERO) >= 0);
                Integer count = dataPathToShardCount.get(nodePath.path);
                if (count != null) {
                    usableBytes = usableBytes.subtract(estShardSizeInBytes.multiply(BigInteger.valueOf(count.intValue())));
                }
                if (bestPath != null && usableBytes.compareTo(maxUsableBytes) <= 0) continue;
                maxUsableBytes = usableBytes;
                bestPath = nodePath;
            }
            dataPath = statePath = bestPath.resolve(shardId);
        }
        return new ShardPath(indexSettings.hasCustomDataPath(), dataPath, statePath, shardId);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ShardPath shardPath = (ShardPath)o;
        if (this.shardId != null ? !this.shardId.equals(shardPath.shardId) : shardPath.shardId != null) {
            return false;
        }
        return !(this.path != null ? !this.path.equals(shardPath.path) : shardPath.path != null);
    }

    public int hashCode() {
        int result = this.path != null ? this.path.hashCode() : 0;
        result = 31 * result + (this.shardId != null ? this.shardId.hashCode() : 0);
        return result;
    }

    public String toString() {
        return "ShardPath{path=" + this.path + ", shard=" + this.shardId + '}';
    }
}

