/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import org.apache.druid.common.config.NullHandling;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.math.expr.BinEqExpr;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprEval;
import org.apache.druid.math.expr.ExprType;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;

interface Function {
    public String name();

    public ExprEval apply(List<Expr> var1, Expr.ObjectBinding var2);

    public static class RpadFunc
    implements Function {
        @Override
        public String name() {
            return "rpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.rpad(base, len, pad));
        }
    }

    public static class LpadFunc
    implements Function {
        @Override
        public String name() {
            return "lpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.lpad(base, len, pad));
        }
    }

    public static class IsNotNullFunc
    implements Function {
        @Override
        public String name() {
            return "notnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.of(expr.value() != null, ExprType.LONG);
        }
    }

    public static class IsNullFunc
    implements Function {
        @Override
        public String name() {
            return "isnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.of(expr.value() == null, ExprType.LONG);
        }
    }

    public static class RepeatFunc
    extends DoubleParamString {
        @Override
        public String name() {
            return "repeat";
        }

        @Override
        protected ExprEval eval(String x, int y) {
            return ExprEval.of(y < 1 ? NullHandling.defaultStringValue() : StringUtils.repeat(x, y));
        }
    }

    public static class ReverseFunc
    extends SingleParam {
        @Override
        public String name() {
            return "reverse";
        }

        @Override
        protected ExprEval eval(ExprEval param) {
            if (param.type() != ExprType.STRING) {
                throw new IAE("Function[%s] needs a string argument", this.name());
            }
            String arg = param.asString();
            return ExprEval.of(arg == null ? NullHandling.defaultStringValue() : new StringBuilder(arg).reverse().toString());
        }
    }

    public static class UpperFunc
    implements Function {
        @Override
        public String name() {
            return "upper";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toUpperCase(arg));
        }
    }

    public static class LowerFunc
    implements Function {
        @Override
        public String name() {
            return "lower";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toLowerCase(arg));
        }
    }

    public static class ReplaceFunc
    implements Function {
        @Override
        public String name() {
            return "replace";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
            String arg = args.get(0).eval(bindings).asString();
            String pattern = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            String replacement = NullHandling.nullToEmptyIfNeeded(args.get(2).eval(bindings).asString());
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.replace(arg, pattern, replacement));
        }
    }

    public static class LeftFunc
    extends DoubleParamString {
        @Override
        public String name() {
            return "left";
        }

        @Override
        protected ExprEval eval(String x, int y) {
            if (y < 0) {
                throw new IAE("Function[%s] needs a postive integer as second argument", this.name());
            }
            return ExprEval.of(y < x.length() ? x.substring(0, y) : x);
        }
    }

    public static class RightFunc
    extends DoubleParamString {
        @Override
        public String name() {
            return "right";
        }

        @Override
        protected ExprEval eval(String x, int y) {
            if (y < 0) {
                throw new IAE("Function[%s] needs a postive integer as second argument", this.name());
            }
            int len = x.length();
            return ExprEval.of(y < len ? x.substring(len - y) : x);
        }
    }

    public static class SubstringFunc
    implements Function {
        @Override
        public String name() {
            return "substring";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(null);
            }
            int index = args.get(1).eval(bindings).asInt();
            int length = args.get(2).eval(bindings).asInt();
            if (index < arg.length()) {
                if (length >= 0) {
                    return ExprEval.of(arg.substring(index, Math.min(index + length, arg.length())));
                }
                return ExprEval.of(arg.substring(index));
            }
            return ExprEval.of(NullHandling.defaultStringValue());
        }
    }

    public static class StrposFunc
    implements Function {
        @Override
        public String name() {
            return "strpos";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() < 2 || args.size() > 3) {
                throw new IAE("Function[%s] needs 2 or 3 arguments", this.name());
            }
            String haystack = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            String needle = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            if (haystack == null || needle == null) {
                return ExprEval.of(null);
            }
            int fromIndex = args.size() >= 3 ? args.get(2).eval(bindings).asInt() : 0;
            return ExprEval.of(haystack.indexOf(needle, fromIndex));
        }
    }

    public static class StringFormatFunc
    implements Function {
        @Override
        public String name() {
            return "format";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() < 1) {
                throw new IAE("Function[%s] needs 1 or more arguments", this.name());
            }
            String formatString = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (formatString == null) {
                return ExprEval.of(null);
            }
            Object[] formatArgs = new Object[args.size() - 1];
            for (int i = 1; i < args.size(); ++i) {
                formatArgs[i - 1] = args.get(i).eval(bindings).value();
            }
            return ExprEval.of(StringUtils.nonStrictFormat(formatString, formatArgs));
        }
    }

    public static class StrlenFunc
    implements Function {
        @Override
        public String name() {
            return "strlen";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            String arg = args.get(0).eval(bindings).asString();
            return arg == null ? ExprEval.ofLong(NullHandling.defaultLongValue()) : ExprEval.of(arg.length());
        }
    }

    public static class ConcatFunc
    implements Function {
        @Override
        public String name() {
            return "concat";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() == 0) {
                return ExprEval.of(null);
            }
            String first = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (first == null) {
                return ExprEval.of(null);
            }
            StringBuilder builder = new StringBuilder(first);
            for (int i = 1; i < args.size(); ++i) {
                String s = NullHandling.nullToEmptyIfNeeded(args.get(i).eval(bindings).asString());
                if (s == null) {
                    return ExprEval.of(null);
                }
                builder.append(s);
            }
            return ExprEval.of(builder.toString());
        }
    }

    public static class NvlFunc
    implements Function {
        @Override
        public String name() {
            return "nvl";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
            ExprEval eval = args.get(0).eval(bindings);
            return eval.value() == null ? args.get(1).eval(bindings) : eval;
        }
    }

    public static class UnixTimestampFunc
    extends TimestampFromEpochFunc {
        @Override
        public String name() {
            return "unix_timestamp";
        }

        @Override
        protected final ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis() / 1000L);
        }
    }

    public static class TimestampFromEpochFunc
    implements Function {
        @Override
        public String name() {
            return "timestamp";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            DateTime date;
            if (args.size() != 1 && args.size() != 2) {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
            ExprEval value = args.get(0).eval(bindings);
            if (value.type() != ExprType.STRING) {
                throw new IAE("first argument should be string type but got %s type", new Object[]{value.type()});
            }
            DateTimes.UtcFormatter formatter = DateTimes.ISO_DATE_OPTIONAL_TIME;
            if (args.size() > 1) {
                ExprEval format = args.get(1).eval(bindings);
                if (format.type() != ExprType.STRING) {
                    throw new IAE("second argument should be string type but got %s type", new Object[]{format.type()});
                }
                formatter = DateTimes.wrapFormatter(DateTimeFormat.forPattern((String)format.asString()));
            }
            try {
                date = formatter.parse(value.asString());
            }
            catch (IllegalArgumentException e) {
                throw new IAE(e, "invalid value %s", value.asString());
            }
            return this.toValue(date);
        }

        protected ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis());
        }
    }

    public static class CastFunc
    extends DoubleParam {
        @Override
        public String name() {
            return "cast";
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            ExprType castTo;
            if (NullHandling.sqlCompatible() && x.value() == null) {
                return ExprEval.of(null);
            }
            try {
                castTo = ExprType.valueOf(StringUtils.toUpperCase(y.asString()));
            }
            catch (IllegalArgumentException e) {
                throw new IAE("invalid type '%s'", y.asString());
            }
            return x.castTo(castTo);
        }
    }

    public static class CaseSimpleFunc
    implements Function {
        @Override
        public String name() {
            return "case_simple";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() < 3) {
                throw new IAE("Function[%s] must have at least 3 arguments", this.name());
            }
            for (int i = 1; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!new BinEqExpr("==", args.get(0), args.get(i)).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }
    }

    public static class CaseSearchedFunc
    implements Function {
        @Override
        public String name() {
            return "case_searched";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() < 2) {
                throw new IAE("Function[%s] must have at least 2 arguments", this.name());
            }
            for (int i = 0; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!args.get(i).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }
    }

    public static class ConditionFunc
    implements Function {
        @Override
        public String name() {
            return "if";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
            ExprEval x = args.get(0).eval(bindings);
            return x.asBoolean() ? args.get(1).eval(bindings) : args.get(2).eval(bindings);
        }
    }

    public static class Scalb
    extends DoubleParam {
        @Override
        public String name() {
            return "scalb";
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            return ExprEval.of(Math.scalb(x.asDouble(), y.asInt()));
        }
    }

    public static class Pow
    extends DoubleParamMath {
        @Override
        public String name() {
            return "pow";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.pow(x, y));
        }
    }

    public static class NextAfter
    extends DoubleParamMath {
        @Override
        public String name() {
            return "nextAfter";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.nextAfter(x, y));
        }
    }

    public static class Min
    extends DoubleParamMath {
        @Override
        public String name() {
            return "min";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.min(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.min(x, y));
        }
    }

    public static class Max
    extends DoubleParamMath {
        @Override
        public String name() {
            return "max";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.max(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.max(x, y));
        }
    }

    public static class Remainder
    extends DoubleParamMath {
        @Override
        public String name() {
            return "remainder";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.IEEEremainder(x, y));
        }
    }

    public static class Hypot
    extends DoubleParamMath {
        @Override
        public String name() {
            return "hypot";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.hypot(x, y));
        }
    }

    public static class CopySign
    extends DoubleParamMath {
        @Override
        public String name() {
            return "copySign";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.copySign(x, y));
        }
    }

    public static class Atan2
    extends DoubleParamMath {
        @Override
        public String name() {
            return "atan2";
        }

        @Override
        protected ExprEval eval(double y, double x) {
            return ExprEval.of(Math.atan2(y, x));
        }
    }

    public static class Ulp
    extends SingleParamMath {
        @Override
        public String name() {
            return "ulp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ulp(param));
        }
    }

    public static class ToRadians
    extends SingleParamMath {
        @Override
        public String name() {
            return "toRadians";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toRadians(param));
        }
    }

    public static class ToDegrees
    extends SingleParamMath {
        @Override
        public String name() {
            return "toDegrees";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toDegrees(param));
        }
    }

    public static class Tanh
    extends SingleParamMath {
        @Override
        public String name() {
            return "tanh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tanh(param));
        }
    }

    public static class Tan
    extends SingleParamMath {
        @Override
        public String name() {
            return "tan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tan(param));
        }
    }

    public static class Sqrt
    extends SingleParamMath {
        @Override
        public String name() {
            return "sqrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sqrt(param));
        }
    }

    public static class Sinh
    extends SingleParamMath {
        @Override
        public String name() {
            return "sinh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sinh(param));
        }
    }

    public static class Sin
    extends SingleParamMath {
        @Override
        public String name() {
            return "sin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sin(param));
        }
    }

    public static class Signum
    extends SingleParamMath {
        @Override
        public String name() {
            return "signum";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.signum(param));
        }
    }

    public static class Round
    implements Function {
        @Override
        public String name() {
            return "round";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1 && args.size() != 2) {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
            ExprEval value1 = args.get(0).eval(bindings);
            if (value1.type() != ExprType.LONG && value1.type() != ExprType.DOUBLE) {
                throw new IAE("The first argument to the function[%s] should be integer or double type but get the %s type", new Object[]{this.name(), value1.type()});
            }
            if (args.size() == 1) {
                return this.eval(value1);
            }
            ExprEval value2 = args.get(1).eval(bindings);
            if (value2.type() != ExprType.LONG) {
                throw new IAE("The second argument to the function[%s] should be integer type but get the %s type", new Object[]{this.name(), value2.type()});
            }
            return this.eval(value1, value2.asInt());
        }

        private ExprEval eval(ExprEval param) {
            return this.eval(param, 0);
        }

        private ExprEval eval(ExprEval param, int scale) {
            if (param.type() == ExprType.LONG) {
                return ExprEval.of(BigDecimal.valueOf(param.asLong()).setScale(scale, RoundingMode.HALF_UP).longValue());
            }
            if (param.type() == ExprType.DOUBLE) {
                return ExprEval.of(BigDecimal.valueOf(param.asDouble()).setScale(scale, RoundingMode.HALF_UP).doubleValue());
            }
            return ExprEval.of(null);
        }
    }

    public static class Rint
    extends SingleParamMath {
        @Override
        public String name() {
            return "rint";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.rint(param));
        }
    }

    public static class NextUp
    extends SingleParamMath {
        @Override
        public String name() {
            return "nextUp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.nextUp(param));
        }
    }

    public static class Log1p
    extends SingleParamMath {
        @Override
        public String name() {
            return "log1p";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log1p(param));
        }
    }

    public static class Log10
    extends SingleParamMath {
        @Override
        public String name() {
            return "log10";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log10(param));
        }
    }

    public static class Log
    extends SingleParamMath {
        @Override
        public String name() {
            return "log";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log(param));
        }
    }

    public static class GetExponent
    extends SingleParamMath {
        @Override
        public String name() {
            return "getExponent";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.getExponent(param));
        }
    }

    public static class Floor
    extends SingleParamMath {
        @Override
        public String name() {
            return "floor";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.floor(param));
        }
    }

    public static class Expm1
    extends SingleParamMath {
        @Override
        public String name() {
            return "expm1";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.expm1(param));
        }
    }

    public static class Exp
    extends SingleParamMath {
        @Override
        public String name() {
            return "exp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.exp(param));
        }
    }

    public static class Div
    extends DoubleParamMath {
        @Override
        public String name() {
            return "div";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x / y);
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of((long)(x / y));
        }
    }

    public static class Cot
    extends SingleParamMath {
        @Override
        public String name() {
            return "cot";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param) / Math.sin(param));
        }
    }

    public static class Cosh
    extends SingleParamMath {
        @Override
        public String name() {
            return "cosh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cosh(param));
        }
    }

    public static class Cos
    extends SingleParamMath {
        @Override
        public String name() {
            return "cos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param));
        }
    }

    public static class Ceil
    extends SingleParamMath {
        @Override
        public String name() {
            return "ceil";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ceil(param));
        }
    }

    public static class Cbrt
    extends SingleParamMath {
        @Override
        public String name() {
            return "cbrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cbrt(param));
        }
    }

    public static class Atan
    extends SingleParamMath {
        @Override
        public String name() {
            return "atan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.atan(param));
        }
    }

    public static class Asin
    extends SingleParamMath {
        @Override
        public String name() {
            return "asin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.asin(param));
        }
    }

    public static class Acos
    extends SingleParamMath {
        @Override
        public String name() {
            return "acos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.acos(param));
        }
    }

    public static class Abs
    extends SingleParamMath {
        @Override
        public String name() {
            return "abs";
        }

        @Override
        protected ExprEval eval(long param) {
            return ExprEval.of(Math.abs(param));
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.abs(param));
        }
    }

    public static class Pi
    implements Function {
        private static final double PI = Math.PI;

        @Override
        public String name() {
            return "pi";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() >= 1) {
                throw new IAE("Function[%s] needs 0 argument", this.name());
            }
            return ExprEval.of(Math.PI);
        }
    }

    public static class ParseLong
    implements Function {
        @Override
        public String name() {
            return "parse_long";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            long retVal;
            int radix;
            if (args.size() == 1) {
                radix = 10;
            } else if (args.size() == 2) {
                radix = args.get(1).eval(bindings).asInt();
            } else {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
            String input = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (input == null) {
                return ExprEval.ofLong(null);
            }
            try {
                retVal = radix == 16 && (input.startsWith("0x") || input.startsWith("0X")) ? Long.parseLong(input.substring(2), radix) : Long.parseLong(input, radix);
            }
            catch (NumberFormatException e) {
                return ExprEval.ofLong(null);
            }
            return ExprEval.of(retVal);
        }
    }

    public static abstract class DoubleParamString
    extends DoubleParam {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (x.type() != ExprType.STRING || y.type() != ExprType.LONG) {
                throw new IAE("Function[%s] needs a string as first argument and an integer as second argument", this.name());
            }
            return this.eval(x.asString(), y.asInt());
        }

        protected abstract ExprEval eval(String var1, int var2);
    }

    public static abstract class DoubleParamMath
    extends DoubleParam {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (x.type() == ExprType.STRING || y.type() == ExprType.STRING) {
                return ExprEval.of(null);
            }
            if (x.type() == ExprType.LONG && y.type() == ExprType.LONG) {
                return this.eval(x.asLong(), y.asLong());
            }
            return this.eval(x.asDouble(), y.asDouble());
        }

        protected ExprEval eval(long x, long y) {
            return this.eval((double)x, (double)y);
        }

        protected ExprEval eval(double x, double y) {
            return this.eval((long)x, (long)y);
        }
    }

    public static abstract class SingleParamMath
    extends SingleParam {
        @Override
        protected final ExprEval eval(ExprEval param) {
            if (NullHandling.sqlCompatible() && param.isNumericNull()) {
                return ExprEval.of(null);
            }
            if (param.type() == ExprType.LONG) {
                return this.eval(param.asLong());
            }
            if (param.type() == ExprType.DOUBLE) {
                return this.eval(param.asDouble());
            }
            return ExprEval.of(null);
        }

        protected ExprEval eval(long param) {
            return this.eval((double)param);
        }

        protected ExprEval eval(double param) {
            return this.eval((long)param);
        }
    }

    public static abstract class DoubleParam
    implements Function {
        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
            Expr expr1 = args.get(0);
            Expr expr2 = args.get(1);
            return this.eval(expr1.eval(bindings), expr2.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1, ExprEval var2);
    }

    public static abstract class SingleParam
    implements Function {
        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            Expr expr = args.get(0);
            return this.eval(expr.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1);
    }
}

