/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.vector.complex.reader;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;


/*
 * This class is generated using freemarker and the BaseReader.java template.
 */
public interface BaseReader extends Positionable {
  MajorType getType();
  MaterializedField getField();
  void reset();
  void read(UnionHolder holder);
  void read(int index, UnionHolder holder);
  void copyAsValue(UnionWriter writer);
  boolean isSet();
  void read(ValueHolder holder);

  /**
   * Returns {@code String} representation of the reader's type. In case if
   * {@link #getType()} is primitive, the method is equivalent to
   * {@link #getType().getMinorType().name()}. If the reader has minor type
   * equal to {@link org.apache.drill.common.types.TypeProtos.MinorType#DICT},
   * {@code DICT&lt;keyMinorType,valueMinorType&gt;}, with {@code keyMinorType}
   * and {@code valueMinorType} being key's and value's minor types
   * respectively, will be returned. Used in {@code typeOf} UDF.
   *
   * @return {@code String} representation of reader's type.
   */
  String getTypeString();

  /**
   * Returns the type of the vector, not value. For all vectors, this is
   * the same as the vector's <tt>getField().getType().getMinorType()</tt>.
   * It is used to report the actual vector type in the getDrillType()
   * function.
   */
  MinorType getVectorType();

  public interface MapReader extends BaseReader, Iterable<String>{
    FieldReader reader(String name);
  }

  public interface RepeatedMapReader extends MapReader{
    boolean next();
    int size();
    void copyAsValue(MapWriter writer);
  }

  public interface DictReader extends RepeatedMapReader {
    void copyAsValue(DictWriter writer);

    /**
     * Obtain the index for given key in current row used to find a corresponding value with.
     * Used in generated code when retrieving value from Dict with
     * {@link org.apache.drill.common.expression.PathSegment.NameSegment}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is complex.
     *
     * <p>Despite {@code key} is passed as {@code String} the value is converted to
     * actual type based on {@link org.apache.drill.exec.vector.complex.DictVector#getKeyType()}.
     *
     * @param key literal representing key value
     * @return index for the given key
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    int find(String key);

    /**
     * Obtain the index for given key in current row used to find a corresponding value with.
     * Used in generated code when retrieving value from Dict with
     * {@link org.apache.drill.common.expression.PathSegment.ArraySegment}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is complex.
     *
     * <p>Despite {@code key} is passed as {@code int} the value is converted to
     * actual type based on {@link org.apache.drill.exec.vector.complex.DictVector#getKeyType()}.
     *
     * @param key literal representing key value
     * @return index for the given key
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    int find(int key);

    /**
     * Obtain the index for given key in current row used to find a corresponding value with.
     * Used in generated code when retrieving value from Dict using {@link org.apache.drill.common.expression.PathSegment}
     * with provided {@link org.apache.drill.common.expression.PathSegment#getOriginalValue()}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is complex.
     *
     * <p>The {@code key} is assumed to be of actual type, is not converted and used as is.
     *
     * @param key key value
     * @return index for the given key
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    int find(Object key);

    /**
     * Reads a value corresponding to a {@code key} into the {@code holder}.
     * If there is no entry in the row with the given {@code key}, value is set to null.
     *
     * <p>Used in generated code when retrieving value from Dict with
     * {@link org.apache.drill.common.expression.PathSegment.NameSegment}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is primitive.
     *
     * <p>Despite {@code key} is passed as {@code String} the value is converted to
     * actual type based on {@link org.apache.drill.exec.vector.complex.DictVector#getKeyType()}.
     *
     * @param key literal representing key value
     * @param holder a holder to write value's value into
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    void read(String key, ValueHolder holder);

    /**
     * Reads a value corresponding to a {@code key} into the {@code holder}.
     * If there is no entry in the row with the given {@code key}, value is set to null.
     *
     * <p>Used in generated code when retrieving value from Dict with
     * {@link org.apache.drill.common.expression.PathSegment.ArraySegment}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is primitive.
     *
     * <p>Despite {@code key} is passed as {@code int} the value is converted to
     * actual type based on {@link org.apache.drill.exec.vector.complex.DictVector#getKeyType()}.
     *
     * @param key literal representing key value
     * @param holder a holder to write value's value into
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    void read(int key, ValueHolder holder);

    /**
     * Reads a value corresponding to a {@code key} into the {@code holder}.
     * If there is no entry in the row with the given {@code key}, value is set to null.
     *
     * <p>Used in generated code when retrieving value from Dict using {@link org.apache.drill.common.expression.PathSegment}
     * with provided {@link org.apache.drill.common.expression.PathSegment#getOriginalValue()}
     * in cases when {@link org.apache.drill.exec.vector.complex.DictVector#getValueType()} is primitive.
     *
     * <p>The {@code key} is assumed to be of actual type, is not converted and used as is.
     *
     * @param key key value
     * @param holder a holder to write value's value into
     * @see org.apache.drill.exec.vector.complex.DictVector
     */
    void read(Object key, ValueHolder holder);
  }

  public interface ListReader extends BaseReader{
    FieldReader reader();
  }

  public interface RepeatedListReader extends ListReader{
    boolean next();
    int size();
    void copyAsValue(ListWriter writer);
  }

  public interface ScalarReader extends
   TinyIntReader,  UInt1Reader,  UInt2Reader,  SmallIntReader,  IntReader,  UInt4Reader,  Float4Reader,  TimeReader,  IntervalYearReader,  Decimal9Reader,  BigIntReader,  UInt8Reader,  Float8Reader,  DateReader,  TimeStampReader,  Decimal18Reader,  IntervalDayReader,  IntervalReader,  Decimal28DenseReader,  Decimal38DenseReader,  Decimal38SparseReader,  Decimal28SparseReader,  VarBinaryReader,  VarCharReader,  Var16CharReader,  VarDecimalReader,  BitReader, 
   RepeatedTinyIntReader,  RepeatedUInt1Reader,  RepeatedUInt2Reader,  RepeatedSmallIntReader,  RepeatedIntReader,  RepeatedUInt4Reader,  RepeatedFloat4Reader,  RepeatedTimeReader,  RepeatedIntervalYearReader,  RepeatedDecimal9Reader,  RepeatedBigIntReader,  RepeatedUInt8Reader,  RepeatedFloat8Reader,  RepeatedDateReader,  RepeatedTimeStampReader,  RepeatedDecimal18Reader,  RepeatedIntervalDayReader,  RepeatedIntervalReader,  RepeatedDecimal28DenseReader,  RepeatedDecimal38DenseReader,  RepeatedDecimal38SparseReader,  RepeatedDecimal28SparseReader,  RepeatedVarBinaryReader,  RepeatedVarCharReader,  RepeatedVar16CharReader,  RepeatedVarDecimalReader,  RepeatedBitReader, 
  BaseReader {}

  interface ComplexReader{
    MapReader rootAsMap();
    ListReader rootAsList();
    boolean rootIsMap();
    boolean ok();
  }
}

