/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.drill.exec.vector.complex.impl;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;

import java.util.EnumMap;
import java.util.Map;
import java.util.function.Function;

/**
 * ListWriter-like writer with only difference that it acts only as a factory
 * for concrete type writers for UnionVector data vector.
 */
public class UnionVectorWriter extends AbstractFieldWriter {

  /**
   * Map holding lazily initialized type specific writers for the union data vector
   */
  final Map<MinorType, FieldWriter> typeWriters = new TypeWritersMap();

  /**
   * Data vector here used as a producer for type specific vectors
   * which will be used by type writers for storing concrete values.
   */
  final UnionVector dataVector;

  /**
  * Constructs writer with dataVector of UnionVector type.
  *
  * @param vector union data vector
  * @param parent parent writer
  */
  public UnionVectorWriter(UnionVector vector, FieldWriter parent) {
    super(parent);
    dataVector = vector;
  }

  // FACTORIES FOR COMPLEX TYPE WRITERS

  @Override
  public UnionVectorWriter union() {
    return this;
  }

  @Override
  public MapWriter map() {
    return typeWriters.computeIfAbsent(MinorType.MAP, type -> new SingleMapUnionWriter(dataVector.getMap(), null, false));
  }

  @Override
  public DictWriter dict() {
    return typeWriters.computeIfAbsent(MinorType.DICT, type -> new SingleDictUnionWriter(dataVector.getDict(), null, false));
  }

  @Override
  public ListWriter list() {
    return typeWriters.computeIfAbsent(MinorType.LIST, listType -> new ListUnionWriter(dataVector.getList()));
  }

  // FACTORIES FOR PRIMITIVE TYPE WRITERS

  /**
   * Get concrete writer for writing tinyint data to {@link #dataVector}.
   *
   * @return tinyint writer
   */
  @Override
  public TinyIntWriter tinyInt() {
    return typeWriters.computeIfAbsent(MinorType.TINYINT, TinyIntUnionWriter::new);
  }

  /**
   * Get concrete writer for writing uint1 data to {@link #dataVector}.
   *
   * @return uint1 writer
   */
  @Override
  public UInt1Writer uInt1() {
    return typeWriters.computeIfAbsent(MinorType.UINT1, UInt1UnionWriter::new);
  }

  /**
   * Get concrete writer for writing uint2 data to {@link #dataVector}.
   *
   * @return uint2 writer
   */
  @Override
  public UInt2Writer uInt2() {
    return typeWriters.computeIfAbsent(MinorType.UINT2, UInt2UnionWriter::new);
  }

  /**
   * Get concrete writer for writing smallint data to {@link #dataVector}.
   *
   * @return smallint writer
   */
  @Override
  public SmallIntWriter smallInt() {
    return typeWriters.computeIfAbsent(MinorType.SMALLINT, SmallIntUnionWriter::new);
  }

  /**
   * Get concrete writer for writing int data to {@link #dataVector}.
   *
   * @return int writer
   */
  @Override
  public IntWriter integer() {
    return typeWriters.computeIfAbsent(MinorType.INT, IntUnionWriter::new);
  }

  /**
   * Get concrete writer for writing uint4 data to {@link #dataVector}.
   *
   * @return uint4 writer
   */
  @Override
  public UInt4Writer uInt4() {
    return typeWriters.computeIfAbsent(MinorType.UINT4, UInt4UnionWriter::new);
  }

  /**
   * Get concrete writer for writing float4 data to {@link #dataVector}.
   *
   * @return float4 writer
   */
  @Override
  public Float4Writer float4() {
    return typeWriters.computeIfAbsent(MinorType.FLOAT4, Float4UnionWriter::new);
  }

  /**
   * Get concrete writer for writing time data to {@link #dataVector}.
   *
   * @return time writer
   */
  @Override
  public TimeWriter time() {
    return typeWriters.computeIfAbsent(MinorType.TIME, TimeUnionWriter::new);
  }

  /**
   * Get concrete writer for writing intervalyear data to {@link #dataVector}.
   *
   * @return intervalyear writer
   */
  @Override
  public IntervalYearWriter intervalYear() {
    return typeWriters.computeIfAbsent(MinorType.INTERVALYEAR, IntervalYearUnionWriter::new);
  }

  /**
   * Get concrete writer for writing bigint data to {@link #dataVector}.
   *
   * @return bigint writer
   */
  @Override
  public BigIntWriter bigInt() {
    return typeWriters.computeIfAbsent(MinorType.BIGINT, BigIntUnionWriter::new);
  }

  /**
   * Get concrete writer for writing uint8 data to {@link #dataVector}.
   *
   * @return uint8 writer
   */
  @Override
  public UInt8Writer uInt8() {
    return typeWriters.computeIfAbsent(MinorType.UINT8, UInt8UnionWriter::new);
  }

  /**
   * Get concrete writer for writing float8 data to {@link #dataVector}.
   *
   * @return float8 writer
   */
  @Override
  public Float8Writer float8() {
    return typeWriters.computeIfAbsent(MinorType.FLOAT8, Float8UnionWriter::new);
  }

  /**
   * Get concrete writer for writing date data to {@link #dataVector}.
   *
   * @return date writer
   */
  @Override
  public DateWriter date() {
    return typeWriters.computeIfAbsent(MinorType.DATE, DateUnionWriter::new);
  }

  /**
   * Get concrete writer for writing timestamp data to {@link #dataVector}.
   *
   * @return timestamp writer
   */
  @Override
  public TimeStampWriter timeStamp() {
    return typeWriters.computeIfAbsent(MinorType.TIMESTAMP, TimeStampUnionWriter::new);
  }

  /**
   * Get concrete writer for writing intervalday data to {@link #dataVector}.
   *
   * @return intervalday writer
   */
  @Override
  public IntervalDayWriter intervalDay() {
    return typeWriters.computeIfAbsent(MinorType.INTERVALDAY, IntervalDayUnionWriter::new);
  }

  /**
   * Get concrete writer for writing interval data to {@link #dataVector}.
   *
   * @return interval writer
   */
  @Override
  public IntervalWriter interval() {
    return typeWriters.computeIfAbsent(MinorType.INTERVAL, IntervalUnionWriter::new);
  }

  /**
   * Get concrete writer for writing varbinary data to {@link #dataVector}.
   *
   * @return varbinary writer
   */
  @Override
  public VarBinaryWriter varBinary() {
    return typeWriters.computeIfAbsent(MinorType.VARBINARY, VarBinaryUnionWriter::new);
  }

  /**
   * Get concrete writer for writing varchar data to {@link #dataVector}.
   *
   * @return varchar writer
   */
  @Override
  public VarCharWriter varChar() {
    return typeWriters.computeIfAbsent(MinorType.VARCHAR, VarCharUnionWriter::new);
  }

  /**
   * Get concrete writer for writing var16char data to {@link #dataVector}.
   *
   * @return var16char writer
   */
  @Override
  public Var16CharWriter var16Char() {
    return typeWriters.computeIfAbsent(MinorType.VAR16CHAR, Var16CharUnionWriter::new);
  }

  /**
   * Get concrete writer for writing vardecimal data to {@link #dataVector}.
   *
   * @return vardecimal writer
   */
  @Override
  public VarDecimalWriter varDecimal() {
    return typeWriters.computeIfAbsent(MinorType.VARDECIMAL, VarDecimalUnionWriter::new);
  }

  @Override
  public VarDecimalWriter varDecimal(int precision, int scale) {
    return typeWriters.computeIfAbsent(MinorType.VARDECIMAL, type -> new VarDecimalUnionWriter(type, precision, scale));
  }

  /**
   * Get concrete writer for writing bit data to {@link #dataVector}.
   *
   * @return bit writer
   */
  @Override
  public BitWriter bit() {
    return typeWriters.computeIfAbsent(MinorType.BIT, BitUnionWriter::new);
  }

  // WRITER's METHODS
  @Override
  public void allocate() {
    dataVector.allocateNew();
  }

  @Override
  public void clear() {
    dataVector.clear();
  }

  @Override
  public int getValueCapacity() {
    return dataVector.getValueCapacity();
  }

  @Override
  public MaterializedField getField() {
    return dataVector.getField();
  }

  @Override
  public void close() throws Exception {
    dataVector.close();
  }

  @Override
  public void writeNull() {
    dataVector.getMutator().setNull(UnionVectorWriter.this.idx());
  }

  private void setTypeAndIndex(MinorType type, Positionable positionable) {
    dataVector.getMutator().setType(UnionVectorWriter.this.idx(), type);
    positionable.setPosition(UnionVectorWriter.this.idx());
  }

// TYPE SPECIFIC INNER WRITERS

  class TinyIntUnionWriter extends NullableTinyIntWriterImpl {
    private final MinorType type;

    TinyIntUnionWriter(MinorType type) {
      super(dataVector.getTinyIntVector(), null);
      this.type = type;
    }

    @Override
    public void writeTinyInt(byte value) {
      setTypeAndIndex(type, this);
      super.writeTinyInt(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(TinyIntHolder holder) {
      writeTinyInt(holder.value);
    }
  }

  class UInt1UnionWriter extends NullableUInt1WriterImpl {
    private final MinorType type;

    UInt1UnionWriter(MinorType type) {
      super(dataVector.getUInt1Vector(), null);
      this.type = type;
    }

    @Override
    public void writeUInt1(byte value) {
      setTypeAndIndex(type, this);
      super.writeUInt1(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(UInt1Holder holder) {
      writeUInt1(holder.value);
    }
  }

  class UInt2UnionWriter extends NullableUInt2WriterImpl {
    private final MinorType type;

    UInt2UnionWriter(MinorType type) {
      super(dataVector.getUInt2Vector(), null);
      this.type = type;
    }

    @Override
    public void writeUInt2(char value) {
      setTypeAndIndex(type, this);
      super.writeUInt2(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(UInt2Holder holder) {
      writeUInt2(holder.value);
    }
  }

  class SmallIntUnionWriter extends NullableSmallIntWriterImpl {
    private final MinorType type;

    SmallIntUnionWriter(MinorType type) {
      super(dataVector.getSmallIntVector(), null);
      this.type = type;
    }

    @Override
    public void writeSmallInt(short value) {
      setTypeAndIndex(type, this);
      super.writeSmallInt(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(SmallIntHolder holder) {
      writeSmallInt(holder.value);
    }
  }

  class IntUnionWriter extends NullableIntWriterImpl {
    private final MinorType type;

    IntUnionWriter(MinorType type) {
      super(dataVector.getIntVector(), null);
      this.type = type;
    }

    @Override
    public void writeInt(int value) {
      setTypeAndIndex(type, this);
      super.writeInt(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(IntHolder holder) {
      writeInt(holder.value);
    }
  }

  class UInt4UnionWriter extends NullableUInt4WriterImpl {
    private final MinorType type;

    UInt4UnionWriter(MinorType type) {
      super(dataVector.getUInt4Vector(), null);
      this.type = type;
    }

    @Override
    public void writeUInt4(int value) {
      setTypeAndIndex(type, this);
      super.writeUInt4(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(UInt4Holder holder) {
      writeUInt4(holder.value);
    }
  }

  class Float4UnionWriter extends NullableFloat4WriterImpl {
    private final MinorType type;

    Float4UnionWriter(MinorType type) {
      super(dataVector.getFloat4Vector(), null);
      this.type = type;
    }

    @Override
    public void writeFloat4(float value) {
      setTypeAndIndex(type, this);
      super.writeFloat4(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(Float4Holder holder) {
      writeFloat4(holder.value);
    }
  }

  class TimeUnionWriter extends NullableTimeWriterImpl {
    private final MinorType type;

    TimeUnionWriter(MinorType type) {
      super(dataVector.getTimeVector(), null);
      this.type = type;
    }

    @Override
    public void writeTime(int value) {
      setTypeAndIndex(type, this);
      super.writeTime(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(TimeHolder holder) {
      writeTime(holder.value);
    }
  }

  class IntervalYearUnionWriter extends NullableIntervalYearWriterImpl {
    private final MinorType type;

    IntervalYearUnionWriter(MinorType type) {
      super(dataVector.getIntervalYearVector(), null);
      this.type = type;
    }

    @Override
    public void writeIntervalYear(int value) {
      setTypeAndIndex(type, this);
      super.writeIntervalYear(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(IntervalYearHolder holder) {
      writeIntervalYear(holder.value);
    }
  }

  class BigIntUnionWriter extends NullableBigIntWriterImpl {
    private final MinorType type;

    BigIntUnionWriter(MinorType type) {
      super(dataVector.getBigIntVector(), null);
      this.type = type;
    }

    @Override
    public void writeBigInt(long value) {
      setTypeAndIndex(type, this);
      super.writeBigInt(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(BigIntHolder holder) {
      writeBigInt(holder.value);
    }
  }

  class UInt8UnionWriter extends NullableUInt8WriterImpl {
    private final MinorType type;

    UInt8UnionWriter(MinorType type) {
      super(dataVector.getUInt8Vector(), null);
      this.type = type;
    }

    @Override
    public void writeUInt8(long value) {
      setTypeAndIndex(type, this);
      super.writeUInt8(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(UInt8Holder holder) {
      writeUInt8(holder.value);
    }
  }

  class Float8UnionWriter extends NullableFloat8WriterImpl {
    private final MinorType type;

    Float8UnionWriter(MinorType type) {
      super(dataVector.getFloat8Vector(), null);
      this.type = type;
    }

    @Override
    public void writeFloat8(double value) {
      setTypeAndIndex(type, this);
      super.writeFloat8(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(Float8Holder holder) {
      writeFloat8(holder.value);
    }
  }

  class DateUnionWriter extends NullableDateWriterImpl {
    private final MinorType type;

    DateUnionWriter(MinorType type) {
      super(dataVector.getDateVector(), null);
      this.type = type;
    }

    @Override
    public void writeDate(long value) {
      setTypeAndIndex(type, this);
      super.writeDate(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(DateHolder holder) {
      writeDate(holder.value);
    }
  }

  class TimeStampUnionWriter extends NullableTimeStampWriterImpl {
    private final MinorType type;

    TimeStampUnionWriter(MinorType type) {
      super(dataVector.getTimeStampVector(), null);
      this.type = type;
    }

    @Override
    public void writeTimeStamp(long value) {
      setTypeAndIndex(type, this);
      super.writeTimeStamp(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(TimeStampHolder holder) {
      writeTimeStamp(holder.value);
    }
  }

  class IntervalDayUnionWriter extends NullableIntervalDayWriterImpl {
    private final MinorType type;

    IntervalDayUnionWriter(MinorType type) {
      super(dataVector.getIntervalDayVector(), null);
      this.type = type;
    }

    @Override
    public void writeIntervalDay(int days, int milliseconds) {
      setTypeAndIndex(type, this);
      super.writeIntervalDay(days, milliseconds);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(IntervalDayHolder holder) {
      writeIntervalDay(holder.days, holder.milliseconds);
    }
  }

  class IntervalUnionWriter extends NullableIntervalWriterImpl {
    private final MinorType type;

    IntervalUnionWriter(MinorType type) {
      super(dataVector.getIntervalVector(), null);
      this.type = type;
    }

    @Override
    public void writeInterval(int months, int days, int milliseconds) {
      setTypeAndIndex(type, this);
      super.writeInterval(months, days, milliseconds);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(IntervalHolder holder) {
      writeInterval(holder.months, holder.days, holder.milliseconds);
    }
  }

  class VarBinaryUnionWriter extends NullableVarBinaryWriterImpl {
    private final MinorType type;

    VarBinaryUnionWriter(MinorType type) {
      super(dataVector.getVarBinaryVector(), null);
      this.type = type;
    }

    @Override
    public void writeVarBinary(int start, int end, DrillBuf buffer) {
      setTypeAndIndex(type, this);
      super.writeVarBinary(start, end, buffer);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(VarBinaryHolder holder) {
      writeVarBinary(holder.start, holder.end, holder.buffer);
    }
  }

  class VarCharUnionWriter extends NullableVarCharWriterImpl {
    private final MinorType type;

    VarCharUnionWriter(MinorType type) {
      super(dataVector.getVarCharVector(), null);
      this.type = type;
    }

    @Override
    public void writeVarChar(int start, int end, DrillBuf buffer) {
      setTypeAndIndex(type, this);
      super.writeVarChar(start, end, buffer);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(VarCharHolder holder) {
      writeVarChar(holder.start, holder.end, holder.buffer);
    }
  }

  class Var16CharUnionWriter extends NullableVar16CharWriterImpl {
    private final MinorType type;

    Var16CharUnionWriter(MinorType type) {
      super(dataVector.getVar16CharVector(), null);
      this.type = type;
    }

    @Override
    public void writeVar16Char(int start, int end, DrillBuf buffer) {
      setTypeAndIndex(type, this);
      super.writeVar16Char(start, end, buffer);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(Var16CharHolder holder) {
      writeVar16Char(holder.start, holder.end, holder.buffer);
    }
  }

  class VarDecimalUnionWriter extends NullableVarDecimalWriterImpl {
    private final MinorType type;

    VarDecimalUnionWriter(MinorType type) {
      super(dataVector.getVarDecimalVector(), null);
      this.type = type;
    }

    VarDecimalUnionWriter(MinorType type, int precision, int scale) {
      this(type);
      MaterializedField field = super.vector.getField();
      MajorType typeWithPrecisionAndScale = field.getType().toBuilder()
          .setPrecision(precision).setScale(scale).build();
      field.replaceType(typeWithPrecisionAndScale);
    }

    @Override
    public void writeVarDecimal(BigDecimal value) {
      setTypeAndIndex(type, this);
      super.writeVarDecimal(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void writeVarDecimal(int start, int end, DrillBuf buffer, int precision, int scale) {
      setTypeAndIndex(type, this);
      super.writeVarDecimal(start, end, buffer, precision, scale);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(VarDecimalHolder holder) {
      writeVarDecimal(holder.start, holder.end, holder.buffer, holder.precision, holder.scale);
    }
  }

  class BitUnionWriter extends NullableBitWriterImpl {
    private final MinorType type;

    BitUnionWriter(MinorType type) {
      super(dataVector.getBitVector(), null);
      this.type = type;
    }

    @Override
    public void writeBit(int value) {
      setTypeAndIndex(type, this);
      super.writeBit(value);
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    @Override
    public void write(BitHolder holder) {
      writeBit(holder.value);
    }
  }

  class ListUnionWriter extends UnionListWriter {

    ListUnionWriter(ListVector vector) {
      super(vector);
    }

    @Override
    public void startList() {
      dataVector.getMutator().setType(UnionVectorWriter.this.idx(), MinorType.LIST);
      super.startList();
      dataVector.getMutator().setValueCount(idx() + 1);
    }

    /*
     Overridden methods here are used to initialize early $data$ field to avoid schema change exception
     when transfer pair called to transfer from empty list to list with initialized $data$ vector.
     For example, without the fix exception was thrown on attempt to transfer
        FROM: [`list` (LIST:OPTIONAL), children=([`[DEFAULT]` (LATE:OPTIONAL)])]
        TO:   [`list` (LIST:OPTIONAL), children=([`[DEFAULT]` (LATE:OPTIONAL)], [`$data$` (VARCHAR:OPTIONAL)])]
     */

    @Override
    public TinyIntWriter tinyInt() {
      writer.getWriter(MinorType.TINYINT);
      return super.tinyInt();
    }

    @Override
    public UInt1Writer uInt1() {
      writer.getWriter(MinorType.UINT1);
      return super.uInt1();
    }

    @Override
    public UInt2Writer uInt2() {
      writer.getWriter(MinorType.UINT2);
      return super.uInt2();
    }

    @Override
    public SmallIntWriter smallInt() {
      writer.getWriter(MinorType.SMALLINT);
      return super.smallInt();
    }

    @Override
    public IntWriter integer() {
      writer.getWriter(MinorType.INT);
      return super.integer();
    }

    @Override
    public UInt4Writer uInt4() {
      writer.getWriter(MinorType.UINT4);
      return super.uInt4();
    }

    @Override
    public Float4Writer float4() {
      writer.getWriter(MinorType.FLOAT4);
      return super.float4();
    }

    @Override
    public TimeWriter time() {
      writer.getWriter(MinorType.TIME);
      return super.time();
    }

    @Override
    public IntervalYearWriter intervalYear() {
      writer.getWriter(MinorType.INTERVALYEAR);
      return super.intervalYear();
    }

    @Override
    public BigIntWriter bigInt() {
      writer.getWriter(MinorType.BIGINT);
      return super.bigInt();
    }

    @Override
    public UInt8Writer uInt8() {
      writer.getWriter(MinorType.UINT8);
      return super.uInt8();
    }

    @Override
    public Float8Writer float8() {
      writer.getWriter(MinorType.FLOAT8);
      return super.float8();
    }

    @Override
    public DateWriter date() {
      writer.getWriter(MinorType.DATE);
      return super.date();
    }

    @Override
    public TimeStampWriter timeStamp() {
      writer.getWriter(MinorType.TIMESTAMP);
      return super.timeStamp();
    }

    @Override
    public IntervalDayWriter intervalDay() {
      writer.getWriter(MinorType.INTERVALDAY);
      return super.intervalDay();
    }

    @Override
    public IntervalWriter interval() {
      writer.getWriter(MinorType.INTERVAL);
      return super.interval();
    }

    @Override
    public VarBinaryWriter varBinary() {
      writer.getWriter(MinorType.VARBINARY);
      return super.varBinary();
    }

    @Override
    public VarCharWriter varChar() {
      writer.getWriter(MinorType.VARCHAR);
      return super.varChar();
    }

    @Override
    public Var16CharWriter var16Char() {
      writer.getWriter(MinorType.VAR16CHAR);
      return super.var16Char();
    }

    @Override
    public VarDecimalWriter varDecimal() {
      writer.getWriter(MinorType.VARDECIMAL);
      return super.varDecimal();
    }

    @Override
    public BitWriter bit() {
      writer.getWriter(MinorType.BIT);
      return super.bit();
    }
  }

  class SingleMapUnionWriter extends SingleMapWriter {

    SingleMapUnionWriter(MapVector container, FieldWriter parent, boolean unionEnabled) {
      super(container, parent, unionEnabled);
    }

    @Override
    public void start() {
      dataVector.getMutator().setType(UnionVectorWriter.this.idx(), MinorType.MAP);
      super.start();
      dataVector.getMutator().setValueCount(idx() + 1);
    }
  }

  class SingleDictUnionWriter extends SingleDictWriter {

    SingleDictUnionWriter(DictVector container, FieldWriter parent, boolean unionEnabled) {
      super(container, parent, unionEnabled);
    }

    @Override
    public void start() {
      dataVector.getMutator().setType(UnionVectorWriter.this.idx(), MinorType.DICT);
      super.start();
      dataVector.getMutator().setValueCount(idx() + 1);
    }
  }

  // CONTAINER FOR ALL TYPE-SPECIFIC WRITERS
  private class TypeWritersMap extends EnumMap<MinorType, FieldWriter> {
    TypeWritersMap() {
      super(MinorType.class);
    }

    @Override
    public FieldWriter computeIfAbsent(MinorType key, Function<? super MinorType, ? extends FieldWriter> mappingFunction) {
      FieldWriter fw = get(key);
      if (fw == null) {
        put(key, (fw = mappingFunction.apply(key)));
      }
      // fixes copying in MapUtility for case when column has type STRUCT<f:UNIONTYPE<...>>
      setTypeAndIndex(key, fw);
      return fw;
    }
  }
}
