/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.drill.exec.vector.complex.impl;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;

import java.util.EnumMap;
import java.util.Map;
import java.util.function.Function;

/**
 * Union vector writer for writing list of union-type values
 */
public class UnionVectorListWriter extends UnionVectorWriter {

  private final ListVector listVector;
  private final UInt4Vector offsets;
  private int listPosition;

  public UnionVectorListWriter(ListVector listVector, FieldWriter parent) {
    super(listVector.promoteToUnion(), parent);
    this.listVector = listVector;
    this.offsets = listVector.getOffsetVector();
  }

  // FACTORIES FOR COMPLEX LIST ELEMENT WRITERS

  public UnionVectorWriter union() {
    return this;
  }

  public MapWriter map() {
    return typeWriters.computeIfAbsent(MinorType.MAP, type -> new SingleMapUnionListElementWriter(dataVector.getMap(), null, false));
  }

  public DictWriter dict() {
    return typeWriters.computeIfAbsent(MinorType.DICT, type -> new SingleDictUnionListElementWriter(dataVector.getDict(), null, false));
  }

  public ListWriter list() {
    return typeWriters.computeIfAbsent(MinorType.LIST, type -> new UnionListUnionElementWriter(dataVector.getList()));
  }

  // FACTORIES FOR PRIMITIVE LIST ELEMENT WRITERS

  @Override
  public TinyIntWriter tinyInt() {
    return typeWriters.computeIfAbsent(MinorType.TINYINT, TinyIntUnionListElementWriter::new);
  }

  @Override
  public UInt1Writer uInt1() {
    return typeWriters.computeIfAbsent(MinorType.UINT1, UInt1UnionListElementWriter::new);
  }

  @Override
  public UInt2Writer uInt2() {
    return typeWriters.computeIfAbsent(MinorType.UINT2, UInt2UnionListElementWriter::new);
  }

  @Override
  public SmallIntWriter smallInt() {
    return typeWriters.computeIfAbsent(MinorType.SMALLINT, SmallIntUnionListElementWriter::new);
  }

  @Override
  public IntWriter integer() {
    return typeWriters.computeIfAbsent(MinorType.INT, IntUnionListElementWriter::new);
  }

  @Override
  public UInt4Writer uInt4() {
    return typeWriters.computeIfAbsent(MinorType.UINT4, UInt4UnionListElementWriter::new);
  }

  @Override
  public Float4Writer float4() {
    return typeWriters.computeIfAbsent(MinorType.FLOAT4, Float4UnionListElementWriter::new);
  }

  @Override
  public TimeWriter time() {
    return typeWriters.computeIfAbsent(MinorType.TIME, TimeUnionListElementWriter::new);
  }

  @Override
  public IntervalYearWriter intervalYear() {
    return typeWriters.computeIfAbsent(MinorType.INTERVALYEAR, IntervalYearUnionListElementWriter::new);
  }

  @Override
  public BigIntWriter bigInt() {
    return typeWriters.computeIfAbsent(MinorType.BIGINT, BigIntUnionListElementWriter::new);
  }

  @Override
  public UInt8Writer uInt8() {
    return typeWriters.computeIfAbsent(MinorType.UINT8, UInt8UnionListElementWriter::new);
  }

  @Override
  public Float8Writer float8() {
    return typeWriters.computeIfAbsent(MinorType.FLOAT8, Float8UnionListElementWriter::new);
  }

  @Override
  public DateWriter date() {
    return typeWriters.computeIfAbsent(MinorType.DATE, DateUnionListElementWriter::new);
  }

  @Override
  public TimeStampWriter timeStamp() {
    return typeWriters.computeIfAbsent(MinorType.TIMESTAMP, TimeStampUnionListElementWriter::new);
  }

  @Override
  public IntervalDayWriter intervalDay() {
    return typeWriters.computeIfAbsent(MinorType.INTERVALDAY, IntervalDayUnionListElementWriter::new);
  }

  @Override
  public IntervalWriter interval() {
    return typeWriters.computeIfAbsent(MinorType.INTERVAL, IntervalUnionListElementWriter::new);
  }

  @Override
  public VarBinaryWriter varBinary() {
    return typeWriters.computeIfAbsent(MinorType.VARBINARY, VarBinaryUnionListElementWriter::new);
  }

  @Override
  public VarCharWriter varChar() {
    return typeWriters.computeIfAbsent(MinorType.VARCHAR, VarCharUnionListElementWriter::new);
  }

  @Override
  public Var16CharWriter var16Char() {
    return typeWriters.computeIfAbsent(MinorType.VAR16CHAR, Var16CharUnionListElementWriter::new);
  }

  @Override
  public VarDecimalWriter varDecimal() {
    return typeWriters.computeIfAbsent(MinorType.VARDECIMAL, VarDecimalUnionListElementWriter::new);
  }

  @Override
  public VarDecimalWriter varDecimal(int precision, int scale) {
    return typeWriters.computeIfAbsent(MinorType.VARDECIMAL, type -> new VarDecimalUnionListElementWriter(type, precision, scale));
  }

  @Override
  public BitWriter bit() {
    return typeWriters.computeIfAbsent(MinorType.BIT, BitUnionListElementWriter::new);
  }

  // WRITER's METHODS

  /**
   * Superclass's idx() returns index of element inside list row. So the method uses
   * additional field {@link #listPosition} for storing index of list row for {@link #listVector}.
   *
   * @param index of list in list vector
   */
  @Override
  public void setPosition(int index) {
    this.listPosition = index;
    int dataPosition = offsets.getAccessor().get(listPosition);
    super.setPosition(dataPosition);
  }

  @Override
  public void allocate() {
    listVector.allocateNew();
  }

  @Override
  public void clear() {
    listVector.clear();
  }

  @Override
  public int getValueCapacity() {
    return listVector.getValueCapacity();
  }

  @Override
  public MaterializedField getField() {
    return listVector.getField();
  }

  @Override
  public void close() throws Exception {
    listVector.close();
  }

  private void setNextOffset() {
    final int nextOffset = offsets.getAccessor().get(listPosition + 1);
    listVector.getMutator().setNotNull(listPosition);
    super.setPosition(nextOffset);
  }

  private void increaseOffset() {
    offsets.getMutator().setSafe(listPosition + 1, idx() + 1);
  }

// TYPE SPECIFIC LIST ELEMENTS INNER WRITERS

  private class TinyIntUnionListElementWriter extends UnionVectorWriter.TinyIntUnionWriter {

    private TinyIntUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeTinyInt(byte value) {
      setNextOffset();
      super.writeTinyInt(value);
      increaseOffset();
    }
  }

  private class UInt1UnionListElementWriter extends UnionVectorWriter.UInt1UnionWriter {

    private UInt1UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeUInt1(byte value) {
      setNextOffset();
      super.writeUInt1(value);
      increaseOffset();
    }
  }

  private class UInt2UnionListElementWriter extends UnionVectorWriter.UInt2UnionWriter {

    private UInt2UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeUInt2(char value) {
      setNextOffset();
      super.writeUInt2(value);
      increaseOffset();
    }
  }

  private class SmallIntUnionListElementWriter extends UnionVectorWriter.SmallIntUnionWriter {

    private SmallIntUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeSmallInt(short value) {
      setNextOffset();
      super.writeSmallInt(value);
      increaseOffset();
    }
  }

  private class IntUnionListElementWriter extends UnionVectorWriter.IntUnionWriter {

    private IntUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeInt(int value) {
      setNextOffset();
      super.writeInt(value);
      increaseOffset();
    }
  }

  private class UInt4UnionListElementWriter extends UnionVectorWriter.UInt4UnionWriter {

    private UInt4UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeUInt4(int value) {
      setNextOffset();
      super.writeUInt4(value);
      increaseOffset();
    }
  }

  private class Float4UnionListElementWriter extends UnionVectorWriter.Float4UnionWriter {

    private Float4UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeFloat4(float value) {
      setNextOffset();
      super.writeFloat4(value);
      increaseOffset();
    }
  }

  private class TimeUnionListElementWriter extends UnionVectorWriter.TimeUnionWriter {

    private TimeUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeTime(int value) {
      setNextOffset();
      super.writeTime(value);
      increaseOffset();
    }
  }

  private class IntervalYearUnionListElementWriter extends UnionVectorWriter.IntervalYearUnionWriter {

    private IntervalYearUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeIntervalYear(int value) {
      setNextOffset();
      super.writeIntervalYear(value);
      increaseOffset();
    }
  }

  private class BigIntUnionListElementWriter extends UnionVectorWriter.BigIntUnionWriter {

    private BigIntUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeBigInt(long value) {
      setNextOffset();
      super.writeBigInt(value);
      increaseOffset();
    }
  }

  private class UInt8UnionListElementWriter extends UnionVectorWriter.UInt8UnionWriter {

    private UInt8UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeUInt8(long value) {
      setNextOffset();
      super.writeUInt8(value);
      increaseOffset();
    }
  }

  private class Float8UnionListElementWriter extends UnionVectorWriter.Float8UnionWriter {

    private Float8UnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeFloat8(double value) {
      setNextOffset();
      super.writeFloat8(value);
      increaseOffset();
    }
  }

  private class DateUnionListElementWriter extends UnionVectorWriter.DateUnionWriter {

    private DateUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeDate(long value) {
      setNextOffset();
      super.writeDate(value);
      increaseOffset();
    }
  }

  private class TimeStampUnionListElementWriter extends UnionVectorWriter.TimeStampUnionWriter {

    private TimeStampUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeTimeStamp(long value) {
      setNextOffset();
      super.writeTimeStamp(value);
      increaseOffset();
    }
  }

  private class IntervalDayUnionListElementWriter extends UnionVectorWriter.IntervalDayUnionWriter {

    private IntervalDayUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeIntervalDay(int days, int milliseconds) {
      setNextOffset();
      super.writeIntervalDay(days, milliseconds);
      increaseOffset();
    }
  }

  private class IntervalUnionListElementWriter extends UnionVectorWriter.IntervalUnionWriter {

    private IntervalUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeInterval(int months, int days, int milliseconds) {
      setNextOffset();
      super.writeInterval(months, days, milliseconds);
      increaseOffset();
    }
  }

  private class VarBinaryUnionListElementWriter extends UnionVectorWriter.VarBinaryUnionWriter {

    private VarBinaryUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeVarBinary(int start, int end, DrillBuf buffer) {
      setNextOffset();
      super.writeVarBinary(start, end, buffer);
      increaseOffset();
    }
  }

  private class VarCharUnionListElementWriter extends UnionVectorWriter.VarCharUnionWriter {

    private VarCharUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeVarChar(int start, int end, DrillBuf buffer) {
      setNextOffset();
      super.writeVarChar(start, end, buffer);
      increaseOffset();
    }
  }

  private class Var16CharUnionListElementWriter extends UnionVectorWriter.Var16CharUnionWriter {

    private Var16CharUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeVar16Char(int start, int end, DrillBuf buffer) {
      setNextOffset();
      super.writeVar16Char(start, end, buffer);
      increaseOffset();
    }
  }

  private class VarDecimalUnionListElementWriter extends UnionVectorWriter.VarDecimalUnionWriter {

    private VarDecimalUnionListElementWriter(MinorType type) {
      super(type);
    }

    private VarDecimalUnionListElementWriter(MinorType type, int precision, int scale) {
      super(type, precision, scale);
    }

    @Override
    public void writeVarDecimal(BigDecimal value) {
      setNextOffset();
      super.writeVarDecimal(value);
      increaseOffset();
    }

    @Override
    public void writeVarDecimal(int start, int end, DrillBuf buffer, int precision, int scale) {
      setNextOffset();
      super.writeVarDecimal(start, end, buffer, precision, scale);
      increaseOffset();
    }
  }

  private class BitUnionListElementWriter extends UnionVectorWriter.BitUnionWriter {

    private BitUnionListElementWriter(MinorType type) {
      super(type);
    }

    @Override
    public void writeBit(int value) {
      setNextOffset();
      super.writeBit(value);
      increaseOffset();
    }
  }

  private class UnionListUnionElementWriter extends UnionVectorWriter.ListUnionWriter {

    UnionListUnionElementWriter(ListVector vector) {
      super(vector);
    }

    @Override
    public void startList() {
      setNextOffset();
      super.startList();
      increaseOffset();
    }
  }

  class SingleMapUnionListElementWriter extends UnionVectorWriter.SingleMapUnionWriter {

    SingleMapUnionListElementWriter(MapVector container, FieldWriter parent, boolean unionEnabled) {
      super(container, parent, unionEnabled);
    }

    @Override
    public void start() {
      setNextOffset();
      super.start();
      increaseOffset();
    }
  }

  class SingleDictUnionListElementWriter extends UnionVectorWriter.SingleDictUnionWriter {

    SingleDictUnionListElementWriter(DictVector container, FieldWriter parent, boolean unionEnabled) {
      super(container, parent, unionEnabled);
    }

    @Override
    public void start() {
      setNextOffset();
      super.start();
      increaseOffset();
    }
  }
}
