

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.vector.complex;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;
import java.util.Iterator;
import java.util.Set;

import org.apache.drill.exec.vector.complex.impl.ComplexCopier;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;

import org.apache.drill.exec.util.CallBack;
import org.apache.drill.exec.expr.BasicTypeHelper;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;
import org.apache.drill.exec.record.MaterializedField;

import org.apache.drill.shaded.guava.com.google.common.annotations.VisibleForTesting;

/*
 * This class is generated using freemarker and the UnionVector.java template.
 */
@SuppressWarnings("unused")


/**
 * A vector which can hold values of different types. It does so by using a
 * MapVector which contains a vector for each primitive type that is stored.
 * MapVector is used in order to take advantage of its
 * serialization/deserialization methods, as well as the addOrGet method.
 *
 * For performance reasons, UnionVector stores a cached reference to each
 * subtype vector, to avoid having to do the map lookup each time the vector is
 * accessed.
 */
public class UnionVector implements ValueVector {

  public static final int NULL_MARKER = 0;
  public static final String TYPE_VECTOR_NAME = "types";
  public static final String INTERNAL_MAP_NAME = "internal";
  public static final int TYPE_COUNT = MinorType.values().length;

  // Types must be indexed by ordinal, not by number. That is,
  // use type.ordinal(), not type.getNumber().
  public static final MajorType TYPES[] = new MajorType[TYPE_COUNT];

  static {
    for (MinorType minorType : MinorType.values()) {
      TYPES[minorType.ordinal()] = Types.optional(minorType);
    }
  }

  private MaterializedField field;
  private BufferAllocator allocator;
  private Accessor accessor = new Accessor();
  private Mutator mutator = new Mutator();
  private int valueCount;

  /**
   * Map which holds one vector for each subtype, along with a vector that indicates
   * types and the null state. There appears to be no reason other than convenience
   * for using a map. Future implementations may wish to store vectors directly in
   * the union vector, but must then implement the required vector serialization/
   * deserialization and other functionality.
   */

  private MapVector internalMap;

  /**
   * Cached type vector. The vector's permanent location is in the
   * internal map, it is cached for performance. Call
   * {@link #getTypeVector()} to get the cached copy, or to refresh
   * the cache from the internal map if not set.
   */

  private UInt1Vector typeVector;

  /**
   * Set of cached vectors that duplicate vectors store in the
   * internal map. Used to avoid a name lookup on every access.
   * The cache is populated as vectors are added. But, after the
   * union is sent over the wire, the map is populated, but the
   * array is not. It will be repopulated upon first access to
   * the deserialized vectors.
   */

  private ValueVector cachedSubtypes[] = new ValueVector[MinorType.values().length];

  private FieldReader reader;

  private final CallBack callBack;

  public UnionVector(MaterializedField field, BufferAllocator allocator, CallBack callBack) {

    // The metadata may start off listing subtypes for which vectors
    // do not actually exist. It appears that the semantics are to list
    // the subtypes that *could* appear. For example, in a sort we may
    // have two types: one batch has type A, the other type B, but the
    // batches must list both A and B as subtypes.

    this.field = field.clone();
    this.allocator = allocator;
    this.internalMap = new MapVector(INTERNAL_MAP_NAME, allocator, callBack);
    this.typeVector = internalMap.addOrGet(TYPE_VECTOR_NAME, Types.required(MinorType.UINT1), UInt1Vector.class);
    this.field.addChild(internalMap.getField().clone());
    this.callBack = callBack;
  }

  @Override
  public BufferAllocator getAllocator() {
    return allocator;
  }

  public List<MinorType> getSubTypes() {
    return field.getType().getSubTypeList();
  }

  @SuppressWarnings("unchecked")
  public <T extends ValueVector> T subtype(MinorType type) {
    return (T) cachedSubtypes[type.ordinal()];
  }

  /**
   * Add an externally-created subtype vector. The vector must represent a type that
   * does not yet exist in the union, and must be of OPTIONAL mode. Does not call
   * the callback since the client (presumably) knows that it is adding the type.
   * The caller must also allocate the buffer for the vector.
   *
   * @param vector subtype vector to add
   */

  public void addType(ValueVector vector) {
    MinorType type = vector.getField().getType().getMinorType();
    assert subtype(type) == null;
    assert vector.getField().getType().getMode() == DataMode.OPTIONAL;
    assert vector.getField().getName().equals(type.name().toLowerCase());
    cachedSubtypes[type.ordinal()] = vector;
    internalMap.putChild(type.name(), vector);
    addSubType(type);
  }

  public void addSubType(MinorType type) {
    if (field.getType().getSubTypeList().contains(type)) {
      return;
    }
    field.replaceType(
        MajorType.newBuilder(field.getType()).addSubType(type).build());
    if (callBack != null) {
      callBack.doWork();
    }
  }

  /**
   * "Classic" way to add a subtype when working directly with a union vector.
   * Creates the vector, adds it to the internal structures and creates a
   * new buffer of the default size.
   *
   * @param type the type to add
   * @param vectorClass class of the vector to create
   * @return typed form of the new value vector
   */

  private <T extends ValueVector> T classicAddType(MinorType type, Class<? extends ValueVector> vectorClass) {
    int vectorCount = internalMap.size();
    @SuppressWarnings("unchecked")
    T vector = (T) internalMap.addOrGet(type.name().toLowerCase(), TYPES[type.ordinal()], vectorClass);
    cachedSubtypes[type.ordinal()] = vector;
    if (internalMap.size() > vectorCount) {
      vector.allocateNew();
      addSubType(type);
      if (callBack != null) {
        callBack.doWork();
      }
    }
    return vector;
  }

  public MapVector getMap() {
    MapVector mapVector = subtype(MinorType.MAP);
    if (mapVector == null) {
      mapVector = classicAddType(MinorType.MAP, MapVector.class);
    }
    return mapVector;
  }

  public DictVector getDict() {
    DictVector dictVector = subtype(MinorType.DICT);
    if (dictVector == null) {
      dictVector = classicAddType(MinorType.DICT, DictVector.class);
    }
    return dictVector;
  }

  public ListVector getList() {
    ListVector listVector = subtype(MinorType.LIST);
    if (listVector == null) {
      listVector = classicAddType(MinorType.LIST, ListVector.class);
    }
    return listVector;
  }

  public NullableTinyIntVector getTinyIntVector() {
    NullableTinyIntVector vector = subtype(MinorType.TINYINT);
    if (vector == null) {
      vector = classicAddType(MinorType.TINYINT, NullableTinyIntVector.class);
    }
    return vector;
  }

  public NullableUInt1Vector getUInt1Vector() {
    NullableUInt1Vector vector = subtype(MinorType.UINT1);
    if (vector == null) {
      vector = classicAddType(MinorType.UINT1, NullableUInt1Vector.class);
    }
    return vector;
  }

  public NullableUInt2Vector getUInt2Vector() {
    NullableUInt2Vector vector = subtype(MinorType.UINT2);
    if (vector == null) {
      vector = classicAddType(MinorType.UINT2, NullableUInt2Vector.class);
    }
    return vector;
  }

  public NullableSmallIntVector getSmallIntVector() {
    NullableSmallIntVector vector = subtype(MinorType.SMALLINT);
    if (vector == null) {
      vector = classicAddType(MinorType.SMALLINT, NullableSmallIntVector.class);
    }
    return vector;
  }

  public NullableIntVector getIntVector() {
    NullableIntVector vector = subtype(MinorType.INT);
    if (vector == null) {
      vector = classicAddType(MinorType.INT, NullableIntVector.class);
    }
    return vector;
  }

  public NullableUInt4Vector getUInt4Vector() {
    NullableUInt4Vector vector = subtype(MinorType.UINT4);
    if (vector == null) {
      vector = classicAddType(MinorType.UINT4, NullableUInt4Vector.class);
    }
    return vector;
  }

  public NullableFloat4Vector getFloat4Vector() {
    NullableFloat4Vector vector = subtype(MinorType.FLOAT4);
    if (vector == null) {
      vector = classicAddType(MinorType.FLOAT4, NullableFloat4Vector.class);
    }
    return vector;
  }

  public NullableTimeVector getTimeVector() {
    NullableTimeVector vector = subtype(MinorType.TIME);
    if (vector == null) {
      vector = classicAddType(MinorType.TIME, NullableTimeVector.class);
    }
    return vector;
  }

  public NullableIntervalYearVector getIntervalYearVector() {
    NullableIntervalYearVector vector = subtype(MinorType.INTERVALYEAR);
    if (vector == null) {
      vector = classicAddType(MinorType.INTERVALYEAR, NullableIntervalYearVector.class);
    }
    return vector;
  }

  public NullableBigIntVector getBigIntVector() {
    NullableBigIntVector vector = subtype(MinorType.BIGINT);
    if (vector == null) {
      vector = classicAddType(MinorType.BIGINT, NullableBigIntVector.class);
    }
    return vector;
  }

  public NullableUInt8Vector getUInt8Vector() {
    NullableUInt8Vector vector = subtype(MinorType.UINT8);
    if (vector == null) {
      vector = classicAddType(MinorType.UINT8, NullableUInt8Vector.class);
    }
    return vector;
  }

  public NullableFloat8Vector getFloat8Vector() {
    NullableFloat8Vector vector = subtype(MinorType.FLOAT8);
    if (vector == null) {
      vector = classicAddType(MinorType.FLOAT8, NullableFloat8Vector.class);
    }
    return vector;
  }

  public NullableDateVector getDateVector() {
    NullableDateVector vector = subtype(MinorType.DATE);
    if (vector == null) {
      vector = classicAddType(MinorType.DATE, NullableDateVector.class);
    }
    return vector;
  }

  public NullableTimeStampVector getTimeStampVector() {
    NullableTimeStampVector vector = subtype(MinorType.TIMESTAMP);
    if (vector == null) {
      vector = classicAddType(MinorType.TIMESTAMP, NullableTimeStampVector.class);
    }
    return vector;
  }

  public NullableIntervalDayVector getIntervalDayVector() {
    NullableIntervalDayVector vector = subtype(MinorType.INTERVALDAY);
    if (vector == null) {
      vector = classicAddType(MinorType.INTERVALDAY, NullableIntervalDayVector.class);
    }
    return vector;
  }

  public NullableIntervalVector getIntervalVector() {
    NullableIntervalVector vector = subtype(MinorType.INTERVAL);
    if (vector == null) {
      vector = classicAddType(MinorType.INTERVAL, NullableIntervalVector.class);
    }
    return vector;
  }

  public NullableVarBinaryVector getVarBinaryVector() {
    NullableVarBinaryVector vector = subtype(MinorType.VARBINARY);
    if (vector == null) {
      vector = classicAddType(MinorType.VARBINARY, NullableVarBinaryVector.class);
    }
    return vector;
  }

  public NullableVarCharVector getVarCharVector() {
    NullableVarCharVector vector = subtype(MinorType.VARCHAR);
    if (vector == null) {
      vector = classicAddType(MinorType.VARCHAR, NullableVarCharVector.class);
    }
    return vector;
  }

  public NullableVar16CharVector getVar16CharVector() {
    NullableVar16CharVector vector = subtype(MinorType.VAR16CHAR);
    if (vector == null) {
      vector = classicAddType(MinorType.VAR16CHAR, NullableVar16CharVector.class);
    }
    return vector;
  }

  public NullableVarDecimalVector getVarDecimalVector() {
    NullableVarDecimalVector vector = subtype(MinorType.VARDECIMAL);
    if (vector == null) {
      vector = classicAddType(MinorType.VARDECIMAL, NullableVarDecimalVector.class);
    }
    return vector;
  }

  public NullableBitVector getBitVector() {
    NullableBitVector vector = subtype(MinorType.BIT);
    if (vector == null) {
      vector = classicAddType(MinorType.BIT, NullableBitVector.class);
    }
    return vector;
  }

  /**
   * Add or get a type member given the type.
   *
   * @param type the type of the vector to retrieve
   * @return the (potentially newly created) vector that backs the given type
   */

  public ValueVector getMember(MinorType type) {
    switch (type) {
    case MAP:
      return getMap();
    case LIST:
      return getList();
    case DICT:
      return getDict();
    case TINYINT:
      return getTinyIntVector();
    case UINT1:
      return getUInt1Vector();
    case UINT2:
      return getUInt2Vector();
    case SMALLINT:
      return getSmallIntVector();
    case INT:
      return getIntVector();
    case UINT4:
      return getUInt4Vector();
    case FLOAT4:
      return getFloat4Vector();
    case TIME:
      return getTimeVector();
    case INTERVALYEAR:
      return getIntervalYearVector();
    case BIGINT:
      return getBigIntVector();
    case UINT8:
      return getUInt8Vector();
    case FLOAT8:
      return getFloat8Vector();
    case DATE:
      return getDateVector();
    case TIMESTAMP:
      return getTimeStampVector();
    case INTERVALDAY:
      return getIntervalDayVector();
    case INTERVAL:
      return getIntervalVector();
    case VARBINARY:
      return getVarBinaryVector();
    case VARCHAR:
      return getVarCharVector();
    case VAR16CHAR:
      return getVar16CharVector();
    case VARDECIMAL:
      return getVarDecimalVector();
    case BIT:
      return getBitVector();
    default:
      throw new UnsupportedOperationException(type.toString());
    }
  }

  @SuppressWarnings("unchecked")
  public <T extends ValueVector> T member(MinorType type) {
    return (T) getMember(type);
  }

  public int getTypeValue(int index) {
    return getTypeVector().getAccessor().get(index);
  }

  public UInt1Vector getTypeVector() {
    if (typeVector == null) {
      typeVector = (UInt1Vector) internalMap.getChild(TYPE_VECTOR_NAME);
    }
    return typeVector;
  }

  @VisibleForTesting
  public MapVector getTypeMap() {
    return internalMap;
  }

  @Override
  public void allocateNew() throws OutOfMemoryException {
    internalMap.allocateNew();
    getTypeVector().zeroVector();
  }

  public void allocateNew(int rowCount) throws OutOfMemoryException {
    // The map vector does not have a form that takes a row count,
    // but it should.
    internalMap.allocateNew();
    getTypeVector().zeroVector();
  }

  @Override
  public boolean allocateNewSafe() {
    boolean safe = internalMap.allocateNewSafe();
    if (safe) {
      getTypeVector().zeroVector();
    }
    return safe;
  }

  @Override
  public void setInitialCapacity(int numRecords) { }

  @Override
  public int getValueCapacity() {
    return Math.min(getTypeVector().getValueCapacity(), internalMap.getValueCapacity());
  }

  @Override
  public void close() { }

  @Override
  public void clear() {
    internalMap.clear();
  }

  @Override
  public MaterializedField getField() { return field; }

  @Override
  public void collectLedgers(Set<BufferLedger> ledgers) {
    internalMap.collectLedgers(ledgers);
  }

  @Override
  public int getPayloadByteCount(int valueCount) {
    return internalMap.getPayloadByteCount(valueCount);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator) {
    return new TransferImpl(field, allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator) {
    return new TransferImpl(field.withPath(ref), allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector target) {
    return new TransferImpl((UnionVector) target);
  }

  public void transferTo(UnionVector target) {
    internalMap.makeTransferPair(target.internalMap).transfer();
    target.valueCount = valueCount;
  }

  public void copyFrom(int inIndex, int outIndex, UnionVector from) {
    from.getReader().setPosition(inIndex);
    getWriter().setPosition(outIndex);
    ComplexCopier.copy(from.reader, mutator.writer);
  }

  public void copyFromSafe(int inIndex, int outIndex, UnionVector from) {
    copyFrom(inIndex, outIndex, from);
  }

  @Override
  public void copyEntry(int toIndex, ValueVector from, int fromIndex) {
    copyFromSafe(fromIndex, toIndex, (UnionVector) from);
  }

  /**
   * Add a vector that matches the argument. Transfer the buffer from the argument
   * to the new vector.
   *
   * @param v the vector to clone and add
   * @return the cloned vector that now holds the data from the argument
   */

  public ValueVector addVector(ValueVector v) {
    String name = v.getField().getType().getMinorType().name().toLowerCase();
    MajorType type = v.getField().getType();
    MinorType minorType = type.getMinorType();
    Preconditions.checkState(internalMap.getChild(name) == null, String.format("%s vector already exists", name));
    final ValueVector newVector = internalMap.addOrGet(name, type, BasicTypeHelper.getValueVectorClass(minorType, type.getMode()));
    v.makeTransferPair(newVector).transfer();
    internalMap.putChild(name, newVector);
    cachedSubtypes[minorType.ordinal()] = newVector;
    addSubType(minorType);
    return newVector;
  }

  // Called from SchemaUtil

  public ValueVector setFirstType(ValueVector v, int newValueCount) {

    // We can't check that this really is the first subtype since
    // the subtypes can be declared before vectors are added.

    Preconditions.checkState(accessor.getValueCount() == 0);
    final ValueVector vv = addVector(v);
    MinorType type = v.getField().getType().getMinorType();
    ValueVector.Accessor vAccessor = vv.getAccessor();
    for (int i = 0; i < newValueCount; i++) {
      if (! vAccessor.isNull(i)) {
        mutator.setType(i, type);
      } else {
        mutator.setNull(i);
      }
    }
    mutator.setValueCount(newValueCount);
    return vv;
  }

  @Override
  public void toNullable(ValueVector nullableVector) {
    throw new UnsupportedOperationException();
  }

  private class TransferImpl implements TransferPair {

    private final UnionVector to;

    public TransferImpl(MaterializedField field, BufferAllocator allocator) {
      to = new UnionVector(field, allocator, null);
    }

    public TransferImpl(UnionVector to) {
      this.to = to;
    }

    @Override
    public void transfer() {
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) { }

    @Override
    public ValueVector getTo() {
      return to;
    }

    @Override
    public void copyValueSafe(int from, int to) {
      this.to.copyFrom(from, to, UnionVector.this);
    }
  }

  @Override
  public Accessor getAccessor() { return accessor; }

  @Override
  public Mutator getMutator() { return mutator; }

  @Override
  public FieldReader getReader() {
    if (reader == null) {
      reader = new UnionReader(this);
    }
    return reader;
  }

  public FieldWriter getWriter() {
    if (mutator.writer == null) {
      mutator.writer = new UnionWriter(this);
    }
    return mutator.writer;
  }

  @Override
  public UserBitShared.SerializedField getMetadata() {
    return getField()
            .getAsBuilder()
            .setBufferLength(getBufferSize())
            .setValueCount(valueCount)
            .addChild(internalMap.getMetadata())
            .build();
  }

  @Override
  public int getBufferSize() {
    return internalMap.getBufferSize();
  }

  @Override
  public int getAllocatedSize() {
    return internalMap.getAllocatedSize();
  }

  @Override
  public int getBufferSizeFor(final int valueCount) {
    if (valueCount == 0) {
      return 0;
    }

    long bufferSize = 0;
    for (final ValueVector v : (Iterable<ValueVector>) this) {
      bufferSize += v.getBufferSizeFor(valueCount);
    }

    return (int) bufferSize;
  }

  @Override
  public DrillBuf[] getBuffers(boolean clear) {
    return internalMap.getBuffers(clear);
  }

  @Override
  public void load(UserBitShared.SerializedField metadata, DrillBuf buffer) {
    valueCount = metadata.getValueCount();

    internalMap.load(metadata.getChild(0), buffer);
  }

  @Override
  public Iterator<ValueVector> iterator() {
    return internalMap.iterator();
  }

  public class Accessor extends BaseValueVector.BaseAccessor {

    @Override
    public Object getObject(int index) {
      int ordinal = getTypeVector().getAccessor().get(index);
      if (ordinal == NULL_MARKER) {
        return null;
      }
      // Warning: do not use valueOf as that uses Protobuf
      // field numbers, not enum ordinals.
      MinorType type = MinorType.values()[ordinal];
      switch (type) {
      case TINYINT:
        return getTinyIntVector().getAccessor().getObject(index);
      case UINT1:
        return getUInt1Vector().getAccessor().getObject(index);
      case UINT2:
        return getUInt2Vector().getAccessor().getObject(index);
      case SMALLINT:
        return getSmallIntVector().getAccessor().getObject(index);
      case INT:
        return getIntVector().getAccessor().getObject(index);
      case UINT4:
        return getUInt4Vector().getAccessor().getObject(index);
      case FLOAT4:
        return getFloat4Vector().getAccessor().getObject(index);
      case TIME:
        return getTimeVector().getAccessor().getObject(index);
      case INTERVALYEAR:
        return getIntervalYearVector().getAccessor().getObject(index);
      case BIGINT:
        return getBigIntVector().getAccessor().getObject(index);
      case UINT8:
        return getUInt8Vector().getAccessor().getObject(index);
      case FLOAT8:
        return getFloat8Vector().getAccessor().getObject(index);
      case DATE:
        return getDateVector().getAccessor().getObject(index);
      case TIMESTAMP:
        return getTimeStampVector().getAccessor().getObject(index);
      case INTERVALDAY:
        return getIntervalDayVector().getAccessor().getObject(index);
      case INTERVAL:
        return getIntervalVector().getAccessor().getObject(index);
      case VARBINARY:
        return getVarBinaryVector().getAccessor().getObject(index);
      case VARCHAR:
        return getVarCharVector().getAccessor().getObject(index);
      case VAR16CHAR:
        return getVar16CharVector().getAccessor().getObject(index);
      case VARDECIMAL:
        return getVarDecimalVector().getAccessor().getObject(index);
      case BIT:
        return getBitVector().getAccessor().getObject(index);
      case MAP:
        return getMap().getAccessor().getObject(index);
      case LIST:
        return getList().getAccessor().getObject(index);
      case DICT:
        return getDict().getAccessor().getObject(index);
      default:
        throw new UnsupportedOperationException("Cannot support type: " + type.name());
      }
    }

    public byte[] get(int index) { return null; }

    public void get(int index, ComplexHolder holder) { }

    public void get(int index, UnionHolder holder) {
      FieldReader reader = new UnionReader(UnionVector.this);
      reader.setPosition(index);
      holder.reader = reader;
    }

    @Override
    public int getValueCount() { return valueCount; }

    @Override
    public boolean isNull(int index) {

      // Note that type code == 0 is used to indicate a null.
      // This corresponds to the LATE type, not the NULL type.
      // This is presumably an artifact of an earlier implementation...

      return getTypeVector().getAccessor().get(index) == NULL_MARKER;
    }

    public int isSet(int index) {
      return isNull(index) ? 0 : 1;
    }
  }

  public class Mutator extends BaseValueVector.BaseMutator {

    protected UnionWriter writer;

    @Override
    public void setValueCount(int valueCount) {
      UnionVector.this.valueCount = valueCount;

      // Get each claimed child type. This will force creation
      // of the vector in the internal map so that we can properly
      // set the size of that vector.
      //
      // TODO: This is a waste, but the semantics of this class
      // are murky: it is not fully supported. Without this, if we
      // ask for the child type vector later, it will have zero values
      // even if this Union has a non-zero count.
      //
      // A better long-term solution would be to not add types that
      // are not needed, or to remove those types here. In either case,
      // the internal structure will be consistent with the claimed
      // metadata (every type in metadata will have a matching vector in
      // the internal map.)

      for (MinorType type : getSubTypes()) {
        getMember(type);
      }
      internalMap.getMutator().setValueCount(valueCount);
    }

    public void setSafe(int index, UnionHolder holder) {
      FieldReader reader = holder.reader;
      if (writer == null) {
        writer = new UnionWriter(UnionVector.this);
      }
      writer.setPosition(index);
      MinorType type = reader.getType().getMinorType();
      switch (type) {
      case TINYINT:
        NullableTinyIntHolder tinyIntHolder = new NullableTinyIntHolder();
        reader.read(tinyIntHolder);
        setSafe(index, tinyIntHolder);
        break;
      case UINT1:
        NullableUInt1Holder uInt1Holder = new NullableUInt1Holder();
        reader.read(uInt1Holder);
        setSafe(index, uInt1Holder);
        break;
      case UINT2:
        NullableUInt2Holder uInt2Holder = new NullableUInt2Holder();
        reader.read(uInt2Holder);
        setSafe(index, uInt2Holder);
        break;
      case SMALLINT:
        NullableSmallIntHolder smallIntHolder = new NullableSmallIntHolder();
        reader.read(smallIntHolder);
        setSafe(index, smallIntHolder);
        break;
      case INT:
        NullableIntHolder intHolder = new NullableIntHolder();
        reader.read(intHolder);
        setSafe(index, intHolder);
        break;
      case UINT4:
        NullableUInt4Holder uInt4Holder = new NullableUInt4Holder();
        reader.read(uInt4Holder);
        setSafe(index, uInt4Holder);
        break;
      case FLOAT4:
        NullableFloat4Holder float4Holder = new NullableFloat4Holder();
        reader.read(float4Holder);
        setSafe(index, float4Holder);
        break;
      case TIME:
        NullableTimeHolder timeHolder = new NullableTimeHolder();
        reader.read(timeHolder);
        setSafe(index, timeHolder);
        break;
      case INTERVALYEAR:
        NullableIntervalYearHolder intervalYearHolder = new NullableIntervalYearHolder();
        reader.read(intervalYearHolder);
        setSafe(index, intervalYearHolder);
        break;
      case BIGINT:
        NullableBigIntHolder bigIntHolder = new NullableBigIntHolder();
        reader.read(bigIntHolder);
        setSafe(index, bigIntHolder);
        break;
      case UINT8:
        NullableUInt8Holder uInt8Holder = new NullableUInt8Holder();
        reader.read(uInt8Holder);
        setSafe(index, uInt8Holder);
        break;
      case FLOAT8:
        NullableFloat8Holder float8Holder = new NullableFloat8Holder();
        reader.read(float8Holder);
        setSafe(index, float8Holder);
        break;
      case DATE:
        NullableDateHolder dateHolder = new NullableDateHolder();
        reader.read(dateHolder);
        setSafe(index, dateHolder);
        break;
      case TIMESTAMP:
        NullableTimeStampHolder timeStampHolder = new NullableTimeStampHolder();
        reader.read(timeStampHolder);
        setSafe(index, timeStampHolder);
        break;
      case INTERVALDAY:
        NullableIntervalDayHolder intervalDayHolder = new NullableIntervalDayHolder();
        reader.read(intervalDayHolder);
        setSafe(index, intervalDayHolder);
        break;
      case INTERVAL:
        NullableIntervalHolder intervalHolder = new NullableIntervalHolder();
        reader.read(intervalHolder);
        setSafe(index, intervalHolder);
        break;
      case VARBINARY:
        NullableVarBinaryHolder varBinaryHolder = new NullableVarBinaryHolder();
        reader.read(varBinaryHolder);
        setSafe(index, varBinaryHolder);
        break;
      case VARCHAR:
        NullableVarCharHolder varCharHolder = new NullableVarCharHolder();
        reader.read(varCharHolder);
        setSafe(index, varCharHolder);
        break;
      case VAR16CHAR:
        NullableVar16CharHolder var16CharHolder = new NullableVar16CharHolder();
        reader.read(var16CharHolder);
        setSafe(index, var16CharHolder);
        break;
      case VARDECIMAL:
        NullableVarDecimalHolder varDecimalHolder = new NullableVarDecimalHolder();
        reader.read(varDecimalHolder);
        setSafe(index, varDecimalHolder);
        break;
      case BIT:
        NullableBitHolder bitHolder = new NullableBitHolder();
        reader.read(bitHolder);
        setSafe(index, bitHolder);
        break;
      case MAP:
        ComplexCopier.copy(reader, writer);
        break;
      case LIST:
        ComplexCopier.copy(reader, writer);
        break;
      default:
        throw new UnsupportedOperationException();
      }
    }

    public void setSafe(int index, NullableTinyIntHolder holder) {
      setType(index, MinorType.TINYINT);
      getTinyIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt1Holder holder) {
      setType(index, MinorType.UINT1);
      getUInt1Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt2Holder holder) {
      setType(index, MinorType.UINT2);
      getUInt2Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableSmallIntHolder holder) {
      setType(index, MinorType.SMALLINT);
      getSmallIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntHolder holder) {
      setType(index, MinorType.INT);
      getIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt4Holder holder) {
      setType(index, MinorType.UINT4);
      getUInt4Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat4Holder holder) {
      setType(index, MinorType.FLOAT4);
      getFloat4Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeHolder holder) {
      setType(index, MinorType.TIME);
      getTimeVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalYearHolder holder) {
      setType(index, MinorType.INTERVALYEAR);
      getIntervalYearVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBigIntHolder holder) {
      setType(index, MinorType.BIGINT);
      getBigIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt8Holder holder) {
      setType(index, MinorType.UINT8);
      getUInt8Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat8Holder holder) {
      setType(index, MinorType.FLOAT8);
      getFloat8Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableDateHolder holder) {
      setType(index, MinorType.DATE);
      getDateVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampHolder holder) {
      setType(index, MinorType.TIMESTAMP);
      getTimeStampVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalDayHolder holder) {
      setType(index, MinorType.INTERVALDAY);
      getIntervalDayVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalHolder holder) {
      setType(index, MinorType.INTERVAL);
      getIntervalVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarBinaryHolder holder) {
      setType(index, MinorType.VARBINARY);
      getVarBinaryVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarCharHolder holder) {
      setType(index, MinorType.VARCHAR);
      getVarCharVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVar16CharHolder holder) {
      setType(index, MinorType.VAR16CHAR);
      getVar16CharVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarDecimalHolder holder) {
      setType(index, MinorType.VARDECIMAL);
      getVarDecimalVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBitHolder holder) {
      setType(index, MinorType.BIT);
      getBitVector().getMutator().setSafe(index, holder);
    }


    public void setType(int index, MinorType type) {
      getTypeVector().getMutator().setSafe(index, type.ordinal());
    }

    public void setNull(int index) {
      getTypeVector().getMutator().setSafe(index, NULL_MARKER);
    }

    @Override
    public void reset() { }

    @Override
    public void generateTestData(int values) { }
  }

  @Override
  public void exchange(ValueVector other) {
    throw new UnsupportedOperationException("Union vector does not yet support exchange()");
  }
}
