
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.drill.exec.vector.accessor;

import java.math.BigDecimal;
import java.util.Arrays;

import org.joda.time.DateTimeZone;
import org.joda.time.Period;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;

/**
 * This class is generated using Freemarker and the KeyAccessors.java template.
 */
public class KeyAccessors {


  public static class BooleanKeyAccessor extends AbstractKeyAccessor {

    public BooleanKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(boolean key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key == keyReader.getBoolean()) {
          return true;
        }
      }
      return false;
    }
  }

  public static class IntegerKeyAccessor extends AbstractKeyAccessor {

    public IntegerKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(int key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key == keyReader.getInt()) {
          return true;
        }
      }
      return false;
    }
  }

  public static class LongKeyAccessor extends AbstractKeyAccessor {

    public LongKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(long key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key == keyReader.getLong()) {
          return true;
        }
      }
      return false;
    }
  }

  public static class DoubleKeyAccessor extends AbstractKeyAccessor {

    public DoubleKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(double key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key == keyReader.getDouble()) {
          return true;
        }
      }
      return false;
    }
  }

  public static class StringKeyAccessor extends AbstractKeyAccessor {

    public StringKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(String key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getString())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static class BytesKeyAccessor extends AbstractKeyAccessor {

    public BytesKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(byte[] key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (Arrays.equals(key, keyReader.getBytes())) {
          return true;
        }
      }
      return false;
    }
  }

  public static class DecimalKeyAccessor extends AbstractKeyAccessor {

    public DecimalKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(BigDecimal key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getDecimal())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static class PeriodKeyAccessor extends AbstractKeyAccessor {

    public PeriodKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(Period key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getPeriod())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static class DateKeyAccessor extends AbstractKeyAccessor {

    public DateKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(LocalDate key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getDate())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static class TimeKeyAccessor extends AbstractKeyAccessor {

    public TimeKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(LocalTime key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getTime())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static class TimestampKeyAccessor extends AbstractKeyAccessor {

    public TimestampKeyAccessor(DictReader dictReader, ScalarReader keyReader) {
      super(dictReader, keyReader);
    }

    @Override
    public boolean find(Instant key) {
      dictReader.rewind();
      while (dictReader.next()) {
        if (key.equals(keyReader.getTimestamp())) { // key should not be null so it is safe
          return true;
        }
      }
      return false;
    }
  }

  public static KeyAccessor getAccessor(DictReader dictReader, ScalarReader keyReader) {
    switch (keyReader.valueType()) {
      case BOOLEAN:
        return new BooleanKeyAccessor(dictReader, keyReader);
      case BYTES:
        return new BytesKeyAccessor(dictReader, keyReader);
      case DECIMAL:
        return new DecimalKeyAccessor(dictReader, keyReader);
      case DOUBLE:
        return new DoubleKeyAccessor(dictReader, keyReader);
      case INTEGER:
        return new IntegerKeyAccessor(dictReader, keyReader);
      case LONG:
        return new LongKeyAccessor(dictReader, keyReader);
      case PERIOD:
        return new PeriodKeyAccessor(dictReader, keyReader);
      case STRING:
        return new StringKeyAccessor(dictReader, keyReader);
      case DATE:
        return new DateKeyAccessor(dictReader, keyReader);
      case TIME:
        return new TimeKeyAccessor(dictReader, keyReader);
      case TIMESTAMP:
        return new TimestampKeyAccessor(dictReader, keyReader);
      default:
        throw new IllegalStateException("Unexpected key type: " + keyReader.valueType());
    }
  }
}
