/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// This class is generated using Freemarker and the ColumnAccessors.java template.

package org.apache.drill.exec.vector.accessor;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;

import org.apache.drill.common.types.TypeProtos.MajorType;
import org.apache.drill.common.types.Types;
import org.apache.drill.exec.vector.DateUtilities;
import org.apache.drill.exec.record.metadata.ColumnMetadata;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.util.DecimalUtility;
import org.apache.drill.exec.vector.accessor.reader.BaseScalarReader.BaseVarWidthReader;
import org.apache.drill.exec.vector.accessor.reader.BaseScalarReader.BaseFixedWidthReader;
import org.apache.drill.exec.vector.accessor.reader.VectorAccessor;
import org.apache.drill.exec.vector.accessor.writer.AbstractFixedWidthWriter.BaseFixedWidthWriter;
import org.apache.drill.exec.vector.accessor.writer.AbstractFixedWidthWriter.BaseIntWriter;
import org.apache.drill.exec.vector.accessor.writer.BaseVarWidthWriter;

import org.apache.drill.shaded.guava.com.google.common.base.Charsets;

import io.netty.buffer.DrillBuf;

import org.joda.time.Period;

/**
 * Basic accessors for most Drill vector types and modes. Each class has a bare-bones
 * accessors that converts from the "native" Drill type to the vectors. Many classes
 * also have "convenience" methods that convert from other Java types.
 * <p>
 * Writers work only with single vectors. Readers work with either single
 * vectors or a "hyper vector": a collection of vectors indexed together.
 * The details are hidden behind the {@link RowIndex} interface. If the reader
 * accesses a single vector, then the mutator is cached at bind time. However,
 * if the reader works with a hyper vector, then the vector is null at bind
 * time and must be retrieved for each row (since the vector differs row-by-
 * row.)
 */

public class ColumnAccessors {
  public static final LocalDateTime LOCAL_EPOCH = LocalDateTime.of(1970, 1, 1, 0, 0, 0);

  //------------------------------------------------------------------------
  // TinyInt readers and writers

  public static class TinyIntColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = TinyIntVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getByte(readOffset * VALUE_WIDTH);
    }
  }

  public static class TinyIntColumnWriter extends BaseIntWriter {

    private static final int VALUE_WIDTH = TinyIntVector.VALUE_WIDTH;

    private final TinyIntVector vector;

    public TinyIntColumnWriter(final ValueVector vector) {
      this.vector = (TinyIntVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setByte(offset, (byte) value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setByte(offset, (byte) value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      TinyIntColumnReader source = (TinyIntColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // UInt1 readers and writers

  public static class UInt1ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = UInt1Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getByte(readOffset * VALUE_WIDTH) & 0xFF;
    }
  }

  public static class UInt1ColumnWriter extends BaseIntWriter {

    private static final int VALUE_WIDTH = UInt1Vector.VALUE_WIDTH;

    private final UInt1Vector vector;

    public UInt1ColumnWriter(final ValueVector vector) {
      this.vector = (UInt1Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setByte(offset, (byte) value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setByte(offset, (byte) value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      UInt1ColumnReader source = (UInt1ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // UInt2 readers and writers

  public static class UInt2ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = UInt2Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getShort(readOffset * VALUE_WIDTH) & 0xFFFF;
    }
  }

  public static class UInt2ColumnWriter extends BaseIntWriter {

    private static final int VALUE_WIDTH = UInt2Vector.VALUE_WIDTH;

    private final UInt2Vector vector;

    public UInt2ColumnWriter(final ValueVector vector) {
      this.vector = (UInt2Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setShort(offset, (short) value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setShort(offset, (short) value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      UInt2ColumnReader source = (UInt2ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // SmallInt readers and writers

  public static class SmallIntColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = SmallIntVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getShort(readOffset * VALUE_WIDTH);
    }
  }

  public static class SmallIntColumnWriter extends BaseIntWriter {

    private static final int VALUE_WIDTH = SmallIntVector.VALUE_WIDTH;

    private final SmallIntVector vector;

    public SmallIntColumnWriter(final ValueVector vector) {
      this.vector = (SmallIntVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setShort(offset, (short) value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setShort(offset, (short) value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      SmallIntColumnReader source = (SmallIntColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Int readers and writers

  public static class IntColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = IntVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getInt(readOffset * VALUE_WIDTH);
    }
  }

  public static class IntColumnWriter extends BaseIntWriter {

    private static final int VALUE_WIDTH = IntVector.VALUE_WIDTH;

    private final IntVector vector;

    public IntColumnWriter(final ValueVector vector) {
      this.vector = (IntVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setInt(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      IntColumnReader source = (IntColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // UInt4 readers and writers

  public static class UInt4ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = UInt4Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      // Should be the following:
      // return ((long) buf.unsafeGetInt(readOffset * VALUE_WIDTH)) & 0xFFFF_FFFF;
      // else, the unsigned values of 32 bits are mapped to negative.
      return buf.getInt(readOffset * VALUE_WIDTH);
    }
  }

  public static class UInt4ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = UInt4Vector.VALUE_WIDTH;

    private final UInt4Vector vector;

    public UInt4ColumnWriter(final ValueVector vector) {
      this.vector = (UInt4Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setInt(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setInt((int) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, (int) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      UInt4ColumnReader source = (UInt4ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Float4 readers and writers

  public static class Float4ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Float4Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.FLOAT;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public float getFloat() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return Float.intBitsToFloat(buf.getInt(readOffset * VALUE_WIDTH));
    }

    @Override
    public double getDouble() {
      return getFloat();
    }
  }

  public static class Float4ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Float4Vector.VALUE_WIDTH;

    private final Float4Vector vector;

    public Float4ColumnWriter(final ValueVector vector) {
      this.vector = (Float4Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.FLOAT;
    }

    @Override
    public final void setFloat(final float value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, Float.floatToRawIntBits((float) value));
      vectorIndex.nextElement();
    }

    public final void writeFloat(final DrillBuf buf, final float value) {
      final int offset = 0;
      buf.setInt(offset, Float.floatToRawIntBits((float) value));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setFloat(value);
    }

    @Override
    public final void setLong(final long value) {
      setFloat(value);
    }

    @Override
    public final void setDouble(final double value) {
      setFloat((float) value);
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      setFloat(value.floatValue());
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setFloat((float) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeFloat(buf, (float) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Float4ColumnReader source = (Float4ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Time readers and writers

  public static class TimeColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = TimeVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.TIME;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public int getInt() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getInt(readOffset * VALUE_WIDTH);
    }

    @Override
    public final LocalTime getTime() {
      return DateUtilities.fromDrillTime(getInt());
    }
  }

  public static class TimeColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = TimeVector.VALUE_WIDTH;

    private final TimeVector vector;

    public TimeColumnWriter(final ValueVector vector) {
      this.vector = (TimeVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.INTEGER;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.TIME;
    }

    @Override
    public final void setInt(final int value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeInt(final DrillBuf buf, final int value) {
      final int offset = 0;
      buf.setInt(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setTime(final LocalTime value) {
      setInt(DateUtilities.toDrillTime(value));
    }

    @Override
    public final void setValue(final Object value) {
      setTime((LocalTime) value);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeInt(buf, DateUtilities.toDrillTime((LocalTime) value));
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      TimeColumnReader source = (TimeColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // IntervalYear readers and writers

  public static class IntervalYearColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = IntervalYearVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public Period getPeriod() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return DateUtilities.fromIntervalYear(
          buf.getInt(readOffset * VALUE_WIDTH));
    }
  }

  public static class IntervalYearColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = IntervalYearVector.VALUE_WIDTH;

    private final IntervalYearVector vector;

    public IntervalYearColumnWriter(final ValueVector vector) {
      this.vector = (IntervalYearVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override
    public final void setPeriod(final Period value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset,
          value.getYears() * 12 + value.getMonths());
      vectorIndex.nextElement();
    }

    public final void writePeriod(final DrillBuf buf, final Period value) {
      final int offset = 0;
      buf.setInt(offset,
          value.getYears() * 12 + value.getMonths());
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setPeriod((Period) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writePeriod(buf, (Period) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      IntervalYearColumnReader source = (IntervalYearColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Decimal9 readers and writers

  public static class Decimal9ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Decimal9Vector.VALUE_WIDTH;

    private MajorType type;

    @Override
    public void bindVector(ColumnMetadata schema, VectorAccessor va) {
      super.bindVector(schema, va);
      type = va.type();
    }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public BigDecimal getDecimal() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return DecimalUtility.getBigDecimalFromPrimitiveTypes(
          buf.getInt(readOffset * VALUE_WIDTH),
          type.getScale());
    }
  }

  public static class Decimal9ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Decimal9Vector.VALUE_WIDTH;

    private final Decimal9Vector vector;
    private MajorType type;

    public Decimal9ColumnWriter(final ValueVector vector) {
      type = vector.getField().getType();
      this.vector = (Decimal9Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset,
        DecimalUtility.getDecimal9FromBigDecimal(value,
            type.getScale()));
      vectorIndex.nextElement();
    }

    public final void writeDecimal(final DrillBuf buf, final BigDecimal value) {
      final int offset = 0;
      buf.setInt(offset,
        DecimalUtility.getDecimal9FromBigDecimal(value,
            type.getScale()));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setLong(final long value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDouble(final double value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setFloat(final float value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setDecimal((BigDecimal) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeDecimal(buf, (BigDecimal) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Decimal9ColumnReader source = (Decimal9ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // BigInt readers and writers

  public static class BigIntColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = BigIntVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public long getLong() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getLong(readOffset * VALUE_WIDTH);
    }
  }

  public static class BigIntColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = BigIntVector.VALUE_WIDTH;

    private final BigIntVector vector;

    public BigIntColumnWriter(final ValueVector vector) {
      this.vector = (BigIntVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public final void setLong(final long value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeLong(final DrillBuf buf, final long value) {
      final int offset = 0;
      buf.setLong(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setLong(value);
    }

    @Override
    public final void setFloat(final float value) {
      // Does not catch overflow from
      // double. See Math.round for details.
      setLong(Math.round(value));
    }

    @Override
    public final void setDouble(final double value) {
      // Does not catch overflow from
      // double. See Math.round for details.
      setLong(Math.round(value));
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      try {
        // Catches long overflow.
        setLong(value.longValueExact());
      } catch (ArithmeticException e) {
        throw InvalidConversionError.writeError(schema(), value, e);
      }
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setLong((long) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeLong(buf, (long) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      BigIntColumnReader source = (BigIntColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // UInt8 readers and writers

  public static class UInt8ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = UInt8Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public long getLong() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getLong(readOffset * VALUE_WIDTH);
    }
  }

  public static class UInt8ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = UInt8Vector.VALUE_WIDTH;

    private final UInt8Vector vector;

    public UInt8ColumnWriter(final ValueVector vector) {
      this.vector = (UInt8Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public final void setLong(final long value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeLong(final DrillBuf buf, final long value) {
      final int offset = 0;
      buf.setLong(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setLong((long) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeLong(buf, (long) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      UInt8ColumnReader source = (UInt8ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Float8 readers and writers

  public static class Float8ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Float8Vector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.DOUBLE;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public double getDouble() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return Double.longBitsToDouble(buf.getLong(readOffset * VALUE_WIDTH));
    }
  }

  public static class Float8ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Float8Vector.VALUE_WIDTH;

    private final Float8Vector vector;

    public Float8ColumnWriter(final ValueVector vector) {
      this.vector = (Float8Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.DOUBLE;
    }

    @Override
    public final void setDouble(final double value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset, Double.doubleToRawLongBits(value));
      vectorIndex.nextElement();
    }

    public final void writeDouble(final DrillBuf buf, final double value) {
      final int offset = 0;
      buf.setLong(offset, Double.doubleToRawLongBits(value));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setDouble(value);
    }

    @Override
    public final void setLong(final long value) {
      setDouble(value);
    }

    @Override
    public final void setFloat(final float value) {
      setDouble(value);
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      setDouble(value.doubleValue());
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setDouble((double) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeDouble(buf, (double) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Float8ColumnReader source = (Float8ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Date readers and writers

  public static class DateColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = DateVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.DATE;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public long getLong() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getLong(readOffset * VALUE_WIDTH);
    }

    @Override
    public final LocalDate getDate() {
      return DateUtilities.fromDrillDate(getLong());
    }
  }

  public static class DateColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = DateVector.VALUE_WIDTH;

    private final DateVector vector;

    public DateColumnWriter(final ValueVector vector) {
      this.vector = (DateVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.DATE;
    }

    @Override
    public final void setLong(final long value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeLong(final DrillBuf buf, final long value) {
      final int offset = 0;
      buf.setLong(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setDate(final LocalDate value) {
      setLong(DateUtilities.toDrillDate(value));
    }

    @Override
    public final void setValue(final Object value) {
      setDate((LocalDate) value);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeLong(buf, DateUtilities.toDrillDate((LocalDate) value));
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      DateColumnReader source = (DateColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // TimeStamp readers and writers

  public static class TimeStampColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = TimeStampVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.TIMESTAMP;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public long getLong() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return buf.getLong(readOffset * VALUE_WIDTH);
    }

    @Override
    public final Instant getTimestamp() {
      return DateUtilities.fromDrillTimestamp(getLong());
    }
  }

  public static class TimeStampColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = TimeStampVector.VALUE_WIDTH;

    private final TimeStampVector vector;

    public TimeStampColumnWriter(final ValueVector vector) {
      this.vector = (TimeStampVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.LONG;
    }

    @Override
    public ValueType extendedType() {
      return ValueType.TIMESTAMP;
    }

    @Override
    public final void setLong(final long value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset, value);
      vectorIndex.nextElement();
    }

    public final void writeLong(final DrillBuf buf, final long value) {
      final int offset = 0;
      buf.setLong(offset, value);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setTimestamp(final Instant value) {
      setLong(DateUtilities.toDrillTimestamp(value));
    }

    @Override
    public final void setValue(final Object value) {
      setTimestamp((Instant) value);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeLong(buf, DateUtilities.toDrillTimestamp((Instant) value));
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      TimeStampColumnReader source = (TimeStampColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Decimal18 readers and writers

  public static class Decimal18ColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Decimal18Vector.VALUE_WIDTH;

    private MajorType type;

    @Override
    public void bindVector(ColumnMetadata schema, VectorAccessor va) {
      super.bindVector(schema, va);
      type = va.type();
    }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public BigDecimal getDecimal() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return DecimalUtility.getBigDecimalFromPrimitiveTypes(
          buf.getLong(readOffset * VALUE_WIDTH),
          type.getScale());
    }
  }

  public static class Decimal18ColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Decimal18Vector.VALUE_WIDTH;

    private final Decimal18Vector vector;
    private MajorType type;

    public Decimal18ColumnWriter(final ValueVector vector) {
      type = vector.getField().getType();
      this.vector = (Decimal18Vector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setLong(offset,
          DecimalUtility.getDecimal18FromBigDecimal(value,
              type.getScale()));
      vectorIndex.nextElement();
    }

    public final void writeDecimal(final DrillBuf buf, final BigDecimal value) {
      final int offset = 0;
      buf.setLong(offset,
          DecimalUtility.getDecimal18FromBigDecimal(value,
              type.getScale()));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setLong(final long value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDouble(final double value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setFloat(final float value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setDecimal((BigDecimal) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeDecimal(buf, (BigDecimal) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Decimal18ColumnReader source = (Decimal18ColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // IntervalDay readers and writers

  public static class IntervalDayColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = IntervalDayVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public Period getPeriod() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      final int offset = readOffset * VALUE_WIDTH;
      return DateUtilities.fromIntervalDay(
          buf.getInt(offset),
          buf.getInt(offset + 4));
    }
  }

  public static class IntervalDayColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = IntervalDayVector.VALUE_WIDTH;

    private final IntervalDayVector vector;

    public IntervalDayColumnWriter(final ValueVector vector) {
      this.vector = (IntervalDayVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override
    public final void setPeriod(final Period value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, value.getDays());
      drillBuf.setInt(offset + 4, DateUtilities.periodToMillis(value));
      vectorIndex.nextElement();
    }

    public final void writePeriod(final DrillBuf buf, final Period value) {
      final int offset = 0;
      buf.setInt(offset, value.getDays());
      buf.setInt(offset + 4, DateUtilities.periodToMillis(value));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setPeriod((Period) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writePeriod(buf, (Period) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      IntervalDayColumnReader source = (IntervalDayColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Interval readers and writers

  public static class IntervalColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = IntervalVector.VALUE_WIDTH;

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public Period getPeriod() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      final int offset = readOffset * VALUE_WIDTH;
      return DateUtilities.fromInterval(
          buf.getInt(offset),
          buf.getInt(offset + 4),
          buf.getInt(offset + 8));
    }
  }

  public static class IntervalColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = IntervalVector.VALUE_WIDTH;

    private final IntervalVector vector;

    public IntervalColumnWriter(final ValueVector vector) {
      this.vector = (IntervalVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.PERIOD;
    }

    @Override
    public final void setPeriod(final Period value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setInt(offset, DateUtilities.periodToMonths(value));
      drillBuf.setInt(offset + 4, value.getDays());
      drillBuf.setInt(offset + 8, DateUtilities.periodToMillis(value));
      vectorIndex.nextElement();
    }

    public final void writePeriod(final DrillBuf buf, final Period value) {
      final int offset = 0;
      buf.setInt(offset, DateUtilities.periodToMonths(value));
      buf.setInt(offset + 4, value.getDays());
      buf.setInt(offset + 8, DateUtilities.periodToMillis(value));
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setPeriod((Period) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writePeriod(buf, (Period) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      IntervalColumnReader source = (IntervalColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Decimal38Sparse readers and writers

  public static class Decimal38SparseColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Decimal38SparseVector.VALUE_WIDTH;

    private MajorType type;

    @Override
    public void bindVector(ColumnMetadata schema, VectorAccessor va) {
      super.bindVector(schema, va);
      type = va.type();
    }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public BigDecimal getDecimal() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return DecimalUtility.getBigDecimalFromSparse(buf, readOffset * VALUE_WIDTH,
          6, type.getScale());
    }
  }

  public static class Decimal38SparseColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Decimal38SparseVector.VALUE_WIDTH;

    private final Decimal38SparseVector vector;
    private MajorType type;

    public Decimal38SparseColumnWriter(final ValueVector vector) {
      type = vector.getField().getType();
      this.vector = (Decimal38SparseVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      DecimalUtility.getSparseFromBigDecimal(value, drillBuf,
          offset, type.getScale(), 6);
      vectorIndex.nextElement();
    }

    public final void writeDecimal(final DrillBuf buf, final BigDecimal value) {
      final int offset = 0;
      DecimalUtility.getSparseFromBigDecimal(value, buf,
          offset, type.getScale(), 6);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setLong(final long value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDouble(final double value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setFloat(final float value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setDecimal((BigDecimal) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeDecimal(buf, (BigDecimal) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Decimal38SparseColumnReader source = (Decimal38SparseColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Decimal28Sparse readers and writers

  public static class Decimal28SparseColumnReader extends BaseFixedWidthReader {

    private static final int VALUE_WIDTH = Decimal28SparseVector.VALUE_WIDTH;

    private MajorType type;

    @Override
    public void bindVector(ColumnMetadata schema, VectorAccessor va) {
      super.bindVector(schema, va);
      type = va.type();
    }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public BigDecimal getDecimal() {
      final DrillBuf buf = bufferAccessor.buffer();
      final int readOffset = vectorIndex.offset();
      return DecimalUtility.getBigDecimalFromSparse(buf, readOffset * VALUE_WIDTH,
          5, type.getScale());
    }
  }

  public static class Decimal28SparseColumnWriter extends BaseFixedWidthWriter {

    private static final int VALUE_WIDTH = Decimal28SparseVector.VALUE_WIDTH;

    private final Decimal28SparseVector vector;
    private MajorType type;

    public Decimal28SparseColumnWriter(final ValueVector vector) {
      type = vector.getField().getType();
      this.vector = (Decimal28SparseVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override public int width() { return VALUE_WIDTH; }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      final int offset = prepareWrite() * VALUE_WIDTH;
      DecimalUtility.getSparseFromBigDecimal(value, drillBuf,
          offset, type.getScale(), 5);
      vectorIndex.nextElement();
    }

    public final void writeDecimal(final DrillBuf buf, final BigDecimal value) {
      final int offset = 0;
      DecimalUtility.getSparseFromBigDecimal(value, buf,
          offset, type.getScale(), 5);
      buf.writerIndex(VALUE_WIDTH);
    }

    @Override
    public final void setInt(final int value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setLong(final long value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDouble(final double value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setFloat(final float value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setValue(final Object value) {
      if (value != null) {
        setDecimal((BigDecimal) value);
      }
    }

    @Override
    public final void setDefaultValue(final Object value) {
      try (DrillBuf buf = vector.getAllocator().buffer(VALUE_WIDTH)) {
        writeDecimal(buf, (BigDecimal) value);
        emptyValue = new byte[VALUE_WIDTH];
        buf.getBytes(0, emptyValue);
      }
    }

    @Override
    public final void copy(ColumnReader from) {
      Decimal28SparseColumnReader source = (Decimal28SparseColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final int sourceOffset = source.offsetIndex() * VALUE_WIDTH;
      final int destOffset = prepareWrite() * VALUE_WIDTH;
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, VALUE_WIDTH);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // VarBinary readers and writers

  public static class VarBinaryColumnReader extends BaseVarWidthReader {

    @Override
    public ValueType valueType() {
      return ValueType.BYTES;
    }

    @Override
    public byte[] getBytes() {
      final DrillBuf buf = bufferAccessor.buffer();
      final long entry = offsetsReader.getEntry();
      return buf.unsafeGetMemory((int) (entry >> 32), (int) (entry & 0xFFFF_FFFF));
    }
  }

  public static class VarBinaryColumnWriter extends BaseVarWidthWriter {

    private final VarBinaryVector vector;

    public VarBinaryColumnWriter(final ValueVector vector) {
      super(((VarBinaryVector) vector).getOffsetVector());
      this.vector = (VarBinaryVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override
    public ValueType valueType() {
      return ValueType.BYTES;
    }

    @Override
    public final void setBytes(final byte[] value, final int len) {
      final int offset = prepareWrite(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.setNextOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void appendBytes(final byte[] value, final int len) {
      vectorIndex.prevElement();
      final int offset = prepareAppend(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.reviseOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void setValue(final Object value) {
      throw new InvalidConversionError("Generic object not supported for type VarBinary, "
          + "setBytes(byte[], final int len)");
    }

    @Override
    public final void setDefaultValue(final Object value) {
      emptyValue = (byte[]) value;
    }

    @Override
    public final void copy(ColumnReader from) {
      VarBinaryColumnReader source = (VarBinaryColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final long entry = source.getEntry();
      final int sourceOffset = (int) (entry >> 32);
      final int len = (int) (entry & 0xFFFF_FFFF);
      final int destOffset = prepareWrite(len);
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, len);
      offsetsWriter.setNextOffset(destOffset + len);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // VarChar readers and writers

  public static class VarCharColumnReader extends BaseVarWidthReader {

    @Override
    public ValueType valueType() {
      return ValueType.STRING;
    }

    @Override
    public byte[] getBytes() {
      final DrillBuf buf = bufferAccessor.buffer();
      final long entry = offsetsReader.getEntry();
      return buf.unsafeGetMemory((int) (entry >> 32), (int) (entry & 0xFFFF_FFFF));
    }

    @Override
    public String getString() {
      return new String(getBytes(), Charsets.UTF_8);
    }
  }

  public static class VarCharColumnWriter extends BaseVarWidthWriter {

    private final VarCharVector vector;

    public VarCharColumnWriter(final ValueVector vector) {
      super(((VarCharVector) vector).getOffsetVector());
      this.vector = (VarCharVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override
    public ValueType valueType() {
      return ValueType.STRING;
    }

    @Override
    public final void setBytes(final byte[] value, final int len) {
      final int offset = prepareWrite(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.setNextOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void appendBytes(final byte[] value, final int len) {
      vectorIndex.prevElement();
      final int offset = prepareAppend(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.reviseOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void setString(final String value) {
      final byte bytes[] = value.getBytes(Charsets.UTF_8);
      setBytes(bytes, bytes.length);
    }

    @Override
    public final void setValue(final Object value) {
      setString((String) value);
    }

    @Override
    public final void setDefaultValue(final Object value) {
      emptyValue = ((String) value).getBytes(Charsets.UTF_8);
    }

    @Override
    public final void copy(ColumnReader from) {
      VarCharColumnReader source = (VarCharColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final long entry = source.getEntry();
      final int sourceOffset = (int) (entry >> 32);
      final int len = (int) (entry & 0xFFFF_FFFF);
      final int destOffset = prepareWrite(len);
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, len);
      offsetsWriter.setNextOffset(destOffset + len);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // Var16Char readers and writers

  public static class Var16CharColumnReader extends BaseVarWidthReader {

    @Override
    public ValueType valueType() {
      return ValueType.STRING;
    }

    @Override
    public byte[] getBytes() {
      final DrillBuf buf = bufferAccessor.buffer();
      final long entry = offsetsReader.getEntry();
      return buf.unsafeGetMemory((int) (entry >> 32), (int) (entry & 0xFFFF_FFFF));
    }

    @Override
    public String getString() {
      return new String(getBytes(), Charsets.UTF_16);
    }
  }

  public static class Var16CharColumnWriter extends BaseVarWidthWriter {

    private final Var16CharVector vector;

    public Var16CharColumnWriter(final ValueVector vector) {
      super(((Var16CharVector) vector).getOffsetVector());
      this.vector = (Var16CharVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override
    public ValueType valueType() {
      return ValueType.STRING;
    }

    @Override
    public final void setBytes(final byte[] value, final int len) {
      final int offset = prepareWrite(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.setNextOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void appendBytes(final byte[] value, final int len) {
      vectorIndex.prevElement();
      final int offset = prepareAppend(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.reviseOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void setString(final String value) {
      final byte bytes[] = value.getBytes(Charsets.UTF_16);
      setBytes(bytes, bytes.length);
    }

    @Override
    public final void setValue(final Object value) {
      throw new InvalidConversionError("Generic object not supported for type Var16Char, "
          + "setBytes(byte[], final int len)");
    }

    @Override
    public final void setDefaultValue(final Object value) {
      emptyValue = ((String) value).getBytes(Charsets.UTF_16);
    }

    @Override
    public final void copy(ColumnReader from) {
      Var16CharColumnReader source = (Var16CharColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final long entry = source.getEntry();
      final int sourceOffset = (int) (entry >> 32);
      final int len = (int) (entry & 0xFFFF_FFFF);
      final int destOffset = prepareWrite(len);
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, len);
      offsetsWriter.setNextOffset(destOffset + len);
      vectorIndex.nextElement();
    }
  }

  //------------------------------------------------------------------------
  // VarDecimal readers and writers

  public static class VarDecimalColumnReader extends BaseVarWidthReader {

    private MajorType type;

    @Override
    public void bindVector(ColumnMetadata schema, VectorAccessor va) {
      super.bindVector(schema, va);
      type = va.type();
    }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public byte[] getBytes() {
      final DrillBuf buf = bufferAccessor.buffer();
      final long entry = offsetsReader.getEntry();
      return buf.unsafeGetMemory((int) (entry >> 32), (int) (entry & 0xFFFF_FFFF));
    }

    @Override
    public BigDecimal getDecimal() {
      final byte[] bytes = getBytes();
      BigInteger unscaledValue = bytes.length == 0 ? BigInteger.ZERO : new BigInteger(bytes);
      return new BigDecimal(unscaledValue, type.getScale());
    }
  }

  public static class VarDecimalColumnWriter extends BaseVarWidthWriter {

    private final VarDecimalVector vector;
    private int precision;
    private int scale;

    public VarDecimalColumnWriter(final ValueVector vector) {
      super(((VarDecimalVector) vector).getOffsetVector());
      // VarDecimal requires a scale. If not set, assume 0
      MajorType type = vector.getField().getType();
      precision = type.hasPrecision() ? type.getPrecision() : Types.maxPrecision(type.getMinorType());
      scale = type.hasScale() ? type.getScale() : 0;
      this.vector = (VarDecimalVector) vector;
    }

    @Override public BaseDataValueVector vector() { return vector; }

    @Override
    public ValueType valueType() {
      return ValueType.DECIMAL;
    }

    @Override
    public final void setBytes(final byte[] value, final int len) {
      final int offset = prepareWrite(len);
      drillBuf.setBytes(offset, value, 0, len);
      offsetsWriter.setNextOffset(offset + len);
      vectorIndex.nextElement();
    }

    @Override
    public final void setInt(final int value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setLong(final long value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDouble(final double value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setFloat(final float value) {
      setDecimal(BigDecimal.valueOf(value));
    }

    @Override
    public final void setDecimal(final BigDecimal value) {
      try {
        final BigDecimal rounded = value.setScale(scale, RoundingMode.HALF_UP);
        DecimalUtility.checkValueOverflow(rounded, precision, scale);
        final byte[] barr = rounded.unscaledValue().toByteArray();
        setBytes(barr,  barr.length);
      } catch (ArithmeticException e) {
        throw new InvalidConversionError("Decimal conversion failed for " + value, e);
      }
    }

    @Override
    public final void setValue(final Object value) {
      throw new InvalidConversionError("Generic object not supported for type VarDecimal, "
          + "setBytes(byte[], final int len)");
    }

    @Override
    public final void setDefaultValue(final Object value) {
      final BigDecimal rounded = ((BigDecimal) value).setScale(scale, RoundingMode.HALF_UP);
      DecimalUtility.checkValueOverflow(rounded, precision, scale);
      emptyValue = rounded.unscaledValue().toByteArray();
    }

    @Override
    public final void copy(ColumnReader from) {
      VarDecimalColumnReader source = (VarDecimalColumnReader) from;
      final DrillBuf sourceBuf = source.buffer();
      final long entry = source.getEntry();
      final int sourceOffset = (int) (entry >> 32);
      final int len = (int) (entry & 0xFFFF_FFFF);
      final int destOffset = prepareWrite(len);
      drillBuf.setBytes(destOffset, sourceBuf, sourceOffset, len);
      offsetsWriter.setNextOffset(destOffset + len);
      vectorIndex.nextElement();
    }
  }

}
